<?php
	namespace TemplateManager\PageLayouts;

	use Exception;
	use InvalidArgumentException;
	use Nox\ORM\ColumnQuery;
	use System\System;

	/**
	 * A utility class to fetch page sections belonging to a specific layout file.
	 */
	class PageLayoutSectionsProvider{
		public static function getLayoutFilePathRelativeToProjectRoot(string $fullLayoutFilePath): string{
			if (!str_contains($fullLayoutFilePath, "uplift-data")){
				throw new InvalidArgumentException("Provided file path does not contain uplift-data in the file path. Did you provide the right file path for the layout file? Path provided: $fullLayoutFilePath");
			}

			// Strip the project root directory from the $fullFilePath
			$projectRootDirectory = realpath(System::PROJECT_ROOT_DIRECTORY);
			$osFullFilePath = realpath($fullLayoutFilePath);

			if ($projectRootDirectory === false){
				throw new Exception("The PROJECT_ROOT_DIRECTORY doesn't exist. This exception shouldn't happen.");
			}

			if ($osFullFilePath === false){
				throw new InvalidArgumentException("The provided fullFilePath doesn't exist.");
			}

			$projectRootRelativeFullFilePath = str_replace($projectRootDirectory, "", $osFullFilePath);

			// Replace all OS backslashes, if any, with forward slashes
			$projectRootRelativeFullFilePath = str_replace("\\", "/", $projectRootRelativeFullFilePath);

			// Left-trim a starting forward slash
			$projectRootRelativeFullFilePath = ltrim($projectRootRelativeFullFilePath, "/");

			// Now, $projectRootRelativeFullFilePath begins with uplift-data
			return $projectRootRelativeFullFilePath;
		}

		/**
		 * @var string|null This is the current path to the page layout being used as context for this class.
		 * The path should start with "uplift-data/themes/.." and go directly to the layout file.
		 */
		private ?string $currentPageLayoutFilePath = null;

		/**
		 * @param string $fullFilePath
		 * @throws InvalidArgumentException
		 * @throws Exception
		 */
		public function __construct(string $fullFilePath)
		{
			// Truncate the fullFilePath to start with uplift-data
			$this->currentPageLayoutFilePath = self::getLayoutFilePathRelativeToProjectRoot($fullFilePath);
		}

		/**
		 * Fetches the PageLayoutSectionDefinition object that represents the current layout file in this
		 * instance's context property: $currentPageLayoutFilePath.
		 * @return PageLayoutSectionsDefinition
		 * @throws Exception
		 */
		public function getSectionDefinition(): PageLayoutSectionsDefinition{
			/** @var ?PageLayoutSectionsDefinition $sectionDefinition */
			$sectionDefinition = PageLayoutSectionsDefinition::queryOne(
				columnQuery: (new ColumnQuery())
					->where("layout_file_name", "=", $this->currentPageLayoutFilePath)
			);

			if ($sectionDefinition === null){
				throw new Exception("No PageLayoutSectionDefinition exists for layout file $this->currentPageLayoutFilePath");
			}

			// Load the linked sections into the definition
			$sectionDefinition->loadSections();

			return $sectionDefinition;
		}
	}