<?php

	namespace TemplateManager\PageLayouts;

	use InvalidArgumentException;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Interfaces\ModelInstance;
	use Nox\ORM\Interfaces\MySQLModelInterface;
	use Nox\ORM\ModelClass;
	use Page\Page;
	use ShortcodeParser\ShortcodeParser;

	class PageLayoutSectionsDefinition extends ModelClass implements ModelInstance
	{
		public ?int $id = null;
		/**
		 * @var ?string The layout file name, including the extension. This is a relative path starting with uplift-data.
		 * E.g., uplift-data/themes/default-theme/layouts/General.php
		 */
		public ?string $layoutFileName;
		/**
		 * @var int When the page layout section definition was created
		 */
		public int $creationTimestamp;

		/**
		 * @var PageLayoutSection[] The page layout sections for this definition. They must be eager-loaded
		 * in user-land to populate this array.
		 */
		public array $pageLayoutSections = [];

		public static function getModel(): MySQLModelInterface
		{
			return new PageLayoutSectionsDefinitionModel();
		}

		public function __construct()
		{
			parent::__construct($this);
		}

		/**
		 * Eager-loads the related PageLayoutSection objects that are attached to this definition. This will
		 * set the private property $pageLayoutSections.
		 * @return void
		 */
		public function loadSections(): self{
			/** @var PageLayoutSection[] $pageLayoutSections */
			$pageLayoutSections = PageLayoutSection::query(
				columnQuery: (new ColumnQuery())
					->where("page_layout_section_definition_id", "=", $this->id)
			);

			$this->pageLayoutSections = $pageLayoutSections;

			return $this;
		}	

		/**
		 * Will fetch the provided page Id's section that matches the sectionName provided.
		 * @param string $sectionName
		 * @param int $pageId
		 * @return string
		 * @throws InvalidArgumentException
		 */
		public function getSectionContent(string $sectionName, int $pageId): string{
			/** @var ?PageLayoutSection $section */
			$section = null;
			foreach($this->pageLayoutSections as $pageLayoutSection){
				if ($pageLayoutSection->sectionName === $sectionName){
					$section = $pageLayoutSection;
					break;
				}
			}

			/** @var ?Page $page */
			$page = Page::fetch($pageId);

			if ($page === null){
				throw new InvalidArgumentException("No page found with Id $pageId");
			}

			if ($section === null){
				throw new InvalidArgumentException("There is no PageLayoutSection attached to the section definition for this layout with the name $sectionName.");
			}

			// Get the content section that matches the $sectionName, for the individual page
			$contentSection = $page->getContentSectionByName($sectionName);

			if ($contentSection === null || $contentSection->content === null) {
				throw new InvalidArgumentException("No content found for section $sectionName.");
			}

			// Return the section's content
			try {
				$shortcodeParsedContent = (new ShortcodeParser())->parse($contentSection->content);
			} catch (Exception $e) {
				throw new RuntimeException("Failed to parse shortcodes: " . $e->getMessage());
			}

			return $shortcodeParsedContent;
		}

		/**
		 * Will fetch the provided page Id's section that matches the sectionName provided.
		 * @param string $sectionName
		 * @return string
		 * @throws InvalidArgumentException
		 */
		public function render(string $sectionName, int $pageId): void{
			/** @var ?PageLayoutSection $section */
			$section = null;
			foreach($this->pageLayoutSections as $pageLayoutSection){
				if ($pageLayoutSection->sectionName === $sectionName){
					$section = $pageLayoutSection;
					break;
				}
			}

			/** @var ?Page $page */
			$page = Page::fetch($pageId);

			if ($page === null){
				throw new InvalidArgumentException("No page found with Id $pageId");
			}

			if ($section === null){
				throw new InvalidArgumentException("There is no PageLayoutSection attached to the section definition for this layout with the name $sectionName.");
			}

			// Get the content section that matches the $sectionName, for the individual page
			$contentSection = $page->getContentSectionByName($sectionName);

			// Render the section's content
			// Parse shortcodes
			$shortcodeParsedContent = (new ShortcodeParser())->parse($contentSection->content ?? "");
			print($shortcodeParsedContent);
		}

	}
