<?php
	namespace System\Updater;

	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\GuzzleException;
	use GuzzleHttp\RequestOptions;
	use MonologWrapper\MonologWrapper;
	use Settings\Setting;
	use Settings\Settings;
	use Settings\SettingService;
	use System\Exceptions\ComposerIsMissingRequireArray;
	use System\Exceptions\PlatformRequirementsException;
	use System\PlatformRequirements;
	use System\System;
	use System\Updater\PostUpdate\Attributes\RunOnlyOnVersion;
	use System\Updater\PostUpdate\AutoOptOutInjectables;
	use System\Updater\PostUpdate\PostUpdateProcess;
	use System\Updater\PostUpdate\ResetCoreNeedsUpdateFlag;
	use System\Updater\PostUpdate\SendBuildInfoToMasterServer;
	use System\Updater\PostUpdate\SetHasAllPermsOnValidRoles;
	use System\Updater\PostUpdate\SyncModels;
	use Uplift\Exceptions\MalformedValue;
	use ZipArchive;

	class UpdaterService{

		const EXTRACTION_LOCATION_PRODUCTION = __DIR__ . "/../../../..";
		const EXTRACTION_LOCATION_TEST = __DIR__ . "/../../../../extraction-test";

		/**
		 * These are basic requirements to check for and download the update - not to know if the new update will be supported
		 * on the current platform. The Updater will check the current system requirements when we receive the update payload by
		 * checking the composer.json require array that comes with the payload.
		 * @throws MissingUpdateRequirement
		 */
		public static function assertSystemHasRequirementsToUpdate(): void{
			if (!class_exists('ZipArchive')){
				throw new MissingUpdateRequirement("PHP is missing the ZipArchive requirement to run the updater.");
			}

			if (!extension_loaded("openssl")){
				throw new MissingUpdateRequirement("The openssl PHP extension is missing from this PHP installation.");
			}

			if (!extension_loaded("curl")){
				throw new MissingUpdateRequirement("The curl PHP extension is missing from this PHP installation.");
			}

			if(ini_get("allow_url_fopen") !== "1") {
				throw new MissingUpdateRequirement("The PHP configuration setting 'allow_url_fopen' is disabled. This must be enabled for updates to be checked or installed. Current value is " . ini_get('allow_url_fopen'));
			}

		}

		/**
		 * @return array{status: int, needsUpdating: int, zipBallURL: string, repoOwner: string, repoName: string, githubAPIKey: string}
		 * @throws GuzzleException
		 * @throws MalformedValue
		 */
		public static function getUpdateStatusResponseFromMasterServer(): array{
			$logger = MonologWrapper::getLogger();
			$client = new Client();
			$host = System::getUpliftControlPanelHost();
			$url = sprintf("%s/uplift/build/updater/check-if-update-needed", $host);
			$uuid = Setting::getSettingValue(Settings::BUILD_UUID->value);
			$apiKey = Setting::getSettingValue(Settings::UPLIFT_CONTROL_PANEL_API_KEY->value);

			$logger->info("Fetching build update info from master server.");

			$response = $client->request(
				method:"GET",
				uri:$url,
				options: [
					RequestOptions::HEADERS => [
						"Uplift-UUID"=>$uuid,
						"Uplift-API-Key"=>$apiKey,
					],
					RequestOptions::QUERY => [
						"build-version"=>System::VERSION,
					],
				]
			);

			$logger->info("Fetched update info.");

			$contents = $response->getBody()->getContents();
			return json_decode($contents, true);
		}

		/**
		 * @throws GuzzleException
		 * @throws MalformedValue
		 */
		public static function doesSystemNeedUpdating(): bool{
			// Checks the master server to determine if this build needs an update
			$updateData = self::getUpdateStatusResponseFromMasterServer();
			return $updateData['needsUpdating'] === 1;
		}

		/**
		 * @return string The location of the update zip to install
		 * @throws GuzzleException
		 * @throws MalformedValue
		 * @throws MissingZipURL
		 * @throws UpdateFailedOpeningPackageLocation
		 */
		public static function downloadUpdatedBuild(): string{
			// Checks the master server to determine if this build needs an update
			$logger = MonologWrapper::getLogger();
			$logger->info("Downloading latest update.");
			$updateData = self::getUpdateStatusResponseFromMasterServer();
			$zipBallURL = $updateData['zipBallURL'] ?? null;

			if (!$zipBallURL){
				$logger->error("Update server did not give back a zip archive to download.");
				throw new MissingZipURL("Missing zipBallURL array key from master server. This build is probably not out of date.");
			}

			$apiKey = $updateData['githubAPIKey'];
			$client = new Client();
			$logger->info("Download zip file from GitHub releases API.");
			$response = $client->request(
				method:"GET",
				uri:$zipBallURL,
				options: [
					RequestOptions::ALLOW_REDIRECTS => [
						"max"=>2,
						"track_redirects"=>true,
					],
					RequestOptions::HEADERS => [
						"Authorization"=>sprintf("token %s", $apiKey),
						"Accept"=>"application/octet-stream",
					],
					RequestOptions::QUERY => [
						"build-version"=>System::VERSION,
					],
				],
			);

			$logger->info("Update downloaded, writing to a zip file.");
			$fileContents = $response->getBody()->getContents();
			$projectRoot = realpath(__DIR__ . "/../../../..");
			$updatePackageLocation = sprintf("%s%supdate-package.zip", $projectRoot, DIRECTORY_SEPARATOR);
			$handle = fopen($updatePackageLocation, "w");
			if ($handle !== false) {
				fwrite($handle, $fileContents);
				fclose($handle);
				$logger->info("ZIP file successfully written.");
				return realpath($updatePackageLocation);
			}else{
				$logger->error("There was an error calling fopen on the update-package.zip location. Possibly missing permissions or out of memory. Out of memory is unlikely - check file permissions for PHP in this directory.");
				throw new UpdateFailedOpeningPackageLocation("Failed to open file location {$updatePackageLocation}");
			}
		}

		/**
		 * Looks for an entry in the zip archive named "install-entries.json"
		 * This file is a JSON array of directories that should be installed for an update
		 * @param ZipArchive $zip
		 * @return array
		 * @throws InstallDirectoriesMissingFromUpdatePackage
		 */
		private static function getInstallationEntriesFromZip(ZipArchive $zip): array{
			for ($i = 1; $i < $zip->numFiles - 1; $i++){
				$entryName = $zip->getNameIndex($i);
				if (strtolower($entryName) === "install-entries.json"){
					$stream = $zip->getStream($entryName);
					$contents = "";
					do {
						$newContents = fread($stream, 10200);
						if ($newContents !== false && strlen($newContents) > 0) {
							$contents .= $newContents;
						}
					} while (strlen($newContents) > 0 && $newContents !== false);
					return json_decode($contents, true);
				}
			}

			throw new InstallDirectoriesMissingFromUpdatePackage("No file named install-entries.json found in the update package root. This is a required file in all Uplift upload packages. Contact the developers of Uplift to inform them that the recent release is bugged. It will not be installed.");
		}

		/**
		 * Looks for an entry in the zip archive named "composer.json"
		 * This file will be used to determine if the current system requirements will support the next CMS version.
		 * @param ZipArchive $zip
		 * @return array
		 * @throws ComposerJSONMissingFromUpdatePackage
		 */
		private static function getComposerConfigFromZip(ZipArchive $zip): array{
			for ($i = 1; $i < $zip->numFiles - 1; $i++){
				$entryName = $zip->getNameIndex($i);
				if (strtolower($entryName) === "composer.json"){
					$stream = $zip->getStream($entryName);
					$contents = "";
					do {
						$newContents = fread($stream, 10200);
						if ($newContents !== false && strlen($newContents) > 0) {
							$contents .= $newContents;
						}
					} while (strlen($newContents) > 0 && $newContents !== false);
					return json_decode($contents, true);
				}
			}

			throw new ComposerJSONMissingFromUpdatePackage("No file named composer.json found in the update package root. All update payloads must have a composer.json - this is an upstream problem.");
		}

		/**
		 * Installs an update from a package location. The package is a zip file from the GitHub releases API
		 * @throws InstallDirectoriesMissingFromUpdatePackage
		 * @throws PlatformRequirementsException
		 * @throws ComposerJSONMissingFromUpdatePackage
		 */
		public static function installUpdate(
			string $updatePackageLocation,
			string $extractionLocation,
		): void{
			// Open the zip archive
			$zip = new ZipArchive();
			$openResponse = $zip->open($updatePackageLocation);

			$logger = MonologWrapper::getLogger();

			if ($openResponse === true){
				// Fetch the composer.json for this payload. We can check that the requirements listed in the
				// 'require' block of the composer.json are met on this system
				$composerConfig = self::getComposerConfigFromZip($zip);

				if (!isset($composerConfig['require'])){
					$logger->debug(json_encode($composerConfig));
					throw new ComposerIsMissingRequireArray("There is no 'require' array in the composer.json file in the update package. The update cannot be installed. This is an upstream problem. The array is logged to the internal log.");
				}

				// Assert that the update can proceed
				PlatformRequirements::assertPlatformRequirements($composerConfig['require']);

				// Get the entries that should be installed from the package
				$entriesToInstall = self::getInstallationEntriesFromZip($zip);

				// Append the containing folder name to the entries
				// The entries will already start with a "/"
				foreach($entriesToInstall as $index=>$entryNameStub){
					$entriesToInstall[$index] = sprintf("%s", $entryNameStub);
				}

				// Manually iterate every entry in the zip and extract the ones listed above
				for ($i = 0; $i < $zip->numFiles - 1; $i++){
					$entryName = $zip->getNameIndex($i);

					// Check if this entry name starts with any of the valid folders we want to extract
					// in an update
					$passesCheck = false;
					foreach($entriesToInstall as $entryNameToInstall){
						if (str_starts_with(haystack:strtolower($entryName), needle: strtolower($entryNameToInstall))){
							$passesCheck = true;
							break;
						}
					}

					// Not a file or directory we want, skip this entry
					if (!$passesCheck){
						continue;
					}

					$newLocation = sprintf("%s/%s", $extractionLocation, $entryName);

					if (str_ends_with(haystack: $newLocation, needle: "/")){
						// This is a directory
						@mkdir($newLocation, 0777, true);
					}else {
						$readStream = $zip->getStream($entryName);
						$contents = "";
						do {
							$newContents = fread($readStream, 10200);
							if ($newContents !== false && strlen($newContents) > 0) {
								$contents .= $newContents;
							}
						} while (strlen($newContents) > 0 && $newContents !== false);
						fclose($readStream);

						// Check if the directory to extract to exists, if not create it
						$directoryName = dirname($newLocation);
						if (!file_exists($directoryName)){
							@mkdir($directoryName, 0777, true);
						}

						// Write the file
						$newFileHandle = fopen($newLocation, "w");
						fwrite($newFileHandle, $contents);
						fclose($newFileHandle);
					}
				}
			}else{
				// TODO Handle error codes
			}

			$zip->close();

			// Clear the update zip
			gc_collect_cycles();
			unlink($updatePackageLocation);
		}

		/**
		 * Will check if the system is currently in the middle of being updated
		 */
		public static function isSystemUpdating(): bool{
			$isUpdating = Setting::getSettingValue(Settings::IS_CORE_UPDATING->value);
			return $isUpdating === "1";
		}

		/**
		 * Will check if the system is currently in the middle of being updated
		 */
		public static function toggleIsUpdatingFlag(bool $value): void{
			SettingService::saveSetting(Settings::IS_CORE_UPDATING->value, $value ? "1" : "0");
		}

		/**
		 * Runs all post update processes
		 */
		public static function runPostUpdateProcesses(): void{
			$logger = MonologWrapper::getLogger();
			$logger->info("Running post-update processes.");
			if (function_exists("opcache_reset")) {
				$logger->info("Resetting opcache.");
				$success = @opcache_reset();
				if ($success){
					$logger->info("Opcache successfully reset.");
				}else{
					$logger->warning("Failed to reset opcache.");
				}
			}else{
				$logger->info("Not resetting opcache. Function opcache_reset not found.");
			}

			// Register any PostUpdate process classes here - in order
			$classesForPostUpdateProcess = [
				ResetCoreNeedsUpdateFlag::class,
				SyncModels::class,
				SendBuildInfoToMasterServer::class,
				SetHasAllPermsOnValidRoles::class,
				AutoOptOutInjectables::class,
			];

			foreach($classesForPostUpdateProcess as $classForPostUpdateProcess){
				try {
					$classReflection = new \ReflectionClass($classForPostUpdateProcess);
				}catch(\ReflectionException $e){
					$logger->error($e->getMessage());
					continue;
				}

				$runOnlyAttributes = $classReflection->getAttributes(
					RunOnlyOnVersion::class,
					\ReflectionAttribute::IS_INSTANCEOF
				);

				if (!empty($runOnlyAttributes)){
					// There is an attribute that limits this process to a version

					/** @var RunOnlyOnVersion $firstAttribute */
					$firstAttribute = $runOnlyAttributes[0]->newInstance();

					$logger->info("$classForPostUpdateProcess specifies version {$firstAttribute->versionString} needed to run this PostUpdateProcess. Current system version is " . System::VERSION);

					if ($firstAttribute->versionString === System::VERSION){
						$logger->info("Versions match. Process can run.");
						// This process can run
						/** @var PostUpdateProcess $postUpdateProcess */
						$postUpdateProcess = $classReflection->newInstance();
						$postUpdateProcess->runProcess();
					}
				}else{
					// Empty, no RunOnlyOnVersion attributes
					/** @var PostUpdateProcess $postUpdateProcess */
					$postUpdateProcess = $classReflection->newInstance();
					$postUpdateProcess->runProcess();
				}

			}

			$logger->info("All post update processes ran.");
		}
	}