<?php

	namespace System\Updater;

	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use APIPublic\AuthorizeByAPIKey;
	use GuzzleHttp\Exception\GuzzleException;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use NoxEnv;
	use Roles\PermissionCategories;
	use Settings\Setting;
	use Settings\Settings;
	use Settings\SettingService;
	use System\Exceptions\PlatformRequirementsException;
	use Uplift\Exceptions\MalformedValue;

	#[Controller]
	#[RouteBase("/uplift/updater")]
	class UpdaterController extends BaseController{

		/**
		 * Tells the system that it should run an update check to determine if this build needs to be updated.
		 * This endpoint is designed to be called by the Uplift Master Control Panel and not by a local, authenticated
		 * user.
		 */
		#[Route("GET", "/run-update")]
		#[AuthorizeByAPIKey]
		#[UseJSON]
		public function remoteCalledRunUpdateCheck(): JSONResult
		{
			try {
				UpdaterService::assertSystemHasRequirementsToUpdate();
			}catch(MissingUpdateRequirement $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			try {
				$needsUpdating = UpdaterService::doesSystemNeedUpdating();
			}catch(GuzzleException|MalformedValue $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			if ($needsUpdating){
				// Run an update
				UpdaterService::toggleIsUpdatingFlag(true);
				try {
					$updateZipURL = UpdaterService::downloadUpdatedBuild();
				} catch (GuzzleException|MissingZipURL|UpdateFailedOpeningPackageLocation|MalformedValue $e) {
					http_response_code(500);
					UpdaterService::toggleIsUpdatingFlag(false);
					return new JSONError($e->getMessage());
				}
				if (NoxEnv::DEV_ENV === "development") {
					$extractionLocation = UpdaterService::EXTRACTION_LOCATION_TEST;
				}else if (NoxEnv::DEV_ENV === "production"){
					$extractionLocation = UpdaterService::EXTRACTION_LOCATION_PRODUCTION;
				}else{
					http_response_code(500);
					UpdaterService::toggleIsUpdatingFlag(false);
					return new JSONError("Unknown NoxEnv::DEV_ENV value '" . NoxEnv::DEV_ENV . "'");
				}

				try {
					UpdaterService::installUpdate(
						updatePackageLocation: $updateZipURL,
						extractionLocation: $extractionLocation,
					);
					UpdaterService::toggleIsUpdatingFlag(false);
				} catch (InstallDirectoriesMissingFromUpdatePackage|PlatformRequirementsException|ComposerJSONMissingFromUpdatePackage $e) {
					http_response_code(500);
					UpdaterService::toggleIsUpdatingFlag(false);
					return new JSONError($e->getMessage());
				}
			}else{
				return new JSONSuccess([
					"message"=>"No update needed. System is up to date.",
				]);
			}

			return new JSONSuccess([
				"message"=>"Updated successfully.",
			]);
		}

		#[Route("GET", "/check-if-core-needs-updating")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function checkIfCoreNeedsUpdating(): JSONResult
		{
			try {
				UpdaterService::assertSystemHasRequirementsToUpdate();
			}catch(MissingUpdateRequirement $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			// To avoid pinging the master server, and thus GitHub, too often a cache has been implemented.
			$lastUpdateCheckTime = Setting::getSettingValue(Settings::LAST_TIMESTAMP_CHECKED_FOR_UPDATE->value);
			if ($lastUpdateCheckTime !== null){
				$lastUpdateCheckTime = (int) $lastUpdateCheckTime;
				$currentTime = time();
				$lastUpdateCheckThreshold = 60 * 15; // Can only ping once every 15 minutes
				if ($currentTime - $lastUpdateCheckTime < $lastUpdateCheckThreshold) {
					// Cached
					$coreNeedsUpdatingCacheValue = Setting::getSettingValue(Settings::CORE_NEEDS_UPDATING->value);
					if ($coreNeedsUpdatingCacheValue === "1"){
						return new JSONSuccess([
							"needsUpdating" => 1,
							"message" => "Core flagged as needing to be updated.",
							"fromCache" => 1,
							"lastNonCacheUpdateCheck"=>$lastUpdateCheckTime,
						]);
					}else{
						return new JSONSuccess([
							"needsUpdating" => 0,
							"message" => "Core does not need to be updated.",
							"fromCache" => 1,
							"lastNonCacheUpdateCheck"=>$lastUpdateCheckTime,
						]);
					}
				}
			}

			// Set the cache time
			SettingService::saveSetting(Settings::LAST_TIMESTAMP_CHECKED_FOR_UPDATE->value, time());

			try {
				$needsUpdating = UpdaterService::doesSystemNeedUpdating();
			}catch(GuzzleException|MalformedValue $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			if ($needsUpdating){
				SettingService::saveSetting(Settings::CORE_NEEDS_UPDATING->value, "1");
				return new JSONSuccess([
					"needsUpdating"=>1,
					"message"=>"Core flagged as needing to be updated.",
					"fromCache"=>0,
				]);
			}else{
				SettingService::saveSetting(Settings::CORE_NEEDS_UPDATING->value, "0");
				return new JSONSuccess([
					"needsUpdating"=>0,
					"message"=>"Core does not need updating.",
					"fromCache"=>0,
				]);
			}
		}

		/**
		 * This is the local version (non-remote-called) endpoint that will run the core update, if it needs one.
		 */
		#[Route("GET", "/update-core")]
		#[RequireLogin]
		#[UseJSON]
		public function updateCore(): JSONResult
		{
			try {
				UpdaterService::assertSystemHasRequirementsToUpdate();
			}catch(MissingUpdateRequirement $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			if (Setting::getSettingValue(Settings::CORE_NEEDS_UPDATING->value) === "1"){
				// Run an update
				UpdaterService::toggleIsUpdatingFlag(true);
				try {
					$updateZipURL = UpdaterService::downloadUpdatedBuild();
				} catch (GuzzleException|MissingZipURL|UpdateFailedOpeningPackageLocation|MalformedValue $e) {
					http_response_code(500);
					UpdaterService::toggleIsUpdatingFlag(false);
					return new JSONError($e->getMessage());
				}
				if (NoxEnv::DEV_ENV === "development") {
					$extractionLocation = UpdaterService::EXTRACTION_LOCATION_TEST;
				}else if (NoxEnv::DEV_ENV === "production"){
					$extractionLocation = UpdaterService::EXTRACTION_LOCATION_PRODUCTION;
				}else{
					http_response_code(500);
					UpdaterService::toggleIsUpdatingFlag(false);
					return new JSONError("Unknown NoxEnv::DEV_ENV value '" . NoxEnv::DEV_ENV . "'");
				}

				try {
					UpdaterService::installUpdate(
						updatePackageLocation: $updateZipURL,
						extractionLocation: $extractionLocation,
					);
					UpdaterService::toggleIsUpdatingFlag(false);
					SettingService::saveSetting(Settings::CORE_NEEDS_UPDATING->value, "0");
				} catch (InstallDirectoriesMissingFromUpdatePackage|PlatformRequirementsException|ComposerJSONMissingFromUpdatePackage $e) {
					http_response_code(500);
					UpdaterService::toggleIsUpdatingFlag(false);
					return new JSONError($e->getMessage());
				}
			}else{
				http_response_code(500);
				UpdaterService::toggleIsUpdatingFlag(false);
				return new JSONError(sprintf("The setting %s does not indicate a value that the core needs to be updated.", Settings::CORE_NEEDS_UPDATING->value));
			}

			return new JSONSuccess();
		}

		#[Route("GET", "/run-post-update-processes")]
		#[RequireLogin]
		#[UseJSON]
		public function runPostUpdateProcesses(): JSONResult
		{
			UpdaterService::runPostUpdateProcesses();
			return new JSONSuccess([]);
		}

		/**
		 * Same route as the above, but accessible by an API key remotely
		 */
		#[Route("GET", "/run-post-update-processes-from-remote")]
		#[AuthorizeByAPIKey]
		#[UseJSON]
		public function runPostUpdateProcessesCalledFromRemote(): JSONResult
		{
			UpdaterService::runPostUpdateProcesses();
			return new JSONSuccess([]);
		}

	}