<?php
	namespace System;

	use Exception;
	use Settings\Setting;

	class Themes{

		const DEFAULT_THEME_DIRECTORY_NAME = "default-theme";

		/**
		 * Returns a list of theme folder names and their themeName from the manifest.json file in the theme
		 * directory
		 * @return string[] Key is the theme directory name and value is the official theme name from the manifest.json
		 */
		public static function getThemes(): array{
			$directory = System::THEMES_DIRECTORY;
			$fileContents = array_diff(scandir($directory), ['.','..']);
			$themes = [];
			foreach($fileContents as $fileName){
				$fullFilePath = sprintf("%s/%s", $directory, $fileName);
				if (is_dir($fullFilePath)){
					$manifest = sprintf(
						"%s/%s/manifest.json",
						$directory,
						$fileName,
					);
					if (file_exists($manifest)){
						$manifestContents = json_decode(file_get_contents($manifest), true);
						if (is_array($manifestContents)){
							if (array_key_exists(key: "themeName", array:$manifestContents)){
								$themes[$fileName] = $manifestContents['themeName'];
							}
						}
					}
				}
			}


			return $themes;
		}

		public static function getCurrentTheme(): string{
			return Setting::getSettingValue(Setting::SETTING_NAMES['theme_name']);
		}

		/**
		 * Gets the full system path to the current theme directory
		 * @throws Exception
		 */
		public static function getCurrentThemeDirectory(): string{
			$directory = System::THEMES_DIRECTORY;
			$currentThemeFolderName = Setting::getSettingValue(Setting::SETTING_NAMES['theme_name']);

			$fullPath = sprintf("%s/%s", $directory, $currentThemeFolderName);
			$normalizedPath = realpath($fullPath);
			if ($normalizedPath === false){
				throw new Exception(sprintf("The theme directory path (%s) does not exist", $fullPath));
			}

			return $normalizedPath;
		}

		/**
		 * Gets the browser URI for the current theme directory.
		 * E.g. "/uplift-data/themes/default-theme"
		 * @throws Exception
		 */
		public static function getCurrentThemeDirectoryURI(): string{
			$fullFilePath = self::getCurrentThemeDirectory();

			$dataDirectory = realpath(System::DATA_DIRECTORY);
			$uri = str_replace($dataDirectory, "/" . basename($dataDirectory), $fullFilePath);
			return str_replace("\\", "/", $uri);
		}

		/**
		 * Fetches the current theme's shortcodes folder directory path
		 * @throws Exception
		 */
		public static function getCurrentThemeShortcodeViewsDirectory(): string{
			$currentThemeDirectory = self::getCurrentThemeDirectory();
			$shortcodesDirectory = sprintf("%s/shortcode-views", $currentThemeDirectory);
			$normalizedShortcodesDirectory = realpath($shortcodesDirectory);
			if ($normalizedShortcodesDirectory === false){
				throw new Exception(sprintf("The shortcodes directory path (%s) does not exist", $shortcodesDirectory));
			}else{
				return $normalizedShortcodesDirectory;
			}
		}

		/**
		 * Fetches the current theme's includes folder directory path
		 * @throws Exception
		 */
		public static function getCurrentThemeIncludesDirectory(): string{
			$currentThemeDirectory = self::getCurrentThemeDirectory();
			$includesDirectory = sprintf("%s/includes", $currentThemeDirectory);
			$normalizedIncludesDirectory = realpath($includesDirectory);
			if ($normalizedIncludesDirectory === false){
				throw new Exception(sprintf("The includes directory path (%s) does not exist", $includesDirectory));
			}else{
				return $normalizedIncludesDirectory;
			}
		}

		/**
		 * Fetches the current theme's layouts folder directory path
		 * @throws Exception
		 */
		public static function getCurrentThemLayoutsDirectory(): string{
			$currentThemeDirectory = self::getCurrentThemeDirectory();
			$includesDirectory = sprintf("%s/layouts", $currentThemeDirectory);
			$normalizedIncludesDirectory = realpath($includesDirectory);
			if ($normalizedIncludesDirectory === false){
				throw new Exception(sprintf("The layouts directory path (%s) does not exist", $includesDirectory));
			}else{
				return $normalizedIncludesDirectory;
			}
		}

		/**
		 * Determines if a file path is within the current theme's directory
		 * @throws Exception
		 */
		public static function isFilePathInThemeDirectory(string $filePath): bool{
			$filePath = realpath($filePath);

			if ($filePath === false){
				return false;
			}

			if (str_starts_with($filePath, self::getCurrentThemeDirectory())){
				return true;
			}

			return false;
		}

	}