<?php
	namespace System;

	use Exception;
	use LogicException;
	use MonologWrapper\MonologWrapper;
	use Nox\ORM\Abyss;
	use Nox\ORM\Exceptions\MissingDatabaseCredentials;
	use Settings\Setting;
	use Settings\Settings;
	use Uplift\Exceptions\MalformedValue;

	class System
	{
		const VERSION = "4.2.16";
		const PROJECT_ROOT_DIRECTORY = __DIR__ . "/../../..";
		const DATA_DIRECTORY = __DIR__ . "/../../../uplift-data";
		const THEMES_DIRECTORY = self::DATA_DIRECTORY . "/themes";

		/**
		 * @throws MalformedValue
		 */
		public static function getUpliftControlPanelHost(): string{
			if (\NoxEnv::DEV_ENV === "development"){
				return "http://localhost:1337";
			}elseif (\NoxEnv::DEV_ENV === "production"){
				return "https://uplift.footbridgemedia.com";
			}else{
				throw new MalformedValue(sprintf("The value of the NoxEnv::DEV_ENV is unrecognized. Only 'development' or 'production' are recognized. The current value is %s", \NoxEnv::DEV_ENV));
			}
		}

		/**
		 * This function call will register Abyss credentials
		 * @return SystemInstallationState
		 * @throws LogicException
		 */
		public static function getInstallationState(): SystemInstallationState{

			// Check for database tables
			$abyss = new Abyss();
			$logger = MonologWrapper::getLogger();

			// Exception can't happen here, the credentials are added and verified above.
			try {
				$mysqli = $abyss->getConnectionToDatabase(\NoxEnv::MYSQL_DB_NAME);
			}catch(MissingDatabaseCredentials|Exception $e){
				$logger->critical($e->getMessage());
				throw new LogicException("Impossible exception happened. Something is wrong with your code's logic. " . $e->getMessage());
			}

			$tableCountQuery = sprintf("
				SELECT count(*) AS numTables
				FROM information_schema.tables
				WHERE table_schema = '%s'
			", \NoxEnv::MYSQL_DB_NAME);

			$result = $mysqli->query($tableCountQuery);

			/** @var array{numTables: string} $row */
			$row = $result->fetch_assoc();
			$numTables = (int) $row['numTables'];
			if ($numTables === 0){
				return SystemInstallationState::NO_DATABASE_TABLES;
			}

			// All there, system is installed
			return SystemInstallationState::COMPLETE;
		}

		public static function getTimeAgo($timestamp)
		{
			// Gets the difference between the timestamp provided and the current time
			$timeDifference = time() - $timestamp;

			if($timeDifference < 1) {
				return '0 seconds';
			}

			$unitsOfTime = [
				365 * 24 * 60 * 60 => 'year',
				30 * 24 * 60 * 60 => 'month',
				24 * 60 * 60 => 'day',
				60 * 60 => 'hour',
				60 => 'minute',
				1 => 'second'
			];
			$pluralUnits = [
				'year' => 'years',
				'month' => 'months',
				'day' => 'days',
				'hour' => 'hours',
				'minute' => 'minutes',
				'second' => 'seconds'
			];

			foreach($unitsOfTime as $key => $unit) {
				$x = $timeDifference / $key;
				if($x >= 1) {
					$y = round($x);
					return $y . ' ' . ($y > 1 ? $pluralUnits[$unit] : $unit) . ' ago';
				}
			}
		}

		/**
		 * Parses the PHP.ini configuration for either post_max_size or upload_max_filesize (whichever is smallest)
		 * and transforms it into bytes. From Drupal source code.
		 * @see Drupal source code
		 */
		public static function getMaximumUploadSizeInBytes(): int{
			$maxSize = -1;

			// Start with post_max_size.
			$postMaxSizeInBytes = System::parseSizeString(ini_get('post_max_size'));
			if ($postMaxSizeInBytes > 0) {
				$maxSize = $postMaxSizeInBytes;
			}

			// If upload_max_size is less, then reduce. Except if upload_max_size is
			// zero, which indicates no limit.
			$uploadMaxSizeInBytes = System::parseSizeString(ini_get('upload_max_filesize'));
			if ($uploadMaxSizeInBytes > 0 && $uploadMaxSizeInBytes < $postMaxSizeInBytes) {
				$maxSize = $uploadMaxSizeInBytes;
			}

			return $maxSize;
		}

		/**
		 * Converts a size string representing memory amounts (E.g. "30M" or "10K") to an integer of number of bytes
		 */
		public static function parseSizeString(string $size): int {
			$unit = preg_replace('/[^bkmgtpezy]/i', '', $size); // Remove the non-unit characters from the size.
			$size = preg_replace('/[^0-9\.]/', '', $size); // Remove the non-numeric characters from the size.
			if ($unit) {
				// Find the position of the unit in the ordered string which is the power of magnitude to multiply a kilobyte by.
				return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
			}
			else {
				return round($size);
			}
		}

		/**
		 * Checks if the provided API key matches the known API key in the system settings. This will also return false
		 * if there is no API key registered in the system settings.
		 * @param string $apiKey
		 * @return bool
		 */
		public static function isAPIKeyValid(string $apiKey): bool{
			/** @var Setting $currentApiKey */
			$currentApiKey = Setting::fetch(Settings::UPLIFT_CONTROL_PANEL_API_KEY->value);

			if ($currentApiKey === null){
				return false;
			}else{
				return $currentApiKey->value === $apiKey;
			}
		}
	}
