<?php
	namespace System;

	use FileSystemUtilities\FSFile;
	use Settings\Setting;
	use Uplift\Exceptions\NoObjectFound;

	class Layouts{

		/**
		 * Gets a list of all the available layout files. Returns an array of FSFile objects
		 * @return FSFile[]
		 */
		public static function getAvailableLayouts(): array{
			$themeName = Setting::getSettingValue(Setting::SETTING_NAMES['theme_name']);
			$layouts = [];
			if (!empty($themeName)){
				$themeFolder = sprintf("%s/%s", System::THEMES_DIRECTORY, $themeName);
				if (file_exists($themeFolder)){
					$layoutsFolder = $themeFolder . "/layouts";

					if (!file_exists($layoutsFolder)){
						@mkdir($layoutsFolder, 0777, true);
					}

					$layoutFiles = array_diff(scandir($layoutsFolder), [".",".."]);

					foreach($layoutFiles as $fileName){
						if ($fileName !== ".gitkeep"){
							$fullPath = realpath(sprintf("%s/%s", $layoutsFolder, $fileName));
							$layouts[] = new FSFile(
								$fileName,
								$fullPath,
								filesize($fullPath),
							);
						}
					}
				}
			}

			return $layouts;
		}

		/**
		 * Checks if a layout is valid and in the system
		 * @param string $layoutName Layout name without the file extension
		 */
		public static function isLayoutAvailable(string $layoutName): bool{
			foreach(self::getAvailableLayouts() as $fsFile){
				if ($fsFile->fileNameWithoutExtension === $layoutName){
					return true;
				}
			}

			return false;
		}

		/**
		 * Fetches the layout file contents belonging to a $layoutName from a Page object
		 * @param string $layoutName
		 * @return string
		 * @throws NoObjectFound
		 */
		public static function getLayoutFileContentsFromPageLayoutName(string $layoutName): string{
			foreach(self::getAvailableLayouts() as $fsFile) {
				if ($fsFile->fileNameWithoutExtension === $layoutName) {
					return file_get_contents($fsFile->fullFilePath);
				}
			}

			throw new NoObjectFound("No layout file was found with the name $layoutName");
		}
	}