<?php
	namespace Sitemaps;


	use Nox\ORM\ColumnQuery;
	use Page\Page;
	use Page\PublicationStatus;
	use Sitemaps\Exceptions\RouteInUse;
	use System\HttpHelper;
	use ValueError;

	require_once __DIR__ . "/Sitemap.php";

	class SitemapsService
	{
		/**
		 * @param string $name
		 * @param string $route
		 * @param string $content
		 * @param int $contentType
		 * @return Sitemap
		 * @throws RouteInUse
		 * @throws ValueError
		 */
		public static function createSitemap(
			string $name,
			string $route,
			string $content,
			int $contentType,
		): Sitemap
		{
			if (!str_starts_with(haystack: $route, needle: "/")){
				throw new ValueError("Sitemap route must start with a forward slash - /");
			}

			$existingSitemapRoute = Sitemap::queryOne(
				columnQuery:(new ColumnQuery())
					->where("route","=", $route),
			);

			if($existingSitemapRoute !== null) {
				throw new RouteInUse("Route is already in use.");
			}

			$newSitemap = new Sitemap();
			$newSitemap->name = trim($name);
			$newSitemap->route = trim($route);
			$newSitemap->content = trim($content);
			$newSitemap->contentType = $contentType;
			$newSitemap->save();

			return $newSitemap;
		}

		/**
		 * @param Sitemap $sitemap
		 * @param string $name
		 * @param string $route
		 * @param string $content
		 * @param int $contentType
		 * @return Sitemap
		 * @throws RouteInUse
		 * @throws ValueError
		 */
		public static function saveSitemap(
			Sitemap $sitemap,
			string $name,
			string $route,
			string $content,
			int $contentType,
		): Sitemap
		{

			if (!str_starts_with(haystack: $route, needle: "/")){
				throw new ValueError("Sitemap route must start with a forward slash - /");
			}

			$existingSitemapRoute = Sitemap::queryOne(
				columnQuery:(new ColumnQuery())
					->where("route","=", $route)
					->and()
					->where("id","!=", $sitemap->id),
			);

			if($existingSitemapRoute !== null) {
				throw new RouteInUse("Route is already in use.");
			}

			$sitemap->name = trim($name);
			$sitemap->route = trim($route);
			$sitemap->content = trim($content);
			$sitemap->contentType = $contentType;
			$sitemap->save();

			return $sitemap;
		}

		public static function deleteSitemap(Sitemap $sitemap): void
		{
			$sitemap->delete();
		}

		/**
		 * A function that will generate the default /sitemap.xml content
		 */
		public static function getDefaultSitemapXMLContent(): string{
			$xml = new \SimpleXMLElement("<?xml version=\"1.0\" encoding=\"utf-8\" ?><urlset></urlset>");
			$xml->addAttribute("xmlns", "http://www.sitemaps.org/schemas/sitemap/0.9");

			// Get available pages
			/** @var Page[] $pages */
			$pages = Page::query(
				columnQuery: (new ColumnQuery())
					->where("publication_status", "=", PublicationStatus::Published->value)
					->and()
					->where("publication_timestamp", "<=", time())
					->and()
					->where("excludeFromSitemap", "=", 0)
			);

			foreach($pages as $page){
				$url = $xml->addChild("url");
				$pageRoute = $page->pageRoute;

				// Escape any XML-invalid characters that could be in a URL
				$pageRouteEscaped = htmlspecialchars($pageRoute, ENT_XML1);

				$url->addChild("loc", sprintf("%s%s", HttpHelper::getWebsiteBaseURL(), $pageRouteEscaped));
			}

			return $xml->asXML();
		}
	}