<?php

	namespace Sitemaps;

	use Accounts\Account;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Accounts\Attributes\RequireLogin;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use Sitemaps\Exceptions\RouteInUse;
	use ValueError;
	use System\HttpHelper;

	#[Controller]
	#[RouteBase("/uplift")]
	class SitemapManagerController extends BaseController
	{

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/manage-sitemaps")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_SITEMAPS)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "manage-sitemaps/main.php"
			);
		}

		#[Route("GET", "/sitemaps")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_SITEMAPS)]
		public function getSitemaps(): JSONResult
		{
			return new JSONSuccess([
				"sitemaps"=>Sitemap::query(),
			]);
		}

		#[Route("PUT", "/sitemaps")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_SITEMAPS)]
		public function addNewSitemap(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$name = $payload->getTextPayload('sitemap-name');
				$route = $payload->getTextPayload('sitemap-route');
				$content = $payload->getTextPayload('content');
				$contentType = $payload->getTextPayload('sitemap-content-type');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newSitemap = SitemapsService::createSitemap(
					name: $name->contents,
					route: $route->contents,
					content: $content->contents,
					contentType: $contentType->contents,
				);
			} catch (RouteInUse|ValueError $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CREATE_SITEMAP->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"sitemapID"=>$newSitemap->id,
				]),
			);

			return new JSONSuccess([
				"newSitemap"=>$newSitemap,
			]);
		}

		#[Route("PATCH", "@/sitemaps/(?<sitemapID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_SITEMAPS)]
		public function editSitemap(Request $request): JSONResult
		{
			$sitemapID = (int) $request->getParameter("sitemapID");
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			/** @var null | Sitemap $sitemap */
			$sitemap = Sitemap::fetch($sitemapID);

			if ($sitemap === null){
				return new JSONError(sprintf("No sitemap found with ID %d", $sitemapID));
			}

			try {
				$name = $payload->getTextPayload('sitemap-name');
				$route = $payload->getTextPayload('sitemap-route');
				$content = $payload->getTextPayload('content');
				$contentType = $payload->getTextPayload('sitemap-content-type');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				SitemapsService::saveSitemap(
					sitemap: $sitemap,
					name: $name->contents,
					route: $route->contents,
					content: $content->contents,
					contentType: $contentType->contents,
				);
			} catch (RouteInUse|ValueError $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_SITEMAP->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"sitemapID"=>$sitemapID,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("DELETE", "@/sitemaps/(?<sitemapID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_SITEMAPS)]
		public function deleteSitemap(Request $request): JSONResult
		{
			$sitemapID = (int) $request->getParameter('sitemapID');
			$account = Account::getCurrentUser();

			/** @var Sitemap|null $sitemap */
			$sitemap = Sitemap::fetch($sitemapID);

			if ($sitemap === null){
				return new JSONError(sprintf("No sitemap found for ID %s", $sitemapID));
			}

			SitemapsService::deleteSitemap($sitemap);

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_SITEMAP->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"sitemapID"=>$sitemapID,
				]),
			);

			return new JSONSuccess();
		}
	}