<?php
	namespace ShortcodeParser;

	class ShortcodeLexicalParser{

		/**
		 * Lexically parse a shortcode-formatted string into a Shortcode object.
		 * @param string $codeString E.g., "{{ some-name }}"
		 * @throws ShortcodeSyntaxError
		 * @throws InvalidShortcode
		 */
		public function getShortcodeFromString(string $codeString): Shortcode{
			$codeString = trim($codeString);

			if (!str_starts_with(haystack: $codeString, needle: "{{")){
				throw new ShortcodeSyntaxError("Shortcodes must start with {{");
			}

			if (!str_ends_with(haystack: $codeString, needle: "}}")){
				throw new ShortcodeSyntaxError("Shortcodes must end with }}");
			}

			$innerShortcodeString = substr($codeString, 2, -2);

			$shortcode = new Shortcode();
			$currentUnfinishedShortcodeAttribute = new ShortcodeAttribute();

			// Split the string into an array of characters
			$data = str_split($innerShortcodeString);
			$data[] = "\0"; // To signify the end of the string

			$currentAttribute = "";
			$currentAttributeValue = "";
			$currentParseState = "";
			$parsedName = false; // Has the name already been parsed?
			$currentAttributeOpeningQuoteCharacter = "";

			foreach($data as $char){
				if ($char == "="){
					// Prepare to accept an attribute value

					// Only valid if parsing was consuming the name
					if ($currentParseState == "parsing-attribute-name"){
						// Emit the name to the current unfinished attribute
						$currentUnfinishedShortcodeAttribute->name = $currentAttribute;
						$currentParseState = "parsing-attribute-value";
					}elseif ($currentParseState == "parsing-attribute-value"){
						// Consume it
						$currentAttributeValue .= $char;
					}else{
						// Bogus = sign, ignore it
					}
				}elseif ($char == "\"" || $char == "'"){
					if ($currentParseState == "parsing-attribute-value"){ // Currently, parsing the attribute value
						if ($currentAttributeOpeningQuoteCharacter == ""){
							// There is no current opening quote
							// Set it to this one
							$currentAttributeOpeningQuoteCharacter = $char;
						}else{
							if ($currentAttributeOpeningQuoteCharacter != $char){
								// Consume it
								$currentAttributeValue .= $char;
							}else{
								// Closing quote, ignore it, but set the closing to clean
								$currentParseState = "parsing-attribute-value-closed-clean";
							}
						}
					}
				}elseif ($char == " " || $char == "\0"){
					if ($currentParseState == "parsing-attribute-name"){
						// Parser hit a space after the name
						// Emit the current attribute as a having a blank string value or check
						// if this is actually the name of the shortcode itself.
						$currentParseState = "";

						// Has the shortcode's name been parsed?
						// If not, then what we just buffered is the name of the shortcode
						if ($parsedName === false){
							$parsedName = true;
							$shortcode->name = $currentAttribute;
						}else{
							// Emit a blank shortcode with $currentAttribute as the name
							$currentUnfinishedShortcodeAttribute->name = $currentAttribute;
							$shortcode->attributes[] = $currentUnfinishedShortcodeAttribute;

							// Begin the next shortcode attribute
							$currentUnfinishedShortcodeAttribute = new ShortcodeAttribute();
						}

						$currentAttribute = "";
					}elseif ($currentParseState == "parsing-attribute-value-closed-clean"){
						$currentParseState = ""; // No longer in any state

						// Set the value of the current unfinished attribute
						$currentUnfinishedShortcodeAttribute->value = $currentAttributeValue;
						$shortcode->attributes[] = $currentUnfinishedShortcodeAttribute;

						// Clean up buffered variables
						$currentAttribute = "";
						$currentAttributeValue = "";
						$currentAttributeOpeningQuoteCharacter = "";

						// Begin the next shortcode attribute
						$currentUnfinishedShortcodeAttribute = new ShortcodeAttribute();
					}elseif ($currentParseState == "parsing-attribute-value" && $currentAttributeOpeningQuoteCharacter != ""){
						if ($char === " "){
							// Just part of the value
							$currentAttributeValue .= $char;
						}else{
							// Did not cleanly close. Throw it away and reset the current unfinished attribute
							// It's bogus, and we won't keep it
							$currentParseState = "";
							$currentAttribute = "";
							$currentAttributeValue = "";
							$currentUnfinishedShortcodeAttribute->name = "";
							$currentUnfinishedShortcodeAttribute->value = "";
						}
					}elseif ($currentParseState == "parsing-attribute-value" && $currentAttributeOpeningQuoteCharacter == ""){
						// Cleanly closed, it was never encased in quotes
						$currentParseState = ""; // No longer in any state
						$currentUnfinishedShortcodeAttribute->value = $currentAttributeValue;
						$shortcode->attributes[] = $currentUnfinishedShortcodeAttribute;
						$currentAttribute = "";
						$currentAttributeValue = "";

						// Begin the next shortcode attribute
						$currentUnfinishedShortcodeAttribute = new ShortcodeAttribute();
					}
				}else{
					// It is none of the above
					if ($currentParseState == ""){
						$currentParseState = "parsing-attribute-name";
					}

					// Consume the character
					if ($currentParseState == "parsing-attribute-name"){
						$currentAttribute .= $char;
					}elseif ($currentParseState == "parsing-attribute-value"){
						$currentAttributeValue .= $char;
					}
				}
			}

			if (empty(trim($shortcode->name))){
				throw new InvalidShortcode("No shortcode name was found in the provided code string: \"{$codeString}\".");
			}

			return $shortcode;
		}
	}