<?php
	namespace ShortcodeParser\Processors;

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\HttpHelper;
	use System\ContentHelper;
	use System\Themes;

	/**
	 * Processor class for the get-services shortcode
	 */
	class GetServicesList extends Processor {

		protected static array $supportedAttributes = [
			"limit",
			"delimiter",
			"tagNameWrapper",
			"attributes",
			"sort",
			"shortcode-file",
			"exclude-page-ids",
		];

		protected static array $supportedTemplateTags = [		
			"{{ SERVICE_NAME }}",
			"{{ SERVICE_URI }}",
			"{{ PAGE_URI }}",
			"{{ SERVICE_PREVIEW }}",
			"{{ PAGE_PREVIEW }}",
			"{{ CURRENT_INDEX }}",
			"{{ SERVICE_THUMB }}",
			"{{ FEATURED_IMAGE }}",
			"{{ FEATURED_IMAGE_NOTHUMB }}",
			"{{ NUM_ITEMS }}",
		];

		public static function getSupportedAttributeNames(): array {
			return self::$supportedAttributes;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws Exception
		 */
		public function runProcessor(): string{

			$pageLimit = 10;
			$delimiter = "";
			$htmlTagNameWrapper = "";
			$sortString = "creation asc";

			$limitAttribute = $this->shortcode->getAttribute("limit");
			if ($limitAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Attribute 'limit' is required for the %s attribute.",
						ShortcodeTypes::GET_LIST_OF_SERVICES->value,
					)
				);
			}elseif((int) $limitAttribute->value === 0){
				$pageLimit = 9999;
			}else{
				$pageLimit = (int) $limitAttribute->value;
			}

			$sortAttribute = $this->shortcode->getAttribute("sort");
			if ($sortAttribute !== null){
				$sortString = $sortAttribute->value;
			}

			$delimiterAttribute = $this->shortcode->getAttribute("delimiter");
			if ($delimiterAttribute !== null){
				$delimiter = $delimiterAttribute->value;
			}

			$htmlTagNameWrapperAttribute = $this->shortcode->getAttribute("tagNameWrapper");
			if ($htmlTagNameWrapperAttribute !== null){
				$htmlTagNameWrapper = $htmlTagNameWrapperAttribute->value;
			}

			// Get view file path
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			
			// If no shortcode-file is specified, use generateDefaultOutput
			if ($shortcodeViewFileAttribute === null){
				return $this->generateDefaultOutput($delimiter, $htmlTagNameWrapper, $pageLimit, $sortString);
			}

			$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			$viewFileNormalizedPath = realpath($viewFile);

			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			// Load view file contents
			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Retrieve the excluded page IDs attribute
			$excludedPageIdsAttribute = $this->shortcode->getAttribute("exclude-page-ids");
			$excludedPageIds = [];
			if ($excludedPageIdsAttribute !== null) {
				// Decode the excluded page IDs from JSON
				$excludedPageIds = json_decode($excludedPageIdsAttribute->value);
			}

			// Query for service pages, excluding the specified page IDs
			$resultOrder = $this->getResultOrder($sortString);
			$columnQuery = (new ColumnQuery())
				->where("pageType", "=", PageType::Service->name)
				->and()
				->where("publication_status", "=", PublicationStatus::Published->value);

			// Exclude pages based on excludedPageIds
			if (!empty($excludedPageIds) && is_array($excludedPageIds)) {
				$columnQuery->and()->where("id", "NOT IN", "(" . implode(",", $excludedPageIds) . ")");
			}

			$servicePages = Page::query(
				columnQuery: $columnQuery,
				resultOrder: $resultOrder,
				pager: (new Pager(pageNumber: 1, limit: $pageLimit))
			);

			// Process ServiceItem template in the view file
			$finalOutput = preg_replace_callback(
				"/{{ begin ServiceItem }}(.+?){{ end ServiceItem }}/ism",
				function($matches) use ($servicePages) {
					$template = $matches[1];
					$renderedItems = "";
					$currentIndex = 0;

					foreach ($servicePages as $servicePage) {
						$renderedItems .= $this->renderServiceTemplate($servicePage, $template, $currentIndex);
						$currentIndex++;
					}

					return $renderedItems;
				},
				$viewFileContents
			);

			$finalOutput = str_replace("{{ NUM_ITEMS }}", count($servicePages), $finalOutput);

			return $finalOutput;
		}

		private function getResultOrder(string $sortString): ResultOrder {
			return match($sortString) {
				"creation asc" => (new ResultOrder())
					->by("creationTime", "asc")
					->by("publication_timestamp", "asc"),
				"creation desc" => (new ResultOrder())
					->by("creationTime", "desc")
					->by("publication_timestamp", "desc"),
				"alphabetical asc" => (new ResultOrder())
					->by("pageName", "asc"),
				"alphabetical desc" => (new ResultOrder())
					->by("pageName", "desc"),
				default => throw new InvalidAttributeValue("The value of 'sort' was not a recognized string."),
			};
		}

		private function renderServiceTemplate(Page $servicePage, string $template, int $currentIndex): string {
			// Fetch the default content section or use the page body as fallback
			$defaultOrFirstContentSection = $servicePage->getDefaultContentSectionOrFirstSection();
			
			// Generate preview text using ContentHelper
			$previewText = ContentHelper::makePreviewFromBody(
				pageBody: $defaultOrFirstContentSection?->content ?? $servicePage->pageBody,
			);

			return str_replace(
				[
					"{{ SERVICE_NAME }}",
					"{{ SERVICE_URI }}",
					"{{ PAGE_URI }}",
					"{{ SERVICE_PREVIEW }}",
					"{{ PAGE_PREVIEW }}",
					"{{ CURRENT_INDEX }}",
					"{{ SERVICE_THUMB }}",
					"{{ FEATURED_IMAGE }}",
					"{{ FEATURED_IMAGE_NOTHUMB }}"
				],
				[
					$servicePage->pageName,
					$servicePage->pageRoute ?? "#",
					$servicePage->pageRoute ?? "#",
					$previewText . "[&hellip;]",
					$previewText . "[&hellip;]",
					$currentIndex,
					$servicePage->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB)[0]->value,
					$servicePage->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB)[0]->value,
					$servicePage->getPageDatas(PageDatas::FEATURED_IMAGE)[0]->value
				],
				$template
			);
		}

		private function generateDefaultOutput(string $delimiter, string $htmlTagNameWrapper, int $pageLimit, string $sortString): string {
			$htmlAttributes = [];
			
			// Build the HTML attributes string
			$htmlAttributesAttribute = $this->shortcode->getAttribute("attributes");
			if ($htmlAttributesAttribute !== null){
				$htmlAttributes = json_decode($htmlAttributesAttribute->value);
				if ($htmlAttributes === null){
					throw new InvalidAttributeValue(sprintf(
						"The value of the 'attributes' attribute was not valid JSON. The error received was: %s",
						json_last_error_msg(),
					));
				}
			}

			$htmlAttributesString = ""; // Not padded yet
			if ($htmlAttributes !== null){
				foreach($htmlAttributes as $attributeName=>$attributeValue){
					$htmlAttributesString .= sprintf(
						" %s=\"%s\"",
						$attributeName,
						$attributeValue,
					);
				}
			}
			$htmlAttributesString .= " "; // Pad the end

			// Fetch the exclude-page-ids attribute and decode it
			$excludedPageIdsAttribute = $this->shortcode->getAttribute("exclude-page-ids");
			$excludedPageIds = [];
			if ($excludedPageIdsAttribute !== null) {
				$excludedPageIds = json_decode($excludedPageIdsAttribute->value);
			}

			// Process service page list generation
			$columnQuery = (new ColumnQuery())
				->where("pageType", "=", PageType::Service->name)
				->and()
				->where("publication_status", "=", PublicationStatus::Published->value);

			// Exclude pages based on excludedPageIds
			if (!empty($excludedPageIds) && is_array($excludedPageIds)) {
				$columnQuery->and()->where("id", "NOT IN", "(" . implode(",", $excludedPageIds) . ")");
			}

			$servicePages = Page::query(
				columnQuery: $columnQuery,
				resultOrder: $this->getResultOrder($sortString),
				pager: (new Pager(pageNumber: 1, limit: $pageLimit))
			);

			$finalResult = "";
			foreach ($servicePages as $servicePage) {
				$htmlAttributes = $htmlAttributesString;
				if ($htmlTagNameWrapper === "a"){
					$htmlAttributes .= sprintf(" href=\"%s\"", $servicePage->pageRoute);
				}
				if (empty($htmlTagNameWrapper)) {
					$finalResult .= sprintf(
						"%s%s",
						$servicePage->pageName ?? "",
						$delimiter,
					);
				} else {
					$finalResult .= sprintf(
						"<%s%s>%s</%s>%s",
						$htmlTagNameWrapper,
						$htmlAttributes,
						$servicePage->pageName ?? "",
						$htmlTagNameWrapper,
						$delimiter,
					);
				}
			}

			return rtrim($finalResult, $delimiter);
		}
	}
