<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";
	require_once __DIR__ . "/ShortcodeViewProcessor.php";

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use Reviews\Review;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\ContentHelper;
	use System\Themes;

	/**
	 * Processor class for the get-random-pages shortcode
	 */
	class GetRandomPages extends ShortcodeViewProcessor {

		protected static string $defaultShortcodeViewBaseFileName = "random-pages.php";

		protected static array $supportedTemplateTags = [
			"{{ begin RandomPageItem }}",
			"{{ end RandomPageItem }}",
			"{{ PAGE_NAME }}",
			"{{ PAGE_URI }}",
		];

		protected static array $supportedAttributes = [
			"limit",
			"shortcode-file",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public static function getDefaultShortcodeViewBaseFileName(): string{
			return self::$defaultShortcodeViewBaseFileName;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws MissingRequiredAttribute
		 * @throws ShortcodeViewFileNotFound
		 */
		public function runProcessor(): string{

			$limitAttribute = $this->shortcode->getAttribute("limit");
			if ($limitAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Missing required 'limit' attribute for shortcode %s",
						ShortcodeTypes::GET_RANDOM_PAGES->value,
					)
				);
			}else{
				$limit = (int) $limitAttribute->value;
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			// Is this using a custom shortcode view file?
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			if ($shortcodeViewFileAttribute !== null){
				// Fetch the file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			}else{
				// Use the default file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			}

			// Get the view file and then its contents
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Replace the template tag HTML wrapper with the necessary articles to show
			$viewFileContents = preg_replace_callback(
				pattern:"/{{ begin RandomPageItem }}(.+?){{ end RandomPageItem }}/ism",
				callback: function($matches) use ($limit){
					$template = $matches[1];
					$finalStringToRender = "";

					// Get random pages
					/** @var Page[] $randomPages */
					$randomPages = Page::query(
						resultOrder: (new ResultOrder())
							->byFunction("RAND()"),
						pager:(new Pager(pageNumber: 1, limit: $limit)),
					);

					foreach($randomPages as $page){
						// Make a copy of the string
						$thisPageRenderString = $template;

						// Make all template tag replacements
						$thisPageRenderString = str_replace(
							search:"{{ PAGE_NAME }}",
							replace:$page->pageName,
							subject:$thisPageRenderString,
						);

						$thisPageRenderString = str_replace(
							search:"{{ PAGE_URI }}",
							replace:$page->pageRoute,
							subject:$thisPageRenderString,
						);

						$finalStringToRender .= $thisPageRenderString;
					}

					return $finalStringToRender;
				},
				subject:$viewFileContents
			);

			return $viewFileContents;
		}
	}