<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";
	require_once __DIR__ . "/ShortcodeViewProcessor.php";

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\ContentHelper;
	use System\Themes;

	// TODO Work on this
	/**
	 * Processor class for the get-projects shortcode
	 */
	class GetProjects extends ShortcodeViewProcessor {

		protected static string $defaultShortcodeViewBaseFileName = "projects-reel.php";

		protected static array $supportedTemplateTags = [
			"{{ COLUMNS }}",
			"{{ PER_PAGE_LIMIT }}",
			"{{ begin ProjectsItem }}",
			"{{ end ProjectsItem }}",
			"{{ PROJECT_TITLE }}",
			"{{ FEATURED_IMAGE }}",
			"{{ FEATURED_IMAGE_NOTHUMB }}",
			"{{ PROJECT_PREVIEW }}",
			"{{ PAGE_URI }}",
			"{{ CREATION_DATE }}",
		];

		protected static array $supportedAttributes = [
			"ipps-per-page",
			"columns",
			"included-project-tags",
			"excluded-project-tags",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public static function getDefaultShortcodeViewBaseFileName(): string{
			return self::$defaultShortcodeViewBaseFileName;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws MissingRequiredAttribute
		 * @throws InvalidAttributeValue
		 * @throws Exception
		 */
		public function runProcessor(): string{

			// Required attribute
			$projectsPerPageAttribute = $this->shortcode->getAttribute("ipps-per-page");
			if ($projectsPerPageAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Attribute 'ipps-per-page' is required in %s",
						ShortcodeTypes::GET_PROJECTS->value,
					)
				);
			}else{
				$projectsPerPage = (int) $projectsPerPageAttribute->value;
			}

			// Handle columns + validation
			$columnsAttribute = $this->shortcode->getAttribute("columns");
			if ($columnsAttribute === null){
				$columns = 1;
			}else{
				$columns = (int) $columnsAttribute->value;
				if ($columns === 0 || $columns < 0){
					throw new InvalidAttributeValue("The value of 'columns' attribute must be a valid, positive integer.");
				}
			}



			// Exclusion and inclusion article category IDs
			$includedProjectTagIDsAttribute = $this->shortcode->getAttribute("included-project-tags");
			$includedProjectTagIDs = [];
			if ($includedProjectTagIDsAttribute !== null){
				// JSON string, then parse it
				$jsonString = $includedProjectTagIDsAttribute->value;
				if (!empty($jsonString)) {
					$jsonIncludedIDs = json_decode($includedProjectTagIDsAttribute->value, true);
					if ($jsonIncludedIDs !== null) {
						$includedProjectTagIDs = $jsonIncludedIDs;
					}else{
						throw new InvalidAttributeValue(
							sprintf(
								"The JSON value for 'included-categories' was invalid. The error received was: %s",
								json_last_error_msg(),
							)
						);
					}
				}
			}

			$excludedProjectTagIDsAttribute = $this->shortcode->getAttribute("excluded-project-tags");
			$excludedProjectTagIDs = [];
			if ($excludedProjectTagIDsAttribute !== null){
				// JSON string, then parse it
				$jsonString = $excludedProjectTagIDsAttribute->value;
				if (!empty($jsonString)) {
					$jsonExcludedIDs = json_decode($excludedProjectTagIDsAttribute->value, true);
					if ($jsonExcludedIDs !== null) {
						$excludedProjectTagIDs = $jsonExcludedIDs;
					}else{
						throw new InvalidAttributeValue(
							sprintf(
								"The JSON value for 'excluded-project-tags' was invalid. The error received was: %s",
								json_last_error_msg(),
							)
						);
					}
				}
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			// Get the view file and then its contents
			$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Replace the template tags
			$viewFileContents = str_replace(
				search: "{{ COLUMNS }}",
				replace: $columns,
				subject: $viewFileContents,
			);

			$viewFileContents = str_replace(
				search: "{{ PER_PAGE_LIMIT }}",
				replace: $projectsPerPage,
				subject: $viewFileContents,
			);

			// Replace the template tag HTML wrapper with the necessary articles to show

			$viewFileContents = preg_replace_callback(
				pattern:"/{{ begin ProjectsItem }}(.+?){{ end ProjectsItem }}/ism",
				callback: function($matches) use ($includedProjectTagIDs, $excludedProjectTagIDs, $projectsPerPage){
					$template = $matches[1];
					$finalStringToRender = "";

					// NOTE
					// Algorithm for this code is described in the GetArticles.php file in the same location -
					// preg_replace_callback

					/** @var Page[] $pagesToRender */
					$pagesToRender = [];

					// Handle where included category IDs is not empty
					if (!empty($includedProjectTagIDs)){
						/** @var array[] $pageProjectTagIDs */
						$pageProjectTagIDs = [];
						/** @var PageData[] $pageDatas */
						$pageDatas = PageData::query(
							columnQuery: (new ColumnQuery())
								->where("name","=",PageDatas::PROJECT_TAG->name)
						);

						// Build an array of article categories identified by the page ID
						foreach($pageDatas as $pageData){
							// Only consider project tag IDs that are in the included project tag IDs array
							if (in_array($pageData->value, $includedProjectTagIDs)) {
								if (array_key_exists(key: $pageData->pageID, array: $pageProjectTagIDs)) {
									$pageProjectTagIDs[$pageData->pageID][] = (int)$pageData->value;
								} else {
									$pageProjectTagIDs[$pageData->pageID] = [
										(int)$pageData->value,
									];
								}
							}
						}

						// Filter out pageIDs that have an excluded project tag ID - if any
						if (!empty($excludedProjectTagIDs)){
							foreach($pageProjectTagIDs as $pageID=>$projectTagIDsForPage){
								// If this array is empty, then the article isn't excluded,
								// else if the array is not empty then the article contains an
								// excluded category.
								if (!empty(array_intersect($projectTagIDsForPage, $excludedProjectTagIDs))){
									// Unset it
									unset($pageProjectTagIDs[$pageID]);
								}
							}
						}

						// If this is empty, then the query would throw an error (MySQL syntax: 'IN ()' is invalid)
						if (!empty($pageProjectTagIDs)) {
							// Fetch the pages now
							/** @var Page[] $pagesToRender */
							$pagesToRender = Page::query(
								columnQuery: (new ColumnQuery())
									->where("pageType", "=", PageType::Project->name)
									->and()
									->where("publication_status", "=", PublicationStatus::Published->value)
									->and()
									->where("publication_timestamp", "<=", time())
									->and()
									->where("id", "IN", sprintf(
										"(%s)",
										implode(
											",",
											array_keys($pageProjectTagIDs) // Keys are page IDs
										)
									)),
								resultOrder: (new ResultOrder())
									->by("publication_timestamp", "DESC")
									->by("creationTime", "DESC"),
							);
						}
					}else{
						// Included category IDs is empty
						// Are there excluded categories?
						if (!empty($excludedProjectTagIDs)){
							// There are excluded project tag IDs. This is the heaviest and most expensive query (because
							// all pages have to be fetched and then narrowed down)

							/** @var Page[] $pagesToFilter */
							$pagesToFilter = Page::query(
								columnQuery: (new ColumnQuery())
									->where("pageType", "=", PageType::Project->name)
									->and()
									->where("publication_status", "=", PublicationStatus::Published->value)
									->and()
									->where("publication_timestamp","<=", time()),
								resultOrder:(new ResultOrder())
									->by("publication_timestamp","DESC")
									->by("creationTime","DESC"),
							);

							// Get an array of page IDs
							$pageIDs = [];
							foreach($pagesToFilter as $page){
								$pageIDs[] = $page->id;
							}

							// Fetch all project tags for the pages found
							/** @var PageData[] $projectPagePageDatas */
							$projectPagePageDatas = PageData::query(
								columnQuery: (new ColumnQuery())
									->where("name","=",PageDatas::PROJECT_TAG->name)
									->and()
									->where("page_id","IN",
										sprintf(
										"(%s)",
											implode(",", $pageIDs)
										)
									),
							);

							// Sort them by page ID
							$pageProjectTagIDs = [];

							// Build an array of each page's article categories
							// Each key in this array is an integer page ID
							// and the value is an array of that page's article categories
							foreach($projectPagePageDatas as $pageData){
								if (array_key_exists(key: $pageData->pageID, array: $pageProjectTagIDs)){
									$pageProjectTagIDs[$pageData->pageID][] = (int) $pageData->value;
								}else{
									$pageProjectTagIDs[$pageData->pageID] = [
										(int) $pageData->value,
									];
								}
							}

							// Iterate over
							foreach($pagesToFilter as $page){
								// Check if this page has an article category that is excluded
								// If so, skip this page

								// Does this page have article categories?
								if (array_key_exists(key: $page->id, array: $pageProjectTagIDs)){
									// Check if there are any excluded category IDs
									if (!empty(array_intersect($pageProjectTagIDs[$page->id], $excludedProjectTagIDs))){
										// There are exclusions, skip this page
										continue;
									}
								}

								// If made it here, then add this page to the list of pages to render
								$pagesToRender[] = $page;

								// Was the limit hit? Then break this loop
								if (count($pagesToFilter) >= $projectsPerPage){
									break;
								}
							}
						}else{
							// At this point there are no included category IDs and no excluded IDs.
							// This is a simple, limited fetch query of articles that are published
							/** @var Page[] $pagesToRender */
							$pagesToRender = Page::query(
								columnQuery: (new ColumnQuery())
									->where("pageType", "=", PageType::Project->name)
									->and()
									->where("publication_status", "=", PublicationStatus::Published->value)
									->and()
									->where("publication_timestamp","<=", time()),
								resultOrder:(new ResultOrder())
									->by("publication_timestamp","DESC")
									->by("creationTime","DESC"),
							);
						}
					}

					// Render the articles into the shortcode view now
					foreach($pagesToRender as $page){
						// Make a copy of the string
						$thisPageRenderString = $template;

						// Make all template tag replacements
						$thisPageRenderString = str_replace(
							search:"{{ PROJECT_TITLE }}",
							replace:$page->pageName,
							subject:$thisPageRenderString,
						);

						$featuredImageDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE);
						$featuredImageThumbDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);

						if (!empty($featuredImageThumbDataArray)) {
							$thisPageRenderString = str_replace(
								search: "{{ FEATURED_IMAGE }}", // This actually is the thumb template tag. I know, dumb
								replace: $featuredImageThumbDataArray[0]->value,
								subject: $thisPageRenderString,
							);
						}

						if (!empty($featuredImageDataArray)) {
							$thisPageRenderString = str_replace(
								search: "{{ FEATURED_IMAGE_NOTHUMB }}",
								replace: $featuredImageDataArray[0]->value,
								subject: $thisPageRenderString,
							);
						}

						// Get a preview text for this project's default/first content section, or page body if no sections
						$defaultOrFirstContentSection = $page->getDefaultContentSectionOrFirstSection();
						$previewText = ContentHelper::makePreviewFromBody(
							pageBody: $defaultOrFirstContentSection?->content ?? $page->pageBody,
						);

						$thisPageRenderString = str_replace(
							search: "{{ PROJECT_PREVIEW }}",
							replace: $previewText . "[&hellip;]",
							subject: $thisPageRenderString,
						);

						$thisPageRenderString = str_replace(
							search: "{{ PAGE_URI }}",
							replace: $page->pageRoute,
							subject: $thisPageRenderString,
						);

						$thisPageRenderString = str_replace(
							search: "{{ CREATION_DATE }}",
							replace: date(
								format: "F j, Y",
								timestamp: $page->creationTime
							),
							subject: $thisPageRenderString,
						);

						$finalStringToRender .= $thisPageRenderString;
					}

					return $finalStringToRender;
				},
				subject:$viewFileContents
			);

			return $viewFileContents;
		}
	}