<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";
	require_once __DIR__ . "/ShortcodeViewProcessor.php";

	use Exception;
	use Page\Page;
	use Page\PageDatas;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use System\ContentHelper;
	use System\Themes;

	/**
	 * Processor class for the get-pages shortcode
	 */
	class GetPages extends ShortcodeViewProcessor {
		protected static string $defaultShortcodeViewBaseFileName = "pages.php";

		protected static array $supportedTemplateTags = [
			"{{ begin PageItem }}",
			"{{ end PageItem }}",
			"{{ PAGE_NAME }}",
			"{{ FEATURED_IMAGE }}",
			"{{ FEATURED_IMAGE_NOTHUMB }}",
			"{{ PAGE_PREVIEW }}",
			"{{ PAGE_URI }}",
			"{{ CREATION_DATE }}",
		];

		protected static array $supportedAttributes = [
			"page-ids",
			"shortcode-file",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public static function getDefaultShortcodeViewBaseFileName(): string{
			return self::$defaultShortcodeViewBaseFileName;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @return string
		 * @throws InvalidAttributeValue
		 * @throws ShortcodeViewFileNotFound
		 */
		public function runProcessor(): string{
			$pageIDsAttribute = $this->shortcode->getAttribute("page-ids");
			$pageIDs = [];
			if ($pageIDsAttribute !== null){
				$jsonString = $pageIDsAttribute->value;
				if (!empty($jsonString)) {
					$jsonPageIDs = json_decode($pageIDsAttribute->value, true);
					if ($jsonPageIDs !== null) {
						$pageIDs = $jsonPageIDs;
					}else{
						throw new InvalidAttributeValue(
							sprintf(
								"The JSON value for 'page-ids' was invalid. The error received was: %s",
								json_last_error_msg(),
							)
						);
					}
				}
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			// Is this using a custom shortcode view file?
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			if ($shortcodeViewFileAttribute !== null){
				// Fetch the file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			}else{
				// Use the default file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			}

			// Get the view file and then its contents
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			$viewFileContents = preg_replace_callback(
				pattern:"/{{ begin PageItem }}(.+?){{ end PageItem }}/ism",
				callback:function($matches) use ($pageIDs){
					$template = $matches[1];
					$finalStringToRender = "";

					$pagesToRender = [];

					foreach($pageIDs as $pageID){
						/** @var ?Page $page */
						$page = Page::fetch($pageID);

						if ($page !== null) {

							// Make a copy of the string
							$thisPageRenderString = $template;

							// Make all template tag replacements
							$thisPageRenderString = str_replace(
								search: "{{ PAGE_NAME }}",
								replace: $page->pageName,
								subject: $thisPageRenderString,
							);

							$featuredImageDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE);
							$featuredImageThumbDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);

							if (!empty($featuredImageThumbDataArray)) {
								$thisPageRenderString = str_replace(
									search: "{{ FEATURED_IMAGE }}", // This actually is the thumb template tag. I know, dumb
									replace: $featuredImageThumbDataArray[0]->value,
									subject: $thisPageRenderString,
								);
							}

							if (!empty($featuredImageDataArray)) {
								$thisPageRenderString = str_replace(
									search: "{{ FEATURED_IMAGE_NOTHUMB }}",
									replace: $featuredImageDataArray[0]->value,
									subject: $thisPageRenderString,
								);
							}

							// Get a preview text for this page's default/first content section, or page body if no sections
							$defaultOrFirstContentSection = $page->getDefaultContentSectionOrFirstSection();
							$previewText = ContentHelper::makePreviewFromBody(
								pageBody: $defaultOrFirstContentSection?->content ?? $page->pageBody,
							);

							$thisPageRenderString = str_replace(
								search: "{{ PAGE_PREVIEW }}",
								replace: $previewText . "[&hellip;]",
								subject: $thisPageRenderString,
							);

							$thisPageRenderString = str_replace(
								search: "{{ PAGE_URI }}",
								replace: $page->pageRoute,
								subject: $thisPageRenderString,
							);

							$thisPageRenderString = str_replace(
								search: "{{ CREATION_DATE }}",
								replace: date(
									format: "F j, Y",
									timestamp: $page->creationTime
								),
								subject: $thisPageRenderString,
							);

							$finalStringToRender .= $thisPageRenderString;
						}
					}

					return $finalStringToRender;
				},
				subject: $viewFileContents
			);

			return $viewFileContents;
		}
	}