<?php

	namespace ShortcodeParser\Processors;

	use Exception;
	use ImageGallery\ImageGallery;
	use ImageGallery\ImageGalleryMember;
	use ImageGallery\ImageGallerySortBy;
	use ImageGallery\ImageGallerySortDirection;
	use ImageGallery\NoImageGalleryFound;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\ResultOrder;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeParser;
	use ShortcodeParser\ShortcodeTypes;
	use System\Themes;

	require_once __DIR__ . "/Processor.php";

	class GetGallery extends Processor {
		protected static string $defaultShortcodeViewBaseFileName = "gallery.php";

		protected static array $supportedTemplateTags = [
			"{{ begin GalleryImageItem }}",
			"{{ end GalleryImageItem }}",
			"{{ ImageUri }}",
			"{{ ImageAlt }}",
			"{{ ThumbUri }}",
			"{{ GalleryName }}",
		];

		protected static array $supportedAttributes = [
			"id", "gallery-name", "shortcode-file", "sort-by", "sort-direction"
		];

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @return string
		 * @throws MissingRequiredAttribute
		 * @throws NoImageGalleryFound
		 * @throws Exception
		 */
		public function runProcessor():string
		{
			$idAttribute = $this->shortcode->getAttribute("id");
			$galleryNameAttribute = $this->shortcode->getAttribute("gallery-name");
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			$sortByAttribute = $this->shortcode->getAttribute("sort-by");
			$sortDirectionAttribute = $this->shortcode->getAttribute("sort-direction");

			// required id attribute
			if ($idAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Attribute 'id' is required in %s",
						ShortcodeTypes::GET_GALLERY->value,
					)
				);
			}

			// required gallery name attribute
			if ($galleryNameAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Attribute 'gallery-name' is required in %s",
						ShortcodeTypes::GET_GALLERY->value,
					)
				);
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			if ($shortcodeViewFileAttribute !== null){
				// Fetch the file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			}else{
				// Use the default file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			}

			// Get the view file and then its contents
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Check if the Gallery exists
			/** @var ?ImageGallery $gallery */
			$gallery = ImageGallery::fetch($idAttribute->value);

			if ($gallery === null){
				throw new NoImageGalleryFound("No image gallery found with ID " . $idAttribute->value);
			}

			// Replace the template tag HTML wrapper
			$viewFileContents = preg_replace_callback(
				pattern:"/{{ begin GalleryImageItem }}(.+?){{ end GalleryImageItem }}/ism",
				callback: function($matches) use ($idAttribute, $galleryNameAttribute, $sortByAttribute, $sortDirectionAttribute){
					$template = $matches[1];
					$finalStringToRender = "";
					$sortBy = "position";
					$sortDirection = "ASC";

					// Check if $sortByAttribute is empty
					if(!empty($sortByAttribute)) {
						$case = ImageGallerySortBy::getByValue($sortByAttribute->value);

						// Check if $sortByAttribute->value exists in enum
						if(!empty($case)) {
							// Set $sortBy to $sortByAttribute->value
							$sortBy = $sortByAttribute->value;
						}
					}

					// Check if $sortDirectionAttribute is empty
					if(!empty($sortDirectionAttribute)) {
						$case = ImageGallerySortDirection::getByValue($sortDirectionAttribute->value);

						// Check if $sortDirectionAttribute->value exists in enum
						if(!empty($case)) {
							// Set $sortBy to $sortDirectionAttribute->value
							$sortDirection = $sortDirectionAttribute->value;
						}
					}

					$galleryItems = ImageGalleryMember::query(
						columnQuery: (new ColumnQuery())
							->where("gallery_id","=", $idAttribute->value),
						resultOrder:(new ResultOrder())
							->by($sortBy, $sortDirection),
					);

					foreach($galleryItems as $key=>$galleryItem) {
						// Make a copy of the string
						$thisPageRenderString = $template;

						// Make all template tag replacements
						$thisPageRenderString = str_replace(
							search:"{{ ImageUri }}",
							replace:sprintf("%s", $galleryItem->imageURI),
							subject:$thisPageRenderString,
						);

						// Checks to see if the gallery item has a custom alt attribute, if not use gallery name + key
						if($galleryItem->altAttribute) {
							$thisPageRenderString = str_replace(
								search:"{{ ImageAlt }}",
								replace:sprintf("%s", $galleryItem->altAttribute),
								subject:$thisPageRenderString,
							);
						} else {
							$thisPageRenderString = str_replace(
								search:"{{ ImageAlt }}",
								replace:sprintf("%s", $galleryNameAttribute->value . " " . $key),
								subject:$thisPageRenderString,
							);
						}

						$thisPageRenderString = str_replace(
							search:"{{ ThumbUri }}",
							replace:sprintf("%s", $galleryItem->thumbURI),
							subject:$thisPageRenderString,
						);

						$thisPageRenderString = str_replace(
							search:"{{ GalleryName }}",
							replace:sprintf("%s", $galleryNameAttribute->value),
							subject:$thisPageRenderString,
						);

						$finalStringToRender .= $thisPageRenderString;
					}

					return $finalStringToRender;
				},
				subject:$viewFileContents
			);

			return $viewFileContents;
		}
	}