<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";

	use Page\Page;
	use Page\PageDatas;
	use Settings\Setting;
	use Settings\Settings;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\MissingSettingValueForShortcode;
	use ShortcodeParser\Processors\Exceptions\NoPageCurrentlyRendered;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\PhoneNumberUtility;

	/**
	 * Processor class for the fax-number shortcode
	 */
	class GetCompanyFaxNumber extends Processor {

		protected static array $supportedAttributes = [
			"which",
			"format",
			"pure-number"
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws MissingRequiredAttribute
		 * @throws InvalidAttributeValue
		 * @throws MissingSettingValueForShortcode
		 */
		public function runProcessor(): string{
			$faxNumbersJSON = Setting::getSettingValue(Settings::COMPANY_FAX_NUMBERS->value);
			$faxNumbers = json_decode($faxNumbersJSON, true);
			if (is_array($faxNumbers) && !empty($faxNumbers)){
				$isPureNumber = false;

				$whichIndexAttribute = $this->shortcode->getAttribute("which");
				if ($whichIndexAttribute === null){
					throw new MissingRequiredAttribute(
						sprintf(
							"Missing required 'which' attribute on shortcode %s",
							ShortcodeTypes::FAX_NUMBER->value,
						)
					);
				}else{
					$whichIndex = (int) $whichIndexAttribute->value;
				}

				// Only care if 'pure-number' is there and not equal to "false" or "no"
				$pureNumberAttribute = $this->shortcode->getAttribute("pure-number");
				if ($pureNumberAttribute !== null){
					if ($pureNumberAttribute->value !== "false" && $pureNumberAttribute->value !== "no") {
						$isPureNumber = true;
					}
				}

				if ($whichIndex <= 0){
					throw new InvalidAttributeValue(
						"The value for 'which' must be greater than or equal to 1."
					);
				}

				// Get the actual array index. Which's index is 1-based. We need 0 based
				$arrayIndex = $whichIndex - 1;
				$faxNumber = $faxNumbers[$arrayIndex];

				// A pure number will convert any letters to their phone-digit
				if ($isPureNumber){
					$faxNumber = PhoneNumberUtility::getPurePhoneNumber($faxNumber);
				}

				$formatAttribute = $this->shortcode->getAttribute("format");
				if ($formatAttribute !== null){
					$numberFormat = $formatAttribute->value;

					$numberOfDigitsInFaxNumber = preg_match_all( "/[0-9]/", $faxNumber, $faxNumberDigitsMatch);
					$numberOfCharactersForFormatting = substr_count($numberFormat, 'x');

					if ($numberOfDigitsInFaxNumber !== $numberOfCharactersForFormatting) {
						// There is not an equal amount of phone number digit characters and format characters
						throw new InvalidAttributeValue(
							sprintf(
								"The value for the 'format' attribute is malformed. The number of numerical digits in the company fax number is not equal to the amount of format characters in the 'format' attribute (x's). They must be equal. You provided %d format characters but the fax number requires there to be %d. Company fax number being used is: %s | You provided the format string %s",
								$numberOfCharactersForFormatting,
								$numberOfDigitsInFaxNumber,
								$faxNumber,
								$numberFormat,
							)
						);
					}

					foreach ($faxNumberDigitsMatch[0] as $char) {
						$from = '/'.preg_quote('x', '/').'/';
						$faxNumber = preg_replace(
							pattern:$from,
							replacement:$char,
							subject: $faxNumber,
							limit: 1
						);
					}
				}

				return $faxNumber;
			}else{
				throw new MissingSettingValueForShortcode(
					sprintf(
						"There is no setting defined in the system for %s - which is required for the shortcode %s",
						Settings::COMPANY_FAX_NUMBERS->value,
						ShortcodeTypes::FAX_NUMBER->value,
					)
				);
			}
		}
	}
