<?php
	namespace ShortcodeParser\Processors;

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\HttpHelper;
	use System\ContentHelper;
	use System\Themes;

	/**
	 * Processor class for the get-cities shortcode
	 */
	class GetCitiesList extends Processor {

		protected static array $supportedAttributes = [
			"limit",
			"delimiter",
			"tagNameWrapper",
			"attributes",
			"sort",
			"shortcode-file",
		];

		protected static array $supportedTemplateTags = [		
			"{{ CITY_NAME }}",
			"{{ CITY_STATE }}",
			"{{ CITY_STATE_SHORTHAND }}",
			"{{ CITY_URI }}",
			"{{ PAGE_URI }}",
			"{{ CITY_PREVIEW }}",
			"{{ PAGE_PREVIEW }}",
			"{{ CITY_WEBPAGE }}",
			"{{ CURRENT_INDEX }}",
			"{{ CITY_THUMB }}",
			"{{ FEATURED_IMAGE }}",
			"{{ FEATURED_IMAGE_NOTHUMB }}",
			"{{ NUM_ITEMS }}",
		];

		public static function getSupportedAttributeNames(): array {
			return self::$supportedAttributes;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws Exception
		 */
		public function runProcessor(): string{

			$pageLimit = 10;
			$delimiter = "";
			$htmlTagNameWrapper = "";
			$sortString = "creation asc";

			$limitAttribute = $this->shortcode->getAttribute("limit");
			if ($limitAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Attribute 'limit' is required for the %s attribute.",
						ShortcodeTypes::GET_LIST_OF_CITIES->value,
					)
				);
			}elseif((int) $limitAttribute->value === 0){
				$pageLimit = 9999;
			}else{
				$pageLimit = (int) $limitAttribute->value;
			}

			$sortAttribute = $this->shortcode->getAttribute("sort");
			if ($sortAttribute !== null){
				$sortString = $sortAttribute->value;
			}

			$delimiterAttribute = $this->shortcode->getAttribute("delimiter");
			if ($delimiterAttribute !== null){
				$delimiter = $delimiterAttribute->value;
			}

			$htmlTagNameWrapperAttribute = $this->shortcode->getAttribute("tagNameWrapper");
			if ($htmlTagNameWrapperAttribute !== null){
				$htmlTagNameWrapper = $htmlTagNameWrapperAttribute->value;
			}

			// Get view file path
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			
			// If no shortcode-file is specified, use generateDefaultOutput
			if ($shortcodeViewFileAttribute === null){
				return $this->generateDefaultOutput($delimiter, $htmlTagNameWrapper, $pageLimit, $sortString);
			}

			$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			$viewFileNormalizedPath = realpath($viewFile);

			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			// Load view file contents
			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Query for city pages
			$resultOrder = $this->getResultOrder($sortString);
			$cityPages = Page::query(
				columnQuery: (new ColumnQuery())
					->where("pageType","=",PageType::City->name)
					->and()
					->where("publication_status","=",PublicationStatus::Published->value),
				resultOrder: $resultOrder,
				pager: (new Pager(pageNumber: 1, limit: $pageLimit))
			);

			// Process CityItem template in the view file
			$finalOutput = preg_replace_callback(
				"/{{ begin CityItem }}(.+?){{ end CityItem }}/ism",
				function($matches) use ($cityPages) {
					$template = $matches[1];
					$renderedItems = "";
					$currentIndex = 0;

					foreach ($cityPages as $cityPage) {
						$renderedItems .= $this->renderCityTemplate($cityPage, $template, $currentIndex);
						$currentIndex++;
					}

					return $renderedItems;
				},
				$viewFileContents
			);

			$finalOutput = str_replace("{{ NUM_ITEMS }}", count($cityPages), $finalOutput);

			return $finalOutput;
		}

		private function getResultOrder(string $sortString): ResultOrder {
			return match($sortString) {
				"creation asc" => (new ResultOrder())
					->by("creationTime", "asc")
					->by("publication_timestamp", "asc"),
				"creation desc" => (new ResultOrder())
					->by("creationTime", "desc")
					->by("publication_timestamp", "desc"),
				"alphabetical asc" => (new ResultOrder())
					->by("pageName", "asc"),
				"alphabetical desc" => (new ResultOrder())
					->by("pageName", "desc"),
				default => throw new InvalidAttributeValue("The value of 'sort' was not a recognized string."),
			};
		}

		private function renderCityTemplate(Page $cityPage, string $template, int $currentIndex): string {
			// Fetch the default content section or use the page body as fallback
			$defaultOrFirstContentSection = $cityPage->getDefaultContentSectionOrFirstSection();
			
			// Generate preview text using ContentHelper
			$previewText = ContentHelper::makePreviewFromBody(
				pageBody: $defaultOrFirstContentSection?->content ?? $cityPage->pageBody,
			);
			return str_replace(
				[
					"{{ CITY_NAME }}",
					"{{ CITY_STATE }}",
					"{{ CITY_STATE_SHORTHAND }}",
					"{{ CITY_URI }}",
					"{{ PAGE_URI }}",
					"{{ CITY_PREVIEW }}",
					"{{ PAGE_PREVIEW }}",
					"{{ CITY_WEBPAGE }}",
					"{{ CURRENT_INDEX }}",
					"{{ CITY_THUMB }}",
					"{{ FEATURED_IMAGE }}",
					"{{ FEATURED_IMAGE_NOTHUMB }}"
				],
				[
					$cityPage->getPageDatas(PageDatas::CITY_NAME)[0]->value,
					$cityPage->getPageDatas(PageDatas::STATE_NAME)[0]->value,
					$cityPage->getPageDatas(PageDatas::STATE_NAME_SHORTHAND)[0]->value,
					$cityPage->pageRoute ?? "#",
					$cityPage->pageRoute ?? "#",
					$previewText . "[&hellip;]",
					$previewText . "[&hellip;]",
					$cityPage->getPageDatas(PageDatas::OFFICIAL_CITY_URL)[0]->value,
					$currentIndex,
					$cityPage->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB)[0]->value,
					$cityPage->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB)[0]->value,
					$cityPage->getPageDatas(PageDatas::FEATURED_IMAGE)[0]->value
				],
				$template
			);
		}

		private function generateDefaultOutput(string $delimiter, string $htmlTagNameWrapper, int $pageLimit, string $sortString): string {
			$htmlAttributes = [];
			
			// Build the HTML attributes string
			$htmlAttributesAttribute = $this->shortcode->getAttribute("attributes");
			if ($htmlAttributesAttribute !== null){
				$htmlAttributes = json_decode($htmlAttributesAttribute->value);
				if ($htmlAttributes === null){
					throw new InvalidAttributeValue(sprintf(
						"The value fo the 'attributes' attribute was not valid JSON. The error received was: %s",
						json_last_error_msg(),
					));
				}
			}

			$htmlAttributesString = ""; // Not padded yet
			if ($htmlAttributes !== null){
				foreach($htmlAttributes as $attributeName=>$attributeValue){
					$htmlAttributesString .= sprintf(
						" %s=\"%s\"",
						$attributeName,
						$attributeValue,
					);
				}
			}
			$htmlAttributesString .= " "; // Pad the end

			$cityPages = Page::query(
				columnQuery: (new ColumnQuery())
					->where("pageType","=",PageType::City->name)
					->and()
					->where("publication_status","=",PublicationStatus::Published->value),
				resultOrder: $this->getResultOrder($sortString),
				pager: (new Pager(pageNumber: 1, limit: $pageLimit))
			);

			$finalResult = "";
			foreach ($cityPages as $cityPage) {
				$htmlAttributes = $htmlAttributesString;
				if ($htmlTagNameWrapper === "a"){
					$htmlAttributes .= sprintf(" href=\"%s\"", $cityPage->pageRoute);
				}
				if (empty($htmlTagNameWrapper)) {
					$finalResult .= sprintf(
						"%s, %s%s",
						$cityPage->getPageDatas(PageDatas::CITY_NAME)[0]->value ?? "",
						$cityPage->getPageDatas(PageDatas::STATE_NAME_SHORTHAND)[0]->value ?? "",
						$delimiter,
					);
				} else {
					$finalResult .= sprintf(
						"<%s%s>%s, %s</%s>%s",
						$htmlTagNameWrapper,
						$htmlAttributes,
						$cityPage->getPageDatas(PageDatas::CITY_NAME)[0]->value ?? "",
						$cityPage->getPageDatas(PageDatas::STATE_NAME_SHORTHAND)[0]->value ?? "",
						$htmlTagNameWrapper,
						$delimiter,
					);
				}
			}

			return rtrim($finalResult, $delimiter);
		}
	}
