<?php

	namespace Settings;

	use Google\GoogleAccountSearchConsoleSite;
	use ImageManager\ImageManagerService;
	use Nox\Http\FileUploadPayload;
	use Nox\ORM\ColumnQuery;
	use Uplift\ImageManager\ImageFileAlreadyExists;
	use Uplift\ImageManager\ImageManager;
	use ValueError;

	class SettingService {
		public static function saveSetting(string $name, string $value): Setting{
			$setting = new Setting();
			$setting->name = $name;
			$setting->value = $value;
			$setting->save();

			return $setting;
		}

		/**
		 * Removes a setting from the settings table
		 * @param string $name
		 * @return void
		 */
		public static function removeSetting(string $name): void{
			/** @var Setting|null $existingSetting */
			$existingSetting = Setting::queryOne(
				columnQuery: (new ColumnQuery())
					->where("name", "=", $name)
			);

			$existingSetting?->delete();
		}

		/**
		 * @throws ImageFileAlreadyExists
		 * @throws ValueError
		 */
		public static function saveCompanySettings(
			string $schemaType,
			int $acceloCompanyID,
			string $companyName,
			string $companyStreet,
			string $companyCity,
			string $companyState,
			string $companyPostal,
			?string $phoneNumbersJSON,
			?string $faxNumbersJSON,
			string $additionalAddressesJSON,
			?FileUploadPayload $companyLogo,
			?FileUploadPayload $companyFavicon,
		): void{
			// Verify the JSON strings are valid
			if (json_decode($phoneNumbersJSON) === null){
				throw new ValueError(
					sprintf(
						"Invalid JSON in phone numbers parameter. JSON error: %s",
						json_last_error(),
					),
				);
			}

			if (json_decode($faxNumbersJSON) === null){
				throw new ValueError(
					sprintf(
						"Invalid JSON in phone numbers parameter. JSON error: %s",
						json_last_error(),
					),
				);
			}

			if (json_decode($additionalAddressesJSON) === null){
				throw new ValueError(
					sprintf(
						"Invalid JSON in additional addresses parameter. JSON error: %s",
						json_last_error(),
					),
				);
			}

			// Save the textual settings
			SettingService::saveSetting(
				name: Settings::SCHEMA_TYPE->value,
				value: $schemaType,
			);

			if ($acceloCompanyID > 0) {
				SettingService::saveSetting(
					name: Settings::ACCELO_COMPANY_ID->value,
					value: $acceloCompanyID,
				);
			}else{
				SettingService::removeSetting(Settings::ACCELO_COMPANY_ID->value);
			}

			SettingService::saveSetting(
				name: Settings::COMPANY_NAME->value,
				value: $companyName,
			);

			SettingService::saveSetting(
				name: Settings::COMPANY_STREET->value,
				value: $companyStreet,
			);

			SettingService::saveSetting(
				name: Settings::COMPANY_CITY->value,
				value: $companyCity,
			);

			SettingService::saveSetting(
				name: Settings::COMPANY_STATE->value,
				value: $companyState,
			);

			SettingService::saveSetting(
				name: Settings::COMPANY_POSTAL->value,
				value: $companyPostal,
			);
			
			$phoneNumbers = json_decode($phoneNumbersJSON, true);

			if (!empty($phoneNumbers)) {
				if (!empty($phoneNumbers[0])) {
					SettingService::saveSetting(
						name: Settings::COMPANY_PHONE_NUMBERS->value,
						value: $phoneNumbersJSON,
					);
				}
			}

			$faxNumbers = json_decode($faxNumbersJSON, true);

			if (!empty($faxNumbers)) {
				if (!empty($faxNumbers[0])) {
					SettingService::saveSetting(
						name: Settings::COMPANY_FAX_NUMBERS->value,
						value: $faxNumbersJSON,
					);
				}
			}

			// Handle saving additional addresses as CustomSetting objects
			CustomSettingService::clearAllCustomSettingsOfType(CustomSettings::ADDITIONAL_ADDRESS);
			$additionalAddresses = json_decode($additionalAddressesJSON, true);
			/** @var array{street: string, city: string, state: string, postal: string, country: string} $addressArray */
			foreach($additionalAddresses as $addressArray){
				// Create a new custom setting for this address
				$customAddressSetting = new CustomSetting();
				$customAddressSetting->name = CustomSettings::ADDITIONAL_ADDRESS->name;
				$customAddressSetting->value = json_encode($addressArray);
				$customAddressSetting->save();
			}

			// Handle saving image URIs for the logo and or the favicon
			if ($companyLogo !== null){
				$fileExtension = pathinfo(path: $companyLogo->fileName, flags:PATHINFO_EXTENSION);
				$fileNameToSave = sprintf("%s.%s", "companyLogo", $fileExtension);
				$pathToUpload = sprintf("%s/%s",
					ImageManager::IMAGES_DIRECTORY,
					$fileNameToSave,
				);

				ImageManagerService::uploadImage(
					filePath: $pathToUpload,
					contents: $companyLogo->contents,
					overrideExistingFile: true,
				);

				$savedURI = sprintf("%s/%s",
					ImageManager::PUBLIC_IMAGES_URI_BASE,
					$fileNameToSave
				);

				self::saveSetting(
					name: Settings::COMPANY_LOGO->value,
					value:$savedURI,
				);
			}

			if ($companyFavicon !== null){
				$fileExtension = pathinfo(path: $companyFavicon->fileName, flags:PATHINFO_EXTENSION);
				$fileNameToSave = sprintf("%s.%s", "companyFavicon", $fileExtension);
				$pathToUpload = sprintf("%s/%s",
					ImageManager::IMAGES_DIRECTORY,
					$fileNameToSave
				);

				ImageManagerService::uploadImage(
					filePath: $pathToUpload,
					contents: $companyFavicon->contents,
					overrideExistingFile: true,
				);

				$savedURI = sprintf("%s/%s",
					ImageManager::PUBLIC_IMAGES_URI_BASE,
					$fileNameToSave
				);

				self::saveSetting(
					name: Settings::COMPANY_FAVICON->value,
					value:$savedURI,
				);
			}
		}

		public static function saveSystemSettings(
			int $forceHTTPs,
			int $enableEntireSiteNoIndex,
			int $siteSuspended,
			int $isIPPFormDisabled,
			string $thumbnailWidth,
			string $thumbnailHeight,
		): void{

			$thumbnailWidth = (int) $thumbnailWidth;
			$thumbnailHeight = (int) $thumbnailHeight;

			SettingService::saveSetting(
				name: Settings::FORCE_HTTPS->value,
				value: $forceHTTPs,
			);

			SettingService::saveSetting(
				name: Settings::ENTIRE_SITE_NO_INDEX->value,
				value: $enableEntireSiteNoIndex,
			);

			SettingService::saveSetting(
				name: Settings::SITE_SUSPENDED->value,
				value: $siteSuspended,
			);

			SettingService::saveSetting(
				name: Settings::IPP_FORM_DISABLED->value,
				value: $isIPPFormDisabled,
			);

			SettingService::saveSetting(
				name: Settings::THUMBNAIL_UPLOAD_SIZE_X->value,
				value: $thumbnailWidth,
			);

			SettingService::saveSetting(
				name: Settings::THUMBNAIL_UPLOAD_SIZE_Y->value,
				value: $thumbnailHeight,
			);
		}

		public static function saveContentSettings(
			string $noReviewsMessage,
		): void{

			SettingService::saveSetting(
				name: Settings::NO_REVIEWS_MESSAGE->value,
				value: $noReviewsMessage,
			);
		}

		public static function saveRobotsContent(
			string $content,
		): void{

			SettingService::saveSetting(
				name: Settings::ROBOTS_FILE_CONTENT->value,
				value: $content,
			);
		}

		/**
		 * Saves the settings for opting out of specific front-end injections
		 */
		public static function saveInjectionSettings(
			bool $optOutLightbox,
		): void{

			SettingService::saveSetting(
				name: Settings::OPT_OUT_LIGHTBOX_INJECTION->value,
				value: $optOutLightbox ? "1" : "0",
			);
		}

		public static function saveSearchConsoleSite(
			int $searchConsoleSiteID,
		): void{

			/** @var ?GoogleAccountSearchConsoleSite $searchConsoleSite */
			$searchConsoleSite = GoogleAccountSearchConsoleSite::fetch($searchConsoleSiteID);

			SettingService::saveSetting(
				name: Settings::ACTIVE_SEARCH_CONSOLE_SITE->value,
				value: $searchConsoleSite->site,
			);
		}
	}