<?php

	namespace Settings;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use GuzzleHttp\Exception\GuzzleException;
	use Installation\InstallationService;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Response;
	use Nox\Http\Request;
	use Nox\ORM\ColumnQuery;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use System\HttpHelper;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\ImageManager\ImageFileAlreadyExists;
	use Uplift\ImageManager\ImageManager;
	use ValueError;

	#[Controller]
	#[RouteBase("/uplift/settings")]
	class SettingController extends BaseController{

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function mainView(): string
		{
			$settings = Setting::query();

			return Renderer::renderView(
				viewFileName: "settings/main.php",
				viewScope:[
					"settings"=>$settings,
					Settings::COMPANY_PHONE_NUMBERS->name=>Setting::getSettingValue(Settings::COMPANY_PHONE_NUMBERS->value),
					Settings::COMPANY_FAX_NUMBERS->name=>Setting::getSettingValue(Settings::COMPANY_FAX_NUMBERS->value),
					Settings::THUMBNAIL_UPLOAD_SIZE_X->name=>Setting::getSettingValue(Settings::THUMBNAIL_UPLOAD_SIZE_X->value) ?? ImageManager::DEFAULT_THUMB_WIDTH,
					Settings::THUMBNAIL_UPLOAD_SIZE_Y->name=>Setting::getSettingValue(Settings::THUMBNAIL_UPLOAD_SIZE_Y->value) ?? ImageManager::DEFAULT_THUMB_HEIGHT,
					Settings::SCHEMA_TYPE->name=>Setting::getSettingValue(Settings::SCHEMA_TYPE->value),
					Settings::COMPANY_NAME->name=>Setting::getSettingValue(Settings::COMPANY_NAME->value),
					Settings::COMPANY_STREET->name=>Setting::getSettingValue(Settings::COMPANY_STREET->value),
					Settings::COMPANY_CITY->name=>Setting::getSettingValue(Settings::COMPANY_CITY->value),
					Settings::COMPANY_STATE->name=>Setting::getSettingValue(Settings::COMPANY_STATE->value),
					Settings::COMPANY_POSTAL->name=>Setting::getSettingValue(Settings::COMPANY_POSTAL->value),
					Settings::COMPANY_FAVICON->name=>Setting::getSettingValue(Settings::COMPANY_FAVICON->value),
					Settings::COMPANY_LOGO->name=>Setting::getSettingValue(Settings::COMPANY_LOGO->value),
					Settings::FORCE_HTTPS->name=>Setting::getSettingValue(Settings::FORCE_HTTPS->value),
					Settings::ENTIRE_SITE_NO_INDEX->name=>Setting::getSettingValue(Settings::ENTIRE_SITE_NO_INDEX->value),
					Settings::SITE_SUSPENDED->name=>Setting::getSettingValue(Settings::SITE_SUSPENDED->value),
					Settings::NO_REVIEWS_MESSAGE->name=>Setting::getSettingValue(Settings::NO_REVIEWS_MESSAGE->value),
					Settings::ROBOTS_FILE_CONTENT->name=>Setting::getSettingValue(Settings::ROBOTS_FILE_CONTENT->value),
					Settings::ACCELO_COMPANY_ID->name=>Setting::getSettingValue(Settings::ACCELO_COMPANY_ID->value),
					Settings::IPP_FORM_DISABLED->name=>Setting::getSettingValue(Settings::IPP_FORM_DISABLED->value),
					Settings::OPT_OUT_LIGHTBOX_INJECTION->name=>Setting::getSettingValue(Settings::OPT_OUT_LIGHTBOX_INJECTION->value),
				]
			);

		}

		#[Route("PATCH", "/company-information")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function saveCompanyInformation(Request $request): JSONResult
		{
			$account = Account::getCurrentUser();
			$payload = $request->getPayload();

			try {
				$schemaType = $payload->getTextPayload("schemaType");
				$acceloCompanyID = $payload->getTextPayload("company-accelo-id");
				$companyName = $payload->getTextPayload("companyName");
				$companyStreet = $payload->getTextPayload("companyStreet");
				$companyCity = $payload->getTextPayload("companyCity");
				$companyState = $payload->getTextPayload("companyState");
				$companyPostal = $payload->getTextPayload("companyPostal");
				$phoneNumbersJSON = $payload->getTextPayload("phone-numbers");
				$faxNumbersJSON = $payload->getTextPayload("fax-numbers");
				$additionalAddressesJSON = $payload->getTextPayload("additional-addresses-json");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$companyLogo = $payload->getFileUploadPayload('companyLogo');
			}catch(NoPayloadFound){
				$companyLogo = null;
			}

			try {
				$companyFavicon = $payload->getFileUploadPayload('companyFavicon');
			}catch(NoPayloadFound){
				$companyFavicon = null;
			}

			try {
				SettingService::saveCompanySettings(
					schemaType: $schemaType->contents,
					acceloCompanyID: (int) $acceloCompanyID->contents,
					companyName: $companyName->contents,
					companyStreet: $companyStreet->contents,
					companyCity: $companyCity->contents,
					companyState: $companyState->contents,
					companyPostal: $companyPostal->contents,
					phoneNumbersJSON: $phoneNumbersJSON->contents,
					faxNumbersJSON: $faxNumbersJSON->contents,
					additionalAddressesJSON: $additionalAddressesJSON->contents,
					companyLogo: $companyLogo,
					companyFavicon: $companyFavicon,
				);
			} catch (ImageFileAlreadyExists|ValueError $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::UPDATE_COMPANY_INFORMATION->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/system")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function saveUpliftSettings(Request $request): JSONResult
		{
			$account = Account::getCurrentUser();
			$payload = $request->getPayload();

			try {
				$payload->getTextPayload("forceHTTPs");
				$forceHTTPs = 1;
			}catch(NoPayloadFound){
				$forceHTTPs = 0;
			}

			try {
				$payload->getTextPayload("noindex");
				$noindex = 1;
			}catch(NoPayloadFound){
				$noindex = 0;
			}

			try {
				$payload->getTextPayload("siteSuspended");
				$siteSuspended = 1;
			}catch(NoPayloadFound){
				$siteSuspended = 0;
			}

			try {
				$payload->getTextPayload("ippFormDisabled");
				$ippFormDisabled = 1;
			}catch(NoPayloadFound){
				$ippFormDisabled = 0;
			}

			try {
				$thumbnailUploadSizeX = $payload->getTextPayload("thumbnailUploadSizeX");
				$thumbnailUploadSizeY = $payload->getTextPayload("thumbnailUploadSizeY");
			} catch (NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			SettingService::saveSystemSettings(
				forceHTTPs:$forceHTTPs,
				enableEntireSiteNoIndex: $noindex,
				siteSuspended: $siteSuspended,
				isIPPFormDisabled: $ippFormDisabled,
				thumbnailWidth: $thumbnailUploadSizeX->contents,
				thumbnailHeight:$thumbnailUploadSizeY->contents,
			);

			ActivityLog::log(
				categoryID: ActivityLogCategories::UPDATE_UPLIFT_SETTINGS->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/content")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function saveContentSettings(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();


			try{
				$noReviewsMessage = $payload->getTextPayload("noReviewsMessage");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			SettingService::saveContentSettings(
				noReviewsMessage: $noReviewsMessage->contents,
			);

			ActivityLog::log(
				categoryID: ActivityLogCategories::UPDATE_CONTENT_SETTINGS->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/robots")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function editRobotSettings(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try{
				$robotsContent = $payload->getTextPayload("robots");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			SettingService::saveRobotsContent(
				content: $robotsContent->contents,
			);

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_ROBOT_SETTINGS->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/injections")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function editInjectionSettings(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			$optOutLightbox = $payload->getTextPayloadNullable("opt-out-lightbox") !== null;

			SettingService::saveInjectionSettings(
				optOutLightbox: $optOutLightbox,
			);

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_OPT_OUT_INJECTABLES_SETTINGS->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"optOutLightbox" => (string)$optOutLightbox
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/search-console/site")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		// #[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function editSearchConsoleActiveSite(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try{
				$newSiteID = $payload->getTextPayload("search-console-site");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			SettingService::saveSearchConsoleSite(
				searchConsoleSiteID: $newSiteID->contents,
			);

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_SEARCH_CONSOLE_ACTIVE_SITE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"newSiteID"=>$newSiteID,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("GET", "/re-register-build")]
		#[RequireLogin]
		#[UseJSON]
		// #[RequirePermission(PermissionCategories::MANAGE_SETTINGS)]
		public function reregisterBuild(Request $request): JSONResult
		{

			$account = Account::getCurrentUser();

			try {
				$registrationResult = InstallationService::registerWithControlPanel(
					acceloCompanyID: Setting::getSettingValue(Settings::ACCELO_COMPANY_ID->value) ?? 0,
					companyName: Setting::getSettingValue(Settings::COMPANY_NAME->value) ?? 0,
					isTestBuild: Setting::getSettingValue(Settings::IS_TEST_BUILD->value) ?? 0,
					indexingDisabled: Setting::getSettingValue(Settings::ENTIRE_SITE_NO_INDEX->value) ?? 0,
					isSuspended: Setting::getSettingValue(Settings::SITE_SUSPENDED->value) ?? 0,
				);
			} catch (GuzzleException $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			} catch (MalformedValue $e) {
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			// Set the values
			SettingService::saveSetting(Settings::BUILD_UUID->value, $registrationResult->uuid);
			SettingService::saveSetting(Settings::UPLIFT_CONTROL_PANEL_API_KEY->value, $registrationResult->apiKey);

			ActivityLog::log(
				categoryID: ActivityLogCategories::RE_REGISTER_BUILD->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess([
				"uuid"=>$registrationResult->uuid,
				"apiKey"=>$registrationResult->apiKey,
			]);
		}

		#[Route("PATCH", "/change-theme")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		public function changeTheme(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try{
				$theme = $payload->getTextPayload("theme");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			SettingService::saveSetting(Settings::THEME_NAME->value, $theme->contents);

			ActivityLog::log(
				categoryID: ActivityLogCategories::CHANGE_THEME->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}
	}