<?php

	namespace SenzuImporter;

	use Accounts\Account;
	use Accounts\Attributes\RequirePermission;
	use Accounts\Attributes\RequireLogin;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use APIPublic\ControlPanelAPIException;
	use GuzzleHttp\Exception\GuzzleException;
	use Nox\Http\ArrayPayload;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use PageEditor\Exceptions\PageLayoutDoesntExist;
	use PageEditor\Exceptions\PageLayoutIsBlank;
	use PageEditor\Exceptions\PageNameExists;
	use PageEditor\Exceptions\PageNameExistsWithSamePageType;
	use PageEditor\Exceptions\PageNameIsBlank;
	use PageEditor\Exceptions\PageRouteInUse;
	use PageEditor\Exceptions\PageRouteIsBlank;
	use PageEditor\Exceptions\PublicationStatusDoesntExist;
	use Roles\PermissionCategories;
	use System\HttpHelper;
	use Uplift\Exceptions\IncompatiblePageType;
	use Uplift\Exceptions\MalformedValue;

	#[Controller]
	#[RouteBase("/uplift/senzu-importer")]
	class SenzuImporterController extends BaseController
	{

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::SENZU_IMPORTER)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "senzu-importer/main.php"
			);
		}


		#[Route("GET", "/get-content")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::SENZU_IMPORTER)]
		#[UseJSON]
		public function fetchSenzuContent(Request $request): JSONResult
		{
			$siteID = $request->getQueryValue("site-id");

			if ($siteID === null){
				http_response_code(400);
				return new JSONError("Missing site-id from GET query.");
			}

			$siteID = (int) $siteID;
			if ($siteID === 0){
				http_response_code(400);
				return new JSONError("site-id cannot be blank.");
			}

			try {
				$contentData = SenzuImporterService::getSenzuContent($siteID);
			} catch (ControlPanelAPIException|GuzzleException|MalformedValue $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"contentData"=>$contentData,
			]);
		}

		#[Route("POST", "/import-content")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::SENZU_IMPORTER)]
		#[UseJSON]
		#[ProcessRequestBody]
		public function importParsedSenzuContent(Request $request): JSONResult
		{
			$requestPayload = $request->getPayload();

			try {
				$senzuSiteID = $requestPayload->getTextPayload("senzuSiteID");
				$pagesToBuild = $requestPayload->getArrayPayload("pages");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			/** @var array{breadcrumbs:array, pageName: string, pageType: string, title: string, description: string, pageBody: string, pageRoute: string, pageDatas: array, featuredImageURI: string, featuredImageThumbURI: string} $pageArray */
			foreach($pagesToBuild->contents as $pageArray) {
				try {
					SenzuImporterService::buildPageWithSenzuContent(
						pageName: $pageArray['pageName'],
						pageType: $pageArray['pageType'],
						pageBody: $pageArray['pageBody'],
						pageRoute: $pageArray['pageRoute'],
						metaTitle: $pageArray['title'],
						metaDescription: $pageArray['description'],
						featuredImageURI: $pageArray['featuredImageURI'] ?? "",
						featuredImageThumbURI: $pageArray['featuredImageThumbURI'] ?? "",
						breadcrumbs: $pageArray['breadcrumbs'],
						data: $pageArray['pageDatas'],
					);

				} catch (PageLayoutDoesntExist|PageNameIsBlank|PageRouteInUse|PageRouteIsBlank|PublicationStatusDoesntExist|IncompatiblePageType|PageLayoutIsBlank|PageNameExists|PageNameExistsWithSamePageType|NoPrimaryKey $e) {
					http_response_code(500);
					return new JSONError($e->getMessage());
				}
			}

			SenzuImporterService::changeSenzuSiteStatus((int) $senzuSiteID->contents, 5);

			ActivityLog::log(
				categoryID: ActivityLogCategories::IMPORTED_SENZU_SITE->value,
				accountID: (Account::getCurrentUser())->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"senzuSiteID"=>$senzuSiteID->contents,
				]),
			);

			return new JSONSuccess([]);
		}
	}