<?php

	namespace Roles;

	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Roles\Exceptions\RoleCannotBeDeleted;
	use ValueError;

	class RolesService
	{
		/**
		 * @param string $name
		 * @return Role
		 * @throws ValueError
		 */
		public static function createRole(
			string $name,
		): Role{
			if (empty($name)){
				throw new ValueError("A role name must be provided.");
			}

			$newRole = new Role();
			$newRole->name = trim($name);
			$newRole->canBeDeleted = 1;
			$newRole->save();

			return $newRole;
		}

		/**
		 * @param Role $role
		 * @param string|null $roleName
		 * @param int|null $canBeDeleted
		 * @return Role
		 */
		public static function editRole(
			Role $role,
			string|null $roleName = null,
			int|null $canBeDeleted = null
		): Role{
			if($roleName !== null) {
				$role->name = trim($roleName);
			}
			if($canBeDeleted !== null) {
				$role->canBeDeleted = $canBeDeleted;
			}

			$role->save();

			return $role;
		}

		/**
		 * @param Role $role
		 * @return void
		 * @throws NoPrimaryKey
		 * @throws RoleCannotBeDeleted
		 */
		public static function deleteRole(
			Role $role
		): void{
			if($role->canBeDeleted === 0) {
				throw new RoleCannotBeDeleted("This role cannot be deleted.");
			}

			$role->delete();
		}

		/**
		 * Returns a role that has the provided name, or null if it doesn't exist.
		 * @param string $name
		 * @return Role|null
		 */
		public static function getRoleByName(
			string $name,
		): ?Role{
			/** @var ?Role $role */
			$role = Role::queryOne(
				columnQuery: (new ColumnQuery())
					->where("name", "=", $name)
			);

			return $role;
		}

		/**
		 * Creates a role with permissions fit for general client use.
		 */
		public static function createClientRole(): Role{
			$clientRole = new Role();
			$clientRole->name = "Client";
			$clientRole->save();

			PermissionsService::createPermission(
				role: $clientRole,
				category: PermissionCategories::ACCESS_CLIENT_IPP_FORM,
				isEnabled: true,
			);

			return $clientRole;
		}
	}