<?php

	namespace Roles;

	use Accounts\Account;
	use Accounts\AccountsService;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\Http\Rewrite;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\Exceptions\RoleCannotBeDeleted;
	use System\HttpHelper;
	use ValueError;

	#[Controller]
	#[RouteBase("/uplift")]
	class RolesController extends BaseController
	{
		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/manage-roles")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_ROLES)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "manage-roles/main.php",
				viewScope:[
					"roles" => Role::query(),
				],
			);
		}

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "@/manage-roles/(?<roleID>\d+)$@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_ROLES)]
		public function manageRoleView(Request $request): string | Rewrite
		{
			$roleID = (int) $request->getParameter("roleID");

			/** @var Role $role */
			$role = Role::fetch($roleID);

			if ($role === null){
				return new Rewrite(
					path: "/uplift/404",
					statusCode:404,
				);
			}

			return Renderer::renderView(
				viewFileName: "manage-roles/manage-role.php",
				viewScope: [
					"role"=>$role,
					"permissions"=> Permission::query(
						columnQuery: (new ColumnQuery())
							->where("role_id", "=", $roleID),
					),
				],
			);
		}

		#[Route("GET", "/roles")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_ROLES)]
		public function getRoles(): JSONResult
		{
			return new JSONSuccess([
				"roles" => Role::query(),
			]);
		}

		#[Route("PUT", "/roles")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_ROLES)]
		public function createRole(Request $request): JSONResult
		{
			$account = Account::getCurrentUser();
			$payload = $request->getPayload();

			try {
				$roleName = $payload->getTextPayload('new-role-name');
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			$hasAllPermissions = $payload->getTextPayloadNullable('has-all-perms');

			try {
				// Create the role
				$newRole = RolesService::createRole(
					name: $roleName->contents,
				);

				// Create permissions for the role
				foreach(PermissionCategories::cases() as $category) {
					try {
						PermissionsService::createPermission(
							role: $newRole,
							category: $category
						);
					} catch (PermissionAlreadyExists) {
					}
				}

				// Check if this role has all permissions
				if ($hasAllPermissions !== null){
					$newRole->hasAllPermissions = 1;
					$newRole->save();
				}

			} catch (ValueError $e ) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CREATE_ROLE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"roleID"=>$newRole->id,
					"roleName"=>$newRole->name,
					"hasAllPermissions"=>$newRole->hasAllPermissions,
				]),
			);

			return new JSONSuccess([
				"newRole"=>$newRole,
			]);
		}

		#[Route("PATCH", "@/roles/(?<roleID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_ROLES)]
		public function editRole(Request $request): JSONResult
		{
			$account = Account::getCurrentUser();
			$payload = $request->getPayload();
			$roleID = (int) $request->getParameter('roleID');

			try {
				$roleName = $payload->getTextPayload("role-name");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			/** @var Role $role */
			$role = Role::fetch($roleID);

			RolesService::editRole(
				role: $role,
				roleName: $roleName->contents
			);

			// Check if this role has all permissions
			$hasAllPermsCheckbox = $payload->getTextPayloadNullable("has-all-perms");
			if ($hasAllPermsCheckbox !== null){
				$role->hasAllPermissions = 1;
				$role->save();
			}else {
				// Doesn't have all permissions
				$role->hasAllPermissions = 0;
				$role->save();

				// Check them individually
				$permissionsChecked = [];

				foreach (PermissionCategories::cases() as $case) {
					$isEnabled = $payload->getTextPayloadNullable($case->value) !== null;
					$permissionsChecked[$case->name] = $isEnabled;

					// Be sure they all exist by trying to create the permissions first
					// When a new permission is added when roles already exist, then there is a lack of Permission rows
					// for that newly created Permission
					try {
						PermissionsService::createPermission(
							role: $role,
							category: $case,
							isEnabled: $isEnabled,
						);
					} catch (PermissionAlreadyExists) {
					}
				}

				/** @var Permission[] $permissions */
				$permissions = Permission::query(
					columnQuery: (new ColumnQuery())
						->where("role_id", "=", $roleID),
				);

				foreach ($permissions as $permission) {
					$permissionCategoryFromPermissionID = PermissionCategories::getByValue($permission->permissionID);
					if (isset($permissionsChecked[$permissionCategoryFromPermissionID->name])) {
						PermissionsService::editPermission(
							permission: $permission,
							isEnabled: $permissionsChecked[$permissionCategoryFromPermissionID->name] === true ? 1 : 0,
						);
					}
				}
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::UPDATE_ROLE_PERMISSIONS->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"roleID"=>$role->id,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("DELETE", "@/roles/(?<roleID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_ROLES)]
		public function deleteRole(Request $request):JSONResult {
			$account = Account::getCurrentUser();
			$roleID = (int) $request->getParameter('roleID');

			/** @var ?Role $role */
			$role = Role::fetch($roleID);

			$permissions = Permission::query(
				columnQuery: (new ColumnQuery())
					->where("role_id", "=", $role->id),
			);

			try {
				// Deletes the selected role
				RolesService::deleteRole(
					role: $role
				);

				// Deletes the permissions associated with the role
				foreach($permissions as $permission) {
					PermissionsService::deletePermission(
						permission: $permission,
						canBeDeleted: $role->canBeDeleted
					);
				}

				// Changes the role of accounts with the roleID that is being deleted
				AccountsService::changeRole(
					role: $role
				);
			}catch(NoPrimaryKey|RoleCannotBeDeleted $e){
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_ROLE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}
	}