<?php

	namespace Reviews;

	require_once __DIR__ . "/Exceptions/MissingFirstName.php";
	require_once __DIR__ . "/Review.php";

	use Nox\ORM\Abyss;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\MissingDatabaseCredentials;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;

	class ReviewsService{

		/**
		 * Queries reviews by using the provided arguments as criteria.
		 * @param int $page
		 * @param int $limit
		 * @param string $query
		 * @return array
		 */
		public static function getReviewsByCriteria(
			int $page = 0,
			int $limit = 0,
			string $query = "",
		): array
		{
			$columnQuery = new ColumnQuery();
			if (!empty($query)){
				$columnQuery->where("body","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("first_name","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("last_name","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("city","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("state","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("CONCAT(first_name,' ',last_name)", "LIKE", sprintf("%%%s%%", $query));
			}

			return Review::query(
				columnQuery: $columnQuery,
				resultOrder: (new ResultOrder())
					->by("posted_timestamp", "DESC"),
				pager: (new \Nox\ORM\Pager(
					pageNumber: $page,
					limit:$limit,
				)),
			);
		}

		/**
		 * Tells how many rows there could be from a given query, but without the page/limit limitations
		 * @param string $query
		 * @return int
		 */
		public static function countTotalReviewsWithCriteria(
			string $query = "",
		): int
		{
			$columnQuery = new ColumnQuery();
			if (!empty($query)){
				$columnQuery->where("body","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("first_name","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("last_name","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("city","LIKE",sprintf("%%%s%%", $query))
					->or()
					->where("state","LIKE",sprintf("%%%s%%", $query));
			}

			return Review::count(
				columnQuery: $columnQuery,
			);
		}

		/**
		 * Create a new review object from the provided field arguments.
		 * @throws MissingFirstName
		 */
		public static function createReview(
			string $firstName,
			string $lastName,
			string $city,
			string $state,
			int $rating,
			string $body,
		): Review
		{

			if (empty($firstName)){
				throw new MissingFirstName;
			}

			$newReview = new Review();
			$newReview->firstName = trim($firstName);
			$newReview->lastName = trim($lastName);
			$newReview->city = trim($city);
			$newReview->state = trim($state);
			$newReview->rating = $rating;
			$newReview->body = $body;
			$newReview->save();

			return $newReview;
		}

		/**
		 * Saves a review objects fields with the newly provided field arguments.
		 * @throws MissingFirstName
		 */
		public static function saveReview(
			Review $review,
			string $firstName,
			string $lastName,
			string $city,
			string $state,
			int $rating,
			string $body,
		): void
		{
			if (empty($firstName)){
				throw new MissingFirstName();
			}

			$review->firstName = trim($firstName);
			$review->lastName = trim($lastName);
			$review->city = trim($city);
			$review->state = trim($state);
			$review->rating = $rating;
			$review->body = $body;
			$review->save();
		}

		/**
		 * Deletes a review object from the database.
		 * @param Review $review
		 * @return void
		 * @throws NoPrimaryKey
		 */
		public static function deleteReview(Review $review): void
		{
			$review->delete();
		}

		/**
		 * Fetches an aggregate overview data set about the reviews in this CMS.
		 * @return void
		 * @throws MissingDatabaseCredentials
		 */
		public static function getReviewsOverview(): ReviewsOverviewDto{
			// Query all reviews
			$numberOfReviews = Review::count();

			// Get the number of reviews in the last 30 days
			$numberOfReviewsInLast30Days = Review::count(
				columnQuery: (new ColumnQuery())
					->where("posted_timestamp", ">=", time() - 86400 * 30)
			);

			// Get the earliest review in the system
			/** @var ?Review $earliestReview */
			$earliestReview = Review::queryOne(
				resultOrder: (new ResultOrder())
					->by("posted_timestamp", "ASC"),
				pager: (new Pager(1, 1))
			);

			// Run a raw query to get the average of the reviews' ratings
			$connection = (new Abyss())->getConnectionToDatabase(\NoxEnv::MYSQL_DB_NAME);
			$dummyReviewsModel = new ReviewsModel();
			$fullyQualifiedTableName = $dummyReviewsModel->getDatabaseName() . "." . $dummyReviewsModel->getName();
			$result = $connection->query("SELECT AVG(`rating`) as `reviewsAverage` FROM $fullyQualifiedTableName");
			$row = $result->fetch_assoc();
			if (is_array($row)){
				$reviewsAverage = $row["reviewsAverage"] ?? null;
			}else{
				$reviewsAverage = null;
			}

			$reviewsOverviewDto = new ReviewsOverviewDto();
			$reviewsOverviewDto->totalReviews = $numberOfReviews;
			$reviewsOverviewDto->firstReviewUnixTimestamp = $earliestReview?->postedTimestamp ?? null;
			$reviewsOverviewDto->newReviewsAddedInLast30Days = $numberOfReviewsInLast30Days;
			$reviewsOverviewDto->averageRating = $reviewsAverage ?? 0;

			return $reviewsOverviewDto;
		}
	}