<?php

	namespace Reviews;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\ORM\Exceptions\MissingDatabaseCredentials;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;

	#[Controller]
	#[RouteBase("/uplift")]
	class ReviewsController extends BaseController{

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/manage-reviews")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_REVIEWS)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "manage-reviews/main.php",
			);
		}

		#[Route("GET", "/reviews")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_REVIEWS)]
		public function getReviews(Request $request): JSONResult
		{
			$limit = (int) $request->getQueryValue("limit");
			$page = (int) $request->getQueryValue("page");
			$query = $request->getQueryValue("query") ?? "";

			$totalReviews = ReviewsService::countTotalReviewsWithCriteria($query);
			$totalPages = ceil($totalReviews / $limit);

			return new JSONSuccess([
				"reviews"=>ReviewsService::getReviewsByCriteria(
					page:$page,
					limit:$limit,
					query:$query,
				),
				"totalPages"=>$totalPages,
				"totalReviews"=>$totalReviews,
			]);
		}

		/**
		 * This API provides the following data about reviews in array format
		 * {totalReviews: int, firstReviewUnixTimestamp: int, newReviewsAddedInLast30Days: int, averageRating: float}
		 * @param Request $request
		 * @return JSONResult
		 * @throws MissingDatabaseCredentials
		 */
		#[Route("GET", "/reviews/overview")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_REVIEWS)]
		public function getReviewsOverviewData(Request $request): JSONResult
		{
			$dto = ReviewsService::getReviewsOverview();
			return new JSONSuccess((array)$dto);
		}

		#[Route("DELETE", "@/review/(?<reviewID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_REVIEWS)]
		public function deleteReview(Request $request): JSONResult
		{
			$reviewID = (int) $request->getParameter("reviewID");
			$account = Account::getCurrentUser();

			/** @var Review|null $review */
			$review = Review::fetch($reviewID);

			if ($review === null){
				return new JSONError(sprintf("No review found for ID %s", $reviewID));
			}

			ReviewsService::deleteReview($review);

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_REVIEW->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"reviewID"=>$reviewID,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PUT", "/reviews")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_REVIEWS)]
		public function addNewReview(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$firstName = $payload->getTextPayload("first-name");
				$lastName = $payload->getTextPayload("last-name");
				$city = $payload->getTextPayload("city");
				$state = $payload->getTextPayload("state");
				$rating = $payload->getTextPayload("rating");
				$body = $payload->getTextPayload("body");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newReview = ReviewsService::createReview(
					firstName: $firstName->contents,
					lastName: $lastName->contents,
					city: $city->contents,
					state: $state->contents,
					rating: (int) $rating->contents,
					body: $body->contents,
				);
			}catch(MissingFirstName){
				return new JSONError("You must provide a first name for the review.");
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::POST_REVIEW->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"reviewID"=>$newReview->id,
				]),
			);

			return new JSONSuccess([
				"newReview"=>$newReview,
			]);
		}

		#[Route("PATCH", "@/review/(?<reviewID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_REVIEWS)]
		public function editReview(Request $request): JSONResult
		{
			$reviewID = (int) $request->getParameter("reviewID");
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			/** @var null | Review $review */
			$review = Review::fetch($reviewID);

			if ($review === null){
				return new JSONError(sprintf("No review found with ID %d", $reviewID));
			}

			try {
				$firstName = $payload->getTextPayload("first-name");
				$lastName = $payload->getTextPayload("last-name");
				$city = $payload->getTextPayload("city");
				$state = $payload->getTextPayload("state");
				$rating = $payload->getTextPayload("rating");
				$body = $payload->getTextPayload("body");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				ReviewsService::saveReview(
					review: $review,
					firstName: $firstName->contents,
					lastName: $lastName->contents,
					city: $city->contents,
					state: $state->contents,
					rating: (int) $rating->contents,
					body: $body->contents,
				);
			}catch(MissingFirstName){
				return new JSONError("You must provide a first name for the review.");
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::UPDATE_REVIEW->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"reviewID"=>$reviewID,
				]),
			);

			return new JSONSuccess();
		}
	}