<?php

	namespace Redirects;

	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;

	require_once __DIR__ . "/Redirect.php";
	require_once __DIR__ . "/Exceptions/DuplicateFromValue.php";

	class RedirectsService{

		/**
		 * @param int $page
		 * @param int $limit
		 * @param string $query
		 * @return Redirect[]
		 */
		public static function getRedirects(
			int $page,
			int $limit,
			string $query,
		): array{
			$columnQuery = new ColumnQuery();
			if (!empty($query)){
				$columnQuery->where("from_uri", "LIKE", sprintf("%%%s%%", $query))
					->or()
					->where("to", "LIKE", sprintf("%%%s%%", $query));
			}

			return Redirect::query(
				columnQuery: $columnQuery,
				pager: new Pager(pageNumber: $page, limit: $limit),
			);
		}

		/**
		 * @param string $query
		 * @return int
		 */
		public static function getTotalRedirectsForQuery(
			string $query,
		): int{
			$columnQuery = new ColumnQuery();
			if (!empty($query)){
				$columnQuery->where("from_uri", "LIKE", sprintf("%%%s%%", $query))
					->or()
					->where("to", "LIKE", sprintf("%%%s%%", $query));
			}

			return Redirect::count(
				columnQuery: $columnQuery,
			);
		}

		/**
		 * @throws DuplicateFromValue
		 */
		public static function addRedirect(
			string $from,
			string $to,
			int $isRegex,
			int $status,
			bool $preserveQueryString
		): Redirect{

			// Trim whitespace from the "from" value
			$from = trim($from);

			// Check fields
			if (empty($from)){
				throw new \ValueError("'From' cannot be empty.");
			}

			if (empty($to)){
				throw new \ValueError("'To' cannot be empty.");
			}

			if (empty($status)){
				throw new \ValueError("'HTTP Response Status Code' cannot be empty or 0.");
			}

			// Check if a redirect with the same from value already exists
			/** @var Redirect | null $existingRedirect */
			$existingRedirect = Redirect::queryOne(
				columnQuery: (new ColumnQuery())
					->where("from_uri", "=", $from),
			);

			if ($existingRedirect !== null){
				throw new DuplicateFromValue();
			}

			$redirect = new Redirect();
			$redirect->fromURI = $from;
			$redirect->to = $to;
			$redirect->statusCode = $status;
			$redirect->isRegEx = $isRegex;
			$redirect->preserveQueryString = $preserveQueryString ? 1 : 0;
			$redirect->save();

			return $redirect;
		}

		/**
		 * @throws DuplicateFromValue
		 */
		public static function updateRedirect(
			Redirect $redirect,
			string $from,
			string $to,
			int $isRegex,
			int $status,
			bool $preserveQueryString
		): void{

			// Trim whitespace from the "from" value
			$from = trim($from);

			// Check fields
			if (empty($from)){
				throw new \ValueError("'From' cannot be empty.");
			}

			if (empty($to)){
				throw new \ValueError("'To' cannot be empty.");
			}

			if (empty($status)){
				throw new \ValueError("'HTTP Response Status Code' cannot be empty or 0.");
			}

			// Check if a redirect with the same from value already exists that isn't this one
			/** @var Redirect | null $existingRedirect */
			$existingRedirect = Redirect::queryOne(
				columnQuery: (new ColumnQuery())
					->where("from_uri", "=", $from)
					->and()
					->where("id","!=",$redirect->id),
			);

			if ($existingRedirect !== null){
				throw new DuplicateFromValue();
			}

			$redirect->fromURI = $from;
			$redirect->to = $to;
			$redirect->statusCode = $status;
			$redirect->isRegEx = $isRegex;
			$redirect->preserveQueryString = $preserveQueryString ? 1 : 0;
			$redirect->save();
		}

		public static function deleteRedirect(Redirect $redirect): void{
			$redirect->delete();
		}

	}