<?php

	namespace Redirects;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use System\HttpHelper;

	#[Controller]
	#[RouteBase("/uplift")]
	class RedirectsController extends BaseController{

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/manage-redirects")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_REDIRECTS)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "manage-redirects/main.php",
			);
		}

		#[Route("GET", "/redirects")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_REDIRECTS)]
		public function getRedirects(Request $request): JSONResult
		{
			$limit = $request->getQueryValue("limit");
			$page = $request->getQueryValue("page");
			$query = $request->getQueryValue("query");

			if ($limit === null){
				return new JSONError("Missing 'limit' GET parameter.");
			}

			if ($page === null){
				return new JSONError("Missing 'page' GET parameter.");
			}

			if ($query === null){
				return new JSONError("Missing 'query' GET parameter.");
			}

			$totalRedirectsForQuery = RedirectsService::getTotalRedirectsForQuery($query);

			return new JSONSuccess([
				"redirects"=>RedirectsService::getRedirects(
					page: $page,
					limit: $limit,
					query: $query,
				),
				"totalRedirects"=>$totalRedirectsForQuery,
				"totalPages"=>ceil($totalRedirectsForQuery / $limit),
			]);
		}

		#[Route("DELETE", "@/redirect/(?<redirectID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_REDIRECTS)]
		public function deleteRedirect(Request $request): JSONResult
		{
			$redirectID = (int) $request->getParameter('redirectID');
			$account = Account::getCurrentUser();

			/** @var Redirect | null $redirect */
			$redirect = Redirect::fetch($redirectID);

			if ($redirect === null){
				return new JSONError(sprintf("No redirect with ID %d", $redirectID));
			}else{
				RedirectsService::deleteRedirect($redirect);

				ActivityLog::log(
					categoryID: ActivityLogCategories::DELETE_REDIRECT->value,
					accountID: $account->id,
					ip: $request->getIP(),
					jsonData: json_encode([
						"redirectID"=>$redirectID,
					]),
				);

				return new JSONSuccess();
			}
		}

		#[Route("PUT", "/redirects")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_REDIRECTS)]
		public function newRedirect(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$from = $payload->getTextPayload("from");
				$to = $payload->getTextPayload("to");
				$isRegex = $payload->getTextPayloadNullable("is-regex") !== null;
				$status = $payload->getTextPayload("status");
				$preserveQueryString = $payload->getTextPayloadNullable("preserve-query-string") !== null;
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newRedirect = RedirectsService::addRedirect(
					from: $from->contents,
					to: $to->contents,
					isRegex: $isRegex,
					status: (int) $status->contents,
					preserveQueryString: $preserveQueryString
				);
			} catch (DuplicateFromValue) {
				return new JSONError("A redirect with that same 'From URI' value is already in the system. You cannot have two directs that would redirect the same URI to a location.");
			} catch(\ValueError $e){
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::NEW_REDIRECT->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"redirectID"=>$newRedirect->id,
				]),
			);

			return new JSONSuccess([
				"newRedirect"=>$newRedirect,
			]);
		}

		#[Route("PATCH", "@/redirect/(?<redirectID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_REDIRECTS)]
		public function updateRedirect(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$redirectID = (int) $request->getParameter('redirectID');
			$account = Account::getCurrentUser();

			/** @var Redirect | null $redirect */
			$redirect = Redirect::fetch($redirectID);

			if ($redirect === null){
				return new JSONError(sprintf("No redirect with ID %d", $redirectID));
			}

			try {
				$from = $payload->getTextPayload('from');
				$to = $payload->getTextPayload('to');
				$isRegex = $payload->getTextPayloadNullable('is-regex') !== null;
				$status = $payload->getTextPayload('status');
				$preserveQueryString = $payload->getTextPayloadNullable("preserve-query-string") !== null;
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				RedirectsService::updateRedirect(
					redirect:$redirect,
					from: $from->contents,
					to: $to->contents,
					isRegex: $isRegex,
					status: (int) $status->contents,
					preserveQueryString: $preserveQueryString
				);
			} catch (DuplicateFromValue) {
				return new JSONError("A redirect with that same 'From URI' value is already in the system. You cannot have two directs that would redirect the same URI to a location.");
			} catch(\ValueError $e){
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_REDIRECT->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"redirectID"=>$redirectID,
				]),
			);

			return new JSONSuccess();
		}
	}