<?php

	namespace RSS;

	use Nox\ORM\ColumnQuery;
	use Page\NoH1InBodyContent;
	use Page\NoMetaDescriptionInHead;
	use Page\Page;
	use Page\PageType;
	use Page\PublicationStatus;
	use ShortcodeParser\ShortcodeParser;
	use System\HttpHelper;

	class RSSService
	{

		public static function getXMLForPageType(PageType|null $pageType): \SimpleXMLElement{
			if ($pageType === null){
				/** @var Page[] $pages */
				$pages = Page::query(
					columnQuery: (new ColumnQuery())
						->where("publication_status", "=", PublicationStatus::Published->value)
						->and()
						->where("publication_timestamp", "<=", time())
						->and()
						->where("excludeFromSitemap","=",0)
						->and()
						->where("pageRouteIsRegex","=",0)
				);
			}else{
				/** @var Page[] $pages */
				$pages = Page::query(
					columnQuery: (new ColumnQuery())
						->where("pageType","=",$pageType->name)
						->and()
						->where("publication_status", "=", PublicationStatus::Published->value)
						->and()
						->where("publication_timestamp", "<=", time())
						->and()
						->where("excludeFromSitemap","=",0)
						->and()
						->where("pageRouteIsRegex","=",0)
				);
			}

			$baseUrl = HttpHelper::getWebsiteBaseURL();
			$shortcodeParser = new ShortcodeParser();
			$rss = new \SimpleXMLElement("<?xml version=\"1.0\" encoding=\"utf-8\" ?><rss></rss>");
			$rss->addAttribute("version", "2.0");
			$rss->addAttribute("xmlns:dc", "http://purl.org/dc/elements/1.1/");
			$rss->addAttribute("xmlns:content", "http://purl.org/rss/1.0/modules/content/");
			$rss->addAttribute("xmlns:rdf", "http://www.w3.org/1999/02/22-rdf-syntax-ns#");
			$rss->addAttribute("xmlns:atom", "http://www.w3.org/2005/Atom");

			$channel = $rss->addChild("channel");
			$channel->addChild("link", $baseUrl);
			$channel->addChild("description");

			if ($pageType === null) {
				$channel->addChild("title", "All Pages");
			}else{
				$channel->addChild("title", sprintf("%s Pages", $pageType->name));
			}

			foreach($pages as $page){
				Page::setCurrentPage($page);
				// Get the page title and description
				try {
					$h1 = $page->getFirstH1ElementContent();
					if (!empty($h1)) {
						$pageTitle = $h1;
					}else{
						$pageTitle = $page->pageName;
					}
				}catch(NoH1InBodyContent){
					$pageTitle = $page->pageName;
				}

				try {
					$metaDescription = $page->getMetaDescription();
				}catch(NoMetaDescriptionInHead){
					$metaDescription = "";
				}

				// Parse the shortcodes in the page title and description
				$pageTitle = $shortcodeParser->parse($pageTitle);
				$metaDescription = $shortcodeParser->parse($metaDescription);

				$item = $channel->addChild("item");
				$item->addChild("title", htmlspecialchars($pageTitle));
				$item->addChild("link", sprintf("%s%s", $baseUrl, $page->pageRoute));
				$item->addChild("description", $metaDescription);
				$item->addChild("pubDate", date("r", $page->creationTime));
			}

			return $rss;
		}

	}
