<?php
	namespace ProjectPostTag;

	use Accounts\Account;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\ORM\ColumnQuery;
	use Page\PageData;
	use Page\PageDatas;
	use System\HttpHelper;

	class ProjectPostTagService{
		/**
		 * @throws ProjectTagLabelEmpty
		 * @throws ProjectTagWithSameNameExists
		 */
		public static function createTag(
			string $newTagLabel,
		): ProjectPostTag{

			$newTagLabel = trim($newTagLabel);

			if (strlen($newTagLabel) === 0){
				throw new ProjectTagLabelEmpty("Project tag labels cannot be empty.");
			}

			/** @var ProjectPostTag | null $existingTag */
			$existingTag = ProjectPostTag::queryOne(
				columnQuery: (new ColumnQuery())
					->where("label", "=", $newTagLabel)
			);

			if ($existingTag !== null) {
				throw new ProjectTagWithSameNameExists("A project post tag with that name already exists.");
			}

			$newTag = new ProjectPostTag();
			$newTag->label = $newTagLabel;
			$newTag->save();

			return $newTag;
		}

		/**
		 * @throws ProjectTagDoesntExist
		 */
		public static function deleteTag(
			int $tagID,
		): void{
			/** @var ProjectPostTag $projectTag */
			$projectTag = ProjectPostTag::fetch($tagID);

			if ($projectTag === null){
				throw new ProjectTagDoesntExist("No project tag with ID {$tagID}");
			}

			// Delete all the links it has to any page
			/** @var PageData[] $pageDataProjectTag */
			$pageDataProjectTag = PageData::query(
				columnQuery: (new ColumnQuery())
					->where("name","=", PageDatas::PROJECT_TAG->name)
					->and()
					->where("value", "=", $tagID),
			);

			foreach($pageDataProjectTag as $pageProjectPostTag){
				$pageProjectPostTag->delete();
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_PROJECT_POST_TAG->value,
				accountID: (Account::getCurrentUser())->id,
				ip: HttpHelper::getIPFromRequest(),
				jsonData: json_encode([
					"tagID" => $projectTag->id,
					"tagLabel" => $projectTag->label,
				]),
			);

			$projectTag->delete();
		}

		/**
		 * @throws ProjectTagLabelEmpty
		 * @throws ProjectTagWithSameNameExists
		 * @throws ProjectTagDoesntExist
		 */
		public static function renameTag(
			int $tagID,
			string $newLabel,
		): ProjectPostTag{
			/** @var ProjectPostTag $projectTag */
			$projectTag = ProjectPostTag::fetch($tagID);

			if ($projectTag === null){
				throw new ProjectTagDoesntExist("No project tag with ID {$tagID}");
			}

			$newLabel = trim($newLabel);

			if (strlen($newLabel) === 0){
				throw new ProjectTagLabelEmpty("Project tag labels cannot be empty.");
			}

			// Does a project tag already exist with that label
			$existingTagWithSameLabel = ProjectPostTag::queryOne(
				columnQuery: (new ColumnQuery())
					->where("id","!=",$tagID)
					->and()
					->where("label","=",$newLabel),
			);

			if ($existingTagWithSameLabel !== null){
				throw new ProjectTagWithSameNameExists("A project tag with that name already exists.");
			}

			$projectTag->label = $newLabel;
			$projectTag->save();

			ActivityLog::log(
				categoryID: ActivityLogCategories::RENAME_PROJECT_POST_TAG->value,
				accountID: (Account::getCurrentUser())->id,
				ip: HttpHelper::getIPFromRequest(),
				jsonData: json_encode([
					"tagID" => $projectTag->id,
					"newTagLabel" => $projectTag->label,
				]),
			);

			return $projectTag;
		}
	}