<?php

	namespace PageEditor;

	use Accounts\Account;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageAttribute\PageAttributePageValue;
	use Page\PageBreadcrumb;
	use Page\PageContentSection;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use PageArchives\PageArchive;
	use PageArchives\PageArchivesService;
	use PageArchives\PageBreadrumbArchive;
	use PageArchives\PageContentSectionArchive;
	use PageArchives\PageDataArchive;
	use PageEditor\Exceptions\NoPageFoundWithID;
	use PageEditor\Exceptions\PageNameExists;
	use PageEditor\Exceptions\PageNameExistsWithSamePageType;
	use PageEditor\Exceptions\PageNameIsBlank;
	use PageEditor\Exceptions\PageTypeNotFound;
	use stdClass;
	use System\HttpHelper;
	use System\Themes;
	use TemplateManager\PageLayouts\PageLayoutSection;
	use TemplateManager\PageLayouts\PageLayoutSectionsDefinition;
	use TemplateManager\PageLayouts\PageLayoutSectionsProvider;
	use Uplift\Exceptions\IncompatiblePageType;
	use Uplift\Exceptions\MalformedValue;
	use ValueError;

	class PageEditorService{

		/**
		 * Eventually this should be removed and the revision history viewer made asynchronous as a JavaScript app
		 */
		const PAGE_HISTORY_VIEW_LIMIT = 200;

		/**
		 * @throws NoPageFoundWithID
		 */
		public static function getPageOrThrow(int $pageID): Page{
			/** @var Page | null $page */
			$page = Page::fetch(
				primaryKey: $pageID,
			);

			if ($page === null){
				throw new NoPageFoundWithID("No page found with provided page ID: {$pageID}");
			}

			return $page;
		}

		/**
		 * @throws PageNameExists
		 * @throws ValueError
		 */
		public static function createPage(
			string $pageName,
			string $pageType,
			bool $overwriteExistingPage = false,
		): Page{
			$pageName = trim($pageName);

			if (empty($pageName)){
				throw new ValueError("The page name cannot be blank.");
			}

			$pageTypeEnum = null;
			foreach(PageType::cases() as $case){
				if ($case->name === $pageType){
					$pageTypeEnum = $case;
					break;
				}
			}

			if ($pageTypeEnum === null){
				throw new ValueError("No page type enumeration found with name {$pageType}.");
			}


			$existingPage = Page::queryOne(
				columnQuery: (new ColumnQuery())
					->where("pageName", "=", $pageName),
			);

			if ($existingPage !== null) {
				if (!$overwriteExistingPage) {
					throw new PageNameExists("A page with the name {$pageName} already exists in the system.");
				}else{
					// Delete the existing page
					$existingPage->delete();
				}
			}

			$defaultPageHead = trim(str_replace("\t", "", <<<HTML
				<title></title>
				<meta name="description" itemprop="description" property="og:description" content="">
			HTML));

			$newPage = new Page();
			$newPage->pageName = $pageName;
			$newPage->pageHead = $defaultPageHead;
			$newPage->publicationStatus = PublicationStatus::Published->value;
			$newPage->pageType = $pageTypeEnum->name;
			$newPage->save();

			return $newPage;
		}

		private static function clearPageData(
			Page $page,
		): void{
			/** @var PageData[] $pageDatas */
			$pageDatas = PageData::query(
				columnQuery: (new ColumnQuery())
					->where("page_id", "=", $page->id),
			);

			foreach($pageDatas as $pageData){
				$pageData->delete();
			}
		}

		/**
		 * Saves the page's properties - no custom data.
		 * @param Page $page
		 * @param string $pageName
		 * @param string $pageRoute
		 * @param bool $pageRouteIsRegex
		 * @param string $pageLayout
		 * @param string $pageBody
		 * @param string $pageHead
		 * @param bool $excludedFromSitemap
		 * @param int $publicationStatus
		 * @param int $publicationTimestamp
		 * @return void
		 * @throws Exceptions\PageLayoutDoesntExist
		 * @throws Exceptions\PageLayoutIsBlank
		 * @throws Exceptions\PageNameExistsWithSamePageType
		 * @throws Exceptions\PageNameIsBlank
		 * @throws Exceptions\PageRouteInUse
		 * @throws Exceptions\PageRouteIsBlank
		 * @throws Exceptions\PublicationStatusDoesntExist
		 * @throws MalformedValue
		 */
		private static function savePage(
			Page $page,
			string $pageName,
			string $pageRoute,
			bool $pageRouteIsRegex,
			string $pageLayout,
			string $pageBody,
			string $pageHead,
			bool $excludedFromSitemap,
			bool $excludedFromSchema,
			int $publicationStatus,
			int $publicationTimestamp,
		): void{
			// Validate the common properties
			PageEditorPageValidationService::validateCommonProperties(
				page:$page,
				pageName:$pageName,
				pageRoute:$pageRoute,
				pageLayout:$pageLayout,
				publicationStatus: $publicationStatus,
			);

			$page->pageName = $pageName;
			$page->pageBody = $pageBody;
			$page->pageHead = $pageHead;
			$page->pageRoute = $pageRoute;
			$page->pageLayout = $pageLayout;
			$page->pageRouteIsRegex = $pageRouteIsRegex ? 1 : 0;
			$page->excludeFromSitemap = $excludedFromSitemap ? 1 : 0;
			$page->excludeSchemaInjection = $excludedFromSchema ? 1 : 0;
			$page->publicationStatus = $publicationStatus;
			$page->publicationTimestamp = $publicationTimestamp;
			$page->save();
		}

		/**
		 * @throws Exceptions\PageRouteInUse
		 * @throws Exceptions\PageRouteIsBlank
		 * @throws Exceptions\PageNameExistsWithSamePageType
		 * @throws Exceptions\PageNameIsBlank
		 * @throws Exceptions\PageLayoutIsBlank
		 * @throws Exceptions\PageLayoutDoesntExist
		 * @throws Exceptions\PublicationStatusDoesntExist
		 * @throws ValueError
		 * @throws MalformedValue
		 */
		public static function saveGeneralPage(
			Page $page,
			string $pageName,
			string $pageRoute,
			bool $pageRouteIsRegex,
			string $pageLayout,
			string $pageBody,
			string $pageHead,
			bool $excludedFromSitemap,
			bool $excludedFromSchema,
			int $publicationStatus,
			int $publicationTimestamp,
		): Page{

			self::savePage(
				page: $page,
				pageName: $pageName,
				pageRoute: $pageRoute,
				pageRouteIsRegex:$pageRouteIsRegex,
				pageLayout: $pageLayout,
				pageBody: $pageBody,
				pageHead: $pageHead,
				excludedFromSitemap: $excludedFromSitemap,
				excludedFromSchema: $excludedFromSchema,
				publicationStatus: $publicationStatus,
				publicationTimestamp: $publicationTimestamp,
			);

			// Clear the page's custom data, if it has any
			// This can happen if the page was converted
			self::clearPageData($page);

			return $page;
		}

		/**
		 * @throws Exceptions\PageRouteInUse
		 * @throws Exceptions\PageRouteIsBlank
		 * @throws Exceptions\PageNameExistsWithSamePageType
		 * @throws Exceptions\PageNameIsBlank
		 * @throws Exceptions\PageLayoutIsBlank
		 * @throws Exceptions\PageLayoutDoesntExist
		 * @throws Exceptions\PublicationStatusDoesntExist
		 * @throws ValueError
		 * @throws MalformedValue
		 */
		public static function saveServicePage(
			Page $page,
			string $pageName,
			string $pageRoute,
			bool $pageRouteIsRegex,
			string $pageLayout,
			string $pageBody,
			string $pageHead,
			bool $excludedFromSitemap,
			bool $excludedFromSchema,
			int $publicationStatus,
			int $publicationTimestamp,
			string $featuredImageURI,
			string $featuredImageThumbURI,
		): Page{

			self::savePage(
				page: $page,
				pageName: $pageName,
				pageRoute: $pageRoute,
				pageRouteIsRegex:$pageRouteIsRegex,
				pageLayout: $pageLayout,
				pageBody: $pageBody,
				pageHead: $pageHead,
				excludedFromSitemap: $excludedFromSitemap,
				excludedFromSchema: $excludedFromSchema,
				publicationStatus: $publicationStatus,
				publicationTimestamp: $publicationTimestamp,
			);

			// Clear the page's custom data, if it has any
			// This can happen if the page was converted
			self::clearPageData($page);

			// Save the featured image
			self::saveFeaturedImageForPage(
				page: $page,
				featuredImageURI:$featuredImageURI,
				featuredImageThumbURI:$featuredImageThumbURI,
			);

			return $page;
		}


		/**
		 * @throws Exceptions\PageRouteInUse
		 * @throws Exceptions\PageRouteIsBlank
		 * @throws Exceptions\PageNameExistsWithSamePageType
		 * @throws Exceptions\PageNameIsBlank
		 * @throws Exceptions\PageLayoutIsBlank
		 * @throws Exceptions\PageLayoutDoesntExist
		 * @throws Exceptions\PublicationStatusDoesntExist
		 * @throws ValueError
		 */
		public static function saveBlogPage(
			Page $page,
			string $pageName,
			string $pageRoute,
			bool $pageRouteIsRegex,
			string $pageLayout,
			string $pageBody,
			string $pageHead,
			bool $excludedFromSitemap,
			bool $excludedFromSchema,
			string $publicationStatus,
			int $publicationTimestamp,
			string $featuredImageURI,
			string $featuredImageThumbURI,
			array $articleCategoryIDs,
		): Page{

			self::savePage(
				page: $page,
				pageName: $pageName,
				pageRoute: $pageRoute,
				pageRouteIsRegex:$pageRouteIsRegex,
				pageLayout: $pageLayout,
				pageBody: $pageBody,
				pageHead: $pageHead,
				excludedFromSitemap: $excludedFromSitemap,
				excludedFromSchema: $excludedFromSchema,
				publicationStatus: $publicationStatus,
				publicationTimestamp: $publicationTimestamp,
			);

			// Clear the page's custom data first
			self::clearPageData($page);

			// Save the featured image
			self::saveFeaturedImageForPage(
				page: $page,
				featuredImageURI:$featuredImageURI,
				featuredImageThumbURI:$featuredImageThumbURI,
			);

			// Prepare array to save custom data
			$pageDatasToSave = [];

			// Article categories
			foreach($articleCategoryIDs as $categoryID){
				$articleCategoryData = new PageData();
				$articleCategoryData->name = PageDatas::ARTICLE_CATEGORY->name;
				$articleCategoryData->value = $categoryID;
				$articleCategoryData->pageID = $page->id;
				$pageDatasToSave[] = $articleCategoryData;
			}

			// Save all the custom page data
			PageData::saveAll($pageDatasToSave);

			return $page;
		}

		/**
		 * @throws Exceptions\PublicationStatusDoesntExist
		 * @throws PageNameExistsWithSamePageType
		 * @throws PageNameIsBlank
		 * @throws Exceptions\PageLayoutDoesntExist
		 * @throws Exceptions\PageRouteInUse
		 * @throws Exceptions\PageRouteIsBlank
		 * @throws Exceptions\PageLayoutIsBlank
		 * @throws ValueError
		 * @throws MalformedValue
		 */
		public static function saveProjectPage(
			Page $page,
			string $pageName,
			string $pageRoute,
			bool $pageRouteIsRegex,
			string $pageLayout,
			string $pageBody,
			string $pageHead,
			bool $excludedFromSitemap,
			bool $excludedFromSchema,
			string $publicationStatus,
			int $publicationTimestamp,
			string $featuredImageURI,
			string $featuredImageThumbURI,
			string $cityName,
			string $stateName,
			string $stateNameShorthand,
			string $brandsProducts,
			bool $customerDidLeaveReview,
			string $customerReviewFirstName,
			string $customerReviewLastName,
			string $customerReviewTestimonial,
			array $projectTagIDs,
		): Page{

			self::savePage(
				page: $page,
				pageName: $pageName,
				pageRoute: $pageRoute,
				pageRouteIsRegex:$pageRouteIsRegex,
				pageLayout: $pageLayout,
				pageBody: $pageBody,
				pageHead: $pageHead,
				excludedFromSitemap: $excludedFromSitemap,
				excludedFromSchema: $excludedFromSchema,
				publicationStatus: $publicationStatus,
				publicationTimestamp: $publicationTimestamp,
			);

			// Clear the page's custom data first
			self::clearPageData($page);

			// Save the featured image
			self::saveFeaturedImageForPage(
				page: $page,
				featuredImageURI:$featuredImageURI,
				featuredImageThumbURI:$featuredImageThumbURI,
			);

			// Prepare array to save custom data
			$pageDatasToSave = [];

			// Project categories
			foreach($projectTagIDs as $tagID){
				$projectTagData = new PageData();
				$projectTagData->name = PageDatas::PROJECT_TAG->name;
				$projectTagData->value = $tagID;
				$projectTagData->pageID = $page->id;
				$pageDatasToSave[] = $projectTagData;
			}

			// Save city name
			$projectCityNameData = new PageData();
			$projectCityNameData->name = PageDatas::CITY_NAME->name;
			$projectCityNameData->value = $cityName;
			$projectCityNameData->pageID = $page->id;
			$pageDatasToSave[] = $projectCityNameData;

			// Save state name
			$projectStateNameData = new PageData();
			$projectStateNameData->name = PageDatas::STATE_NAME->name;
			$projectStateNameData->value = $stateName;
			$projectStateNameData->pageID = $page->id;
			$pageDatasToSave[] = $projectStateNameData;

			// Save state name shorthand
			$projectStateNameShorthandData = new PageData();
			$projectStateNameShorthandData->name = PageDatas::STATE_NAME_SHORTHAND->name;
			$projectStateNameShorthandData->value = $stateNameShorthand;
			$projectStateNameShorthandData->pageID = $page->id;
			$pageDatasToSave[] = $projectStateNameShorthandData;

			// Save brands and/or products
			$projectBrandsProductsData = new PageData();
			$projectBrandsProductsData->name = PageDatas::PROJECT_BRANDS_PRODUCTS->name;
			$projectBrandsProductsData->value = $brandsProducts;
			$projectBrandsProductsData->pageID = $page->id;
			$pageDatasToSave[] = $projectBrandsProductsData;

			// Save flag on if a customer left a review
			$projectCustomerLeftReviewData = new PageData();
			$projectCustomerLeftReviewData->name = PageDatas::CUSTOMER_DID_LEAVE_REVIEW->name;
			$projectCustomerLeftReviewData->value = $customerDidLeaveReview ? 1 : 0;
			$projectCustomerLeftReviewData->pageID = $page->id;
			$pageDatasToSave[] = $projectCustomerLeftReviewData;

			// Save the customer review data if they left a review
			if ($customerDidLeaveReview) {
				// Save customer first name
				$projectCustomerFirstNameData = new PageData();
				$projectCustomerFirstNameData->name = PageDatas::CUSTOMER_REVIEW_FIRST_NAME->name;
				$projectCustomerFirstNameData->value = $customerReviewFirstName;
				$projectCustomerFirstNameData->pageID = $page->id;
				$pageDatasToSave[] = $projectCustomerFirstNameData;

				// Save customer last name
				$projectCustomerLastNameData = new PageData();
				$projectCustomerLastNameData->name = PageDatas::CUSTOMER_REVIEW_LAST_NAME->name;
				$projectCustomerLastNameData->value = $customerReviewLastName;
				$projectCustomerLastNameData->pageID = $page->id;
				$pageDatasToSave[] = $projectCustomerLastNameData;

				// Save customer testimonial body
				$projectCustomerTestimonialData = new PageData();
				$projectCustomerTestimonialData->name = PageDatas::CUSTOMER_REVIEW_TESTIMONIAL->name;
				$projectCustomerTestimonialData->value = $customerReviewTestimonial;
				$projectCustomerTestimonialData->pageID = $page->id;
				$pageDatasToSave[] = $projectCustomerTestimonialData;
			}

			// Save all the custom page data
			PageData::saveAll($pageDatasToSave);

			return $page;
		}

		/**
		 * @throws Exceptions\PublicationStatusDoesntExist
		 * @throws PageNameIsBlank
		 * @throws PageNameExistsWithSamePageType
		 * @throws Exceptions\PageLayoutDoesntExist
		 * @throws Exceptions\PageRouteInUse
		 * @throws Exceptions\PageRouteIsBlank
		 * @throws Exceptions\PageLayoutIsBlank
		 * @throws ValueError
		 * @throws MalformedValue
		 */
		public static function saveCityPage(
			Page $page,
			string $pageName,
			string $pageRoute,
			bool $pageRouteIsRegex,
			string $pageLayout,
			string $pageBody,
			string $pageHead,
			bool $excludedFromSitemap,
			bool $excludedFromSchema,
			string $publicationStatus,
			int $publicationTimestamp,
			string $cityName,
			string $stateName,
			string $stateNameShorthand,
			string $officialCityURL,
			string $featuredImageURI,
			string $featuredImageThumbURI,
		): Page{

			self::savePage(
				page: $page,
				pageName: $pageName,
				pageRoute: $pageRoute,
				pageRouteIsRegex:$pageRouteIsRegex,
				pageLayout: $pageLayout,
				pageBody: $pageBody,
				pageHead: $pageHead,
				excludedFromSitemap: $excludedFromSitemap,
				excludedFromSchema: $excludedFromSchema,
				publicationStatus: $publicationStatus,
				publicationTimestamp: $publicationTimestamp,
			);

			// Clear the page's custom data first
			self::clearPageData($page);

			// Save the featured image
			self::saveFeaturedImageForPage(
				page: $page,
				featuredImageURI:$featuredImageURI,
				featuredImageThumbURI:$featuredImageThumbURI,
			);

			// Save city name
			$cityNameData = new PageData();
			$cityNameData->name = PageDatas::CITY_NAME->name;
			$cityNameData->value = $cityName;
			$cityNameData->pageID = $page->id;
			$pageDatasToSave[] = $cityNameData;

			// Save state name
			$stateNameData = new PageData();
			$stateNameData->name = PageDatas::STATE_NAME->name;
			$stateNameData->value = $stateName;
			$stateNameData->pageID = $page->id;
			$pageDatasToSave[] = $stateNameData;

			// Save state name shorthand
			$stateNameShorthandData = new PageData();
			$stateNameShorthandData->name = PageDatas::STATE_NAME_SHORTHAND->name;
			$stateNameShorthandData->value = $stateNameShorthand;
			$stateNameShorthandData->pageID = $page->id;
			$pageDatasToSave[] = $stateNameShorthandData;

			// Save official city URL
			$cityURLData = new PageData();
			$cityURLData->name = PageDatas::OFFICIAL_CITY_URL->name;
			$cityURLData->value = $officialCityURL;
			$cityURLData->pageID = $page->id;
			$pageDatasToSave[] = $cityURLData;

			// Save all the custom page data
			PageData::saveAll($pageDatasToSave);

			return $page;
		}

		public static function saveFeaturedImageForPage(
			Page $page,
			string $featuredImageURI,
			string $featuredImageThumbURI,
		): void
		{
			$pageDatasToSave = [];

			$featuredImageData = new PageData();
			$featuredImageData->name = PageDatas::FEATURED_IMAGE->name;
			$featuredImageData->value = $featuredImageURI;
			$featuredImageData->pageID = $page->id;
			$pageDatasToSave[] = $featuredImageData;

			$featuredImageThumbData = new PageData();
			$featuredImageThumbData->name = PageDatas::FEATURED_IMAGE_THUMB->name;
			$featuredImageThumbData->value = $featuredImageThumbURI;
			$featuredImageThumbData->pageID = $page->id;
			$pageDatasToSave[] = $featuredImageThumbData;

			PageData::saveAll($pageDatasToSave);
		}

		/**
		 * Returns the breadcrumbs saved.
		 * @param Page $page
		 * @param array $breadcrumbs This array only needs arrays inside of it with "label" and "uri" keys
		 * @return array
		 * @throws NoPrimaryKey
		 */
		public static function saveBreadcrumbs(
			Page $page,
			array $breadcrumbs,
		): array
		{

			// Delete the old ones
			/** @var PageBreadcrumb[] $existingCrumbs */
			$existingCrumbs = PageBreadcrumb::query(
				columnQuery: (new ColumnQuery())
					->where("pageID", "=", $page->id),
			);

			foreach ($existingCrumbs as $crumb) {
				$crumb->delete();
			}

			/** @var PageBreadcrumb[] $crumbs */
			$crumbs = [];

			/** @var array{label: string, uri: string} $breadcrumb */
			foreach ($breadcrumbs as $base1Index => $breadcrumb) {
				$breadcrumbObject = new PageBreadcrumb();
				$breadcrumbObject->pageID = $page->id;
				$breadcrumbObject->label = $breadcrumb['label'];
				$breadcrumbObject->uri = $breadcrumb['uri'];
				$breadcrumbObject->position = $base1Index;
				$crumbs[] = $breadcrumbObject;
			}

			PageBreadcrumb::saveAll($crumbs);
			return $crumbs;
		}

		public static function savePageAttributeValues(
			array $pageAttributeValues,
		): void{
			/** @var array{id: int, pageAttributeID: int, name: string, value: string} $pageAttributeValue */
			foreach($pageAttributeValues as $pageAttributeValue){
				/** @var PageAttributePageValue $pageAttributeValueObject */
				$pageAttributeValueObject = PageAttributePageValue::fetch($pageAttributeValue['id']);
				$pageAttributeValueObject->value = $pageAttributeValue['value'];
				$pageAttributeValueObject->save();
			}
		}

		/**
		 * @throws NoPageFoundWithID
		 * @throws PageTypeNotFound
		 */
		public static function convertPage(
			int $pageID,
			string $newPageType,
		): void{
			/** @var Page | null $page */
			$page = Page::fetch($pageID);

			if ($page === null){
				throw new NoPageFoundWithID("No page found with ID $pageID");
			}

			if (!PageType::has($newPageType)){
				throw new PageTypeNotFound("No page type by the name of $newPageType found.");
			}

			$previousPageType = $page->pageType;
			$page->pageType = $newPageType;
			$page->save();

			ActivityLog::log(
				categoryID: ActivityLogCategories::CONVERT_PAGE_TYPE->value,
				accountID: (Account::getCurrentUser())->id,
				ip: HttpHelper::getIPFromRequest(),
				jsonData: json_encode([
					"pageID"=>$page->id,
					"pageName"=>$page->pageName,
					"previousPageType"=>$previousPageType,
					"newPageType"=>$page->pageType,
				]),
			);
		}

		/**
		 * @throws NoPageFoundWithID
		 * @throws PageNameIsBlank
		 * @throws PageNameExistsWithSamePageType
		 */
		public static function clonePage(
			int $pageID,
			string $newPageName,
		): Page{

			$newPageName = trim($newPageName);

			if (strlen($newPageName) === 0){
				throw new PageNameIsBlank("Cloned page name cannot be blank.");
			}

			/** @var Page | null $page */
			$page = Page::fetch($pageID);

			if ($page === null){
				throw new NoPageFoundWithID("No page found with ID $pageID");
			}

			// Existing page with the same name?
			/** @var Page | null $existingPage */
			$existingPage = Page::queryOne(
				columnQuery: (new ColumnQuery())
					->where("pageName", "=", $newPageName)
			);

			if ($existingPage !== null){
				throw new PageNameExistsWithSamePageType("A page with the name \"$newPageName\" already exists with the current page type that is attempting to be cloned.");
			}

			// Clone the page
			$clonedPage = clone $page;

			// Clear the ID so it will create a new page
			$clonedPage->id = null;
			$clonedPage->pageName = $newPageName;
			$clonedPage->pageRoute = "";
			$clonedPage->publicationStatus = PublicationStatus::Draft->value;
			$clonedPage->publicationTimestamp = 0;
			$clonedPage->save();

			// Clone page sections
			$existingPageSections = PageContentSection::query(
				columnQuery: (new ColumnQuery())
					->where("page_id", "=", $pageID)
			);

			foreach($existingPageSections as $pageSection){
				$pageSection->id = null;
				$pageSection->pageId = $clonedPage->id;
				$pageSection->save();
			}

			// Clone breadcrumbs
			$existingPageBreadcrumbs = PageBreadcrumb::query(
				columnQuery: (new ColumnQuery())
					->where("pageID", "=", $pageID)
			);

			foreach($existingPageBreadcrumbs as $breadcrumb){
				$breadcrumb->id = null;
				$breadcrumb->pageID = $clonedPage->id;
				$breadcrumb->save();
			}

			// Clone all the page data too
			/** @var PageData[] $pageData */
			$pageData = PageData::query(
				columnQuery: (new ColumnQuery())
					->where("page_id","=",$page->id)
			);

			foreach($pageData as $data){
				$data->id = null;
				$data->pageID = $clonedPage->id;
				$data->save();
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CLONE_PAGE->value,
				accountID: (Account::getCurrentUser())->id,
				ip: HttpHelper::getIPFromRequest(),
				jsonData: json_encode([
					"referencePageID"=>$page->id,
					"clonedPageID"=>$clonedPage->id,
				]),
			);

			return $clonedPage;
		}

		/**
		 * @throws NoPrimaryKey
		 * @throws NoPageFoundWithID
		 */
		public static function deletePage(
			int $pageID,
		): void{

			/** @var Page | null $page */
			$page = Page::fetch($pageID);

			if ($page === null){
				throw new NoPageFoundWithID("No page found with ID $pageID");
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_PAGE->value,
				accountID: (Account::getCurrentUser())->id,
				ip: HttpHelper::getIPFromRequest(),
				jsonData: json_encode([
					"pageID"=>$page->id,
					"pageName"=>$page->pageName,
				]),
			);

			$page->delete();
		}

		/**
		 * @return stdClass[]
		 */
		public static function getAllDehydratedPageRevisions(
			int $pageID
		): array{
			/** @var Page $page */
			$page = Page::fetch($pageID);

			/** @var PageArchive[] $archives */
			$archives = PageArchive::query(
				columnQuery: (new ColumnQuery())
					->where("page_id","=",$page->id),
				pager: (new Pager(pageNumber:1, limit: self::PAGE_HISTORY_VIEW_LIMIT))
			);

			$dehydratedObjects = [];

			foreach($archives as $archive){
				$obj = new stdClass();
				$obj->id = $archive->id;
				$obj->archivedTimestamp = $archive->archivedTimestamp;
				$obj->pageHeadLength = strlen($archive->pageHead);
				$obj->pageBodyLength = strlen($archive->pageBody);
				$dehydratedObjects[] = $obj;
			}

			return $dehydratedObjects;
		}

		/**
		 * @return array{page: PageArchive, data: PageDataArchive[], breadcrumbs: PageBreadrumbArchive[], pageContentSections: PageContentSectionArchive[]}
		 */
		public static function getAllPageRevisionArchives(
			int $revisionID
		): array{
			$pageArchive = PageArchive::fetch($revisionID);

			/** @var PageDataArchive[] $pageDataArchives */
			$pageDataArchives = PageDataArchive::query(
				columnQuery: (new ColumnQuery())
					->where("page_archive_id","=",$revisionID)
			);

			/** @var PageBreadrumbArchive[] $breadcrumbArchives */
			$breadcrumbArchives = PageBreadrumbArchive::query(
				columnQuery: (new ColumnQuery())
					->where("page_archive_id","=",$revisionID)
			);

			// The page data handler on the front-end expects a key/value array and not an array of objects
			$pageDataKeyValue = [];
			foreach($pageDataArchives as $pageDataArchive){
				if (array_key_exists(key:$pageDataArchive->name, array:$pageDataKeyValue)){
					if (!is_array($pageDataKeyValue[$pageDataArchive->name])) {
						// Convert it to an array
						$pageDataKeyValue[$pageDataArchive->name] = [
							$pageDataKeyValue[$pageDataArchive->name],
							$pageDataArchive->value,
						];
					}else{
						$pageDataKeyValue[$pageDataArchive->name][] = $pageDataArchive->value;
					}
				}else{
					$pageDataKeyValue[$pageDataArchive->name] = $pageDataArchive->value;
				}
			}

			/** @var PageContentSectionArchive[] $contentSectionArchives */
			$contentSectionArchives = PageContentSectionArchive::query(
				columnQuery: (new ColumnQuery())
					->where("archive_id","=",$revisionID)
			);

			return [
				"page"=>$pageArchive,
				"data"=>$pageDataKeyValue,
				"breadcrumbs"=>$breadcrumbArchives,
				"pageContentSections"=>$contentSectionArchives,
			];
		}

		/**
		 * Restores a revision into the live page
		 * @throws Exceptions\PageLayoutDoesntExist
		 * @throws Exceptions\PageLayoutIsBlank
		 * @throws Exceptions\PageRouteInUse
		 * @throws Exceptions\PageRouteIsBlank
		 * @throws Exceptions\PublicationStatusDoesntExist
		 * @throws NoPrimaryKey
		 * @throws PageNameExistsWithSamePageType
		 * @throws PageNameIsBlank
		 * @throws IncompatiblePageType
		 */
		public static function restoreRevision(int $revisionID): void{
			/** @var PageArchive $archive */
			$archive = PageArchive::fetch($revisionID);

			/** @var PageBreadrumbArchive[] $breadcrumbsArchive */
			$breadcrumbsArchive = PageBreadrumbArchive::query(
				columnQuery: (new ColumnQuery())
					->where("page_archive_id","=",$archive->id),
				resultOrder:(new ResultOrder())
					->by("position", "ASC")
			);

			// Turn the breadcrumbs into an array to use with saveBreadcrumbs()
			$breadcrumbsArchiveArrays = [];
			foreach($breadcrumbsArchive as $breadcrumbArchive){
				$breadcrumbsArchiveArrays[] = [
					"position"=>$breadcrumbArchive->position,
					"uri"=>$breadcrumbArchive->uri,
					"label"=>$breadcrumbArchive->label,
				];
			}

			$pageDataArchivesByKey = [];
			/** @var PageDataArchive[] $pageDataArchives */
			$pageDataArchives = PageDataArchive::query(
				columnQuery: (new ColumnQuery())
					->where("page_archive_id","=",$archive->id),
			);

			foreach($pageDataArchives as $pageDataArchive){
				if (!isset($pageDataArchivesByKey[$pageDataArchive->name])) {
					$pageDataArchivesByKey[$pageDataArchive->name] = $pageDataArchive->value;
				}else{
					if (!is_array($pageDataArchivesByKey[$pageDataArchive->name])) {
						$pageDataArchivesByKey[$pageDataArchive->name] = [
							$pageDataArchivesByKey[$pageDataArchive->name],
							$pageDataArchive->value,
						];
					}else{
						$pageDataArchivesByKey[$pageDataArchive->name][] = $pageDataArchive->value;
					}
				}
			}

			// Restore any page content sections
			/** @var PageContentSectionArchive[] $archivedPageContentSections */
			$archivedPageContentSections = PageContentSectionArchive::query(
				columnQuery: (new ColumnQuery())
					->where("archive_id", "=", $archive->id)
			);

			// Remove all existing page content sections
			/** @var PageContentSection[] $existingPageContentSections */
			$existingPageContentSections = PageContentSection::query(
				columnQuery: (new ColumnQuery())
					->where("page_id", "=", $archive->pageID)
			);

			foreach ($existingPageContentSections as $section){
				$section->delete();
			}

			// Make an array of new PageContentSection objects. Set their Ids to null
			/** @var PageContentSection[] $pageContentSectionsToCreate */
			$pageContentSectionsToCreate = [];
			foreach($archivedPageContentSections as $archivedSection){
				$newSection = new PageContentSection();
				$newSection->pageId = $archive->pageID;
				$newSection->sectionName = $archivedSection->sectionName;
				$newSection->content = $archivedSection->content;
				$newSection->id = null;
				$pageContentSectionsToCreate[] = $newSection;
			}


			/** @var Page $page */
			$page = Page::fetch($archive->pageID);

			if ($archive->pageType === PageType::General->name){
				self::saveGeneralPage(
					page: $page,
					pageName: $archive->pageName,
					pageRoute: $archive->pageRoute,
					pageRouteIsRegex: $archive->pageRouteIsRegex === 1,
					pageLayout: $archive->pageLayout,
					pageBody: $archive->pageBody,
					pageHead: $archive->pageHead,
					excludedFromSitemap: $archive->excludeFromSitemap === 1,
					excludedFromSchema: $archive->excludeSchemaInjection === 1,
					publicationStatus: $archive->publicationStatus,
					publicationTimestamp: $archive->publicationTimestamp
				);
				self::saveBreadcrumbs($page, $breadcrumbsArchiveArrays);
				self::savePageContentSections($page, $pageContentSectionsToCreate);
				PageArchivesService::archiveGeneralPage($page);
			}elseif ($archive->pageType === PageType::Service->name){
				self::saveServicePage(
					page: $page,
					pageName: $archive->pageName,
					pageRoute: $archive->pageRoute,
					pageRouteIsRegex: $archive->pageRouteIsRegex === 1,
					pageLayout: $archive->pageLayout,
					pageBody: $archive->pageBody,
					pageHead: $archive->pageHead,
					excludedFromSitemap: $archive->excludeFromSitemap === 1,
					excludedFromSchema: $archive->excludeSchemaInjection === 1,
					publicationStatus: $archive->publicationStatus,
					publicationTimestamp: $archive->publicationTimestamp,
					featuredImageURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE->name] ?? "",
					featuredImageThumbURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE_THUMB->name] ?? ""
				);
				self::saveBreadcrumbs($page, $breadcrumbsArchiveArrays);
				self::savePageContentSections($page, $pageContentSectionsToCreate);
				PageArchivesService::archiveServicePage($page);
			}elseif ($archive->pageType === PageType::City->name){
				self::saveCityPage(
					page: $page,
					pageName: $archive->pageName,
					pageRoute: $archive->pageRoute,
					pageRouteIsRegex: $archive->pageRouteIsRegex === 1,
					pageLayout: $archive->pageLayout,
					pageBody: $archive->pageBody,
					pageHead: $archive->pageHead,
					excludedFromSitemap: $archive->excludeFromSitemap === 1,
					excludedFromSchema: $archive->excludeSchemaInjection === 1,
					publicationStatus: $archive->publicationStatus,
					publicationTimestamp: $archive->publicationTimestamp,
					cityName: $pageDataArchivesByKey[PageDatas::CITY_NAME->name] ?? "",
					stateName: $pageDataArchivesByKey[PageDatas::STATE_NAME->name] ?? "",
					stateNameShorthand: $pageDataArchivesByKey[PageDatas::STATE_NAME_SHORTHAND->name] ?? "",
					officialCityURL: $pageDataArchivesByKey[PageDatas::OFFICIAL_CITY_URL->name] ?? "",
					featuredImageURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE->name] ?? "",
					featuredImageThumbURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE_THUMB->name] ?? "",
				);
				self::saveBreadcrumbs($page, $breadcrumbsArchiveArrays);
				self::savePageContentSections($page, $pageContentSectionsToCreate);
				PageArchivesService::archiveCityPage($page);
			}elseif ($archive->pageType === PageType::Blog->name){

				// Make sure the ARTICLE_CATEGORY key is an array
				// It won't be an array if there is only one article category
				if (isset($pageDataArchivesByKey[PageDatas::ARTICLE_CATEGORY->name])){
					if (!is_array($pageDataArchivesByKey[PageDatas::ARTICLE_CATEGORY->name])){
						$pageDataArchivesByKey[PageDatas::ARTICLE_CATEGORY->name] = [$pageDataArchivesByKey[PageDatas::ARTICLE_CATEGORY->name]];
					}
				}

				self::saveBlogPage(
					page: $page,
					pageName: $archive->pageName,
					pageRoute: $archive->pageRoute,
					pageRouteIsRegex: $archive->pageRouteIsRegex === 1,
					pageLayout: $archive->pageLayout,
					pageBody: $archive->pageBody,
					pageHead: $archive->pageHead,
					excludedFromSitemap: $archive->excludeFromSitemap === 1,
					excludedFromSchema: $archive->excludeSchemaInjection === 1,
					publicationStatus: $archive->publicationStatus,
					publicationTimestamp: $archive->publicationTimestamp,
					featuredImageURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE->name] ?? "",
					featuredImageThumbURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE_THUMB->name] ?? "",
					articleCategoryIDs:$pageDataArchivesByKey[PageDatas::ARTICLE_CATEGORY->name] ?? [],
				);
				self::saveBreadcrumbs($page, $breadcrumbsArchiveArrays);
				self::savePageContentSections($page, $pageContentSectionsToCreate);
				PageArchivesService::archiveBlogPage($page);
			}elseif ($archive->pageType === PageType::Project->name){

				// Make sure the PROJECT_TAG key is an array
				// It won't be an array if there is only one project tag
				if (isset($pageDataArchivesByKey[PageDatas::PROJECT_TAG->name])){
					if (!is_array($pageDataArchivesByKey[PageDatas::PROJECT_TAG->name])){
						$pageDataArchivesByKey[PageDatas::PROJECT_TAG->name] = [$pageDataArchivesByKey[PageDatas::PROJECT_TAG->name]];
					}
				}

				self::saveProjectPage(
					page: $page,
					pageName: $archive->pageName,
					pageRoute: $archive->pageRoute,
					pageRouteIsRegex: $archive->pageRouteIsRegex === 1,
					pageLayout: $archive->pageLayout,
					pageBody: $archive->pageBody,
					pageHead: $archive->pageHead,
					excludedFromSitemap: $archive->excludeFromSitemap === 1,
					excludedFromSchema: $archive->excludeSchemaInjection === 1,
					publicationStatus: $archive->publicationStatus,
					publicationTimestamp: $archive->publicationTimestamp,
					featuredImageURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE->name] ?? "",
					featuredImageThumbURI: $pageDataArchivesByKey[PageDatas::FEATURED_IMAGE_THUMB->name] ?? "",
					cityName: $pageDataArchivesByKey[PageDatas::CITY_NAME->name] ?? "",
					stateName: $pageDataArchivesByKey[PageDatas::STATE_NAME->name],
					stateNameShorthand: $pageDataArchivesByKey[PageDatas::STATE_NAME_SHORTHAND->name] ?? "",
					brandsProducts: $pageDataArchivesByKey[PageDatas::PROJECT_BRANDS_PRODUCTS->name] ?? "",
					customerDidLeaveReview: isset($pageDataArchivesByKey[PageDatas::CUSTOMER_DID_LEAVE_REVIEW->name]) && $pageDataArchivesByKey[PageDatas::CUSTOMER_DID_LEAVE_REVIEW->name] === "1",
					customerReviewFirstName: $pageDataArchivesByKey[PageDatas::CUSTOMER_REVIEW_FIRST_NAME->name],
					customerReviewLastName: $pageDataArchivesByKey[PageDatas::CUSTOMER_REVIEW_LAST_NAME->name] ?? "",
					customerReviewTestimonial: $pageDataArchivesByKey[PageDatas::CUSTOMER_REVIEW_TESTIMONIAL->name],
					projectTagIDs: $pageDataArchivesByKey[PageDatas::PROJECT_TAG->name] ?? [],
				);
				self::saveBreadcrumbs($page, $breadcrumbsArchiveArrays);
				self::savePageContentSections($page, $pageContentSectionsToCreate);
				PageArchivesService::archiveProjectPage($page);
			}
		}

		/**
		 * Uses the provided layout name to try and find a PageLayoutSectionsDefinition
		 * with that layout file name.
		 * @throws InvalidArgumentException
		 * @throws Exception
		 */
		public static function getPageLayoutSectionsDefinitionFromLayoutFileName(string $layoutFileName): ?PageLayoutSectionsDefinition{
			if (strlen(trim($layoutFileName)) === 0){
				throw new \InvalidArgumentException("layoutFileName cannot be blank.");
			}

			// Add the file extension to it, as it will be without one
			$layoutFileName = $layoutFileName . ".php";

			$themeDirectory = realpath(Themes::getCurrentThemeDirectory());
			$layoutsDirectory = realpath($themeDirectory . "/layouts");
			$fullLayoutFilePath = realpath($layoutsDirectory . "/$layoutFileName");
			$sectionsProvider = new PageLayoutSectionsProvider($fullLayoutFilePath);
			try{
				return $sectionsProvider->getSectionDefinition();
			}catch(Exception $ex){
				return null;
			}
		}

		/**
		 * Saves an array of PageContentSection DTOs that may have null Ids (create them instead of edit them.
		 * Returns the updated objects with all null Ids now populated.
		 * @param Page $page
		 * @param array $pageContentSections
		 * @return PageContentSection[]
		 * @throws Exception
		 */
		public static function savePageContentSections(Page $page, array $pageContentSections): array{
			// First, check if the page layout has a PageContentSectionDefinition
			$sectionDefinitionFromPageLayout = self::getPageLayoutSectionsDefinitionFromLayoutFileName($page->pageLayout);
			/** @var PageContentSection[] $returnObjects */
			$returnObjects = [];
			if ($sectionDefinitionFromPageLayout !== null){
				// Save the array of sections, some Ids may be null and need to be created
				/** @var array{id: ?int, sectionName: string, content: string} $pageContentSectionDto */
				foreach ($pageContentSections as $pageContentSectionDto){
					if ($pageContentSectionDto["id"] === null){
						// Create it
						$newPageContentSection = new PageContentSection();
						$newPageContentSection->pageId = $page->id;
						$newPageContentSection->sectionName = $pageContentSectionDto["sectionName"];
						$newPageContentSection->content = $pageContentSectionDto["content"];
						$newPageContentSection->save();
						$returnObjects[] = $newPageContentSection;
					}else{
						// Fetch the existing object and save it
						/** @var ?PageContentSection $existingContentSection */
						$existingContentSection = PageContentSection::fetch($pageContentSectionDto["id"]);
						if ($existingContentSection === null){
							throw new \InvalidArgumentException("Tried to save a page content section with Id {$pageContentSectionDto['id']}, but that Id does not exist in the database. You may need to copy-paste your changes and reload the page as the sections may have been changed in another browser tab.");
						}

						// Save the new content
						$existingContentSection->content = $pageContentSectionDto["content"];
						$existingContentSection->save();

						$returnObjects[] = $existingContentSection;
					}
				}
			}

			return $returnObjects;
		}
	}