<?php
	namespace PageEditor;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use ArticleCategory\ArticleCategoryService;
	use ArticleCategory\CategoryDoesNotExist;
	use ArticleCategory\CategoryNameIsBlank;
	use ArticleCategory\CategoryWithSameNameExists;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\Http\Rewrite;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Nox\ORM\ResultOrder;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use ObjectUtils\ObjectDehydrator;
	use Page\ArticleCategory;
	use Page\Page;
	use Page\PageAttribute\PageAttributePageValue;
	use Page\PageAttribute\PageAttributeQueryResponse;
	use Page\PageAttribute\PageAttributeService;
	use Page\PageBreadcrumb;
	use Page\PageData;
	use Page\PageService;
	use Page\PageType;
	use PageArchives\PageArchivesService;
	use PageEditor\Exceptions\NoPageFoundWithID;
	use ProjectPostTag\ProjectPostTag;
	use ProjectPostTag\ProjectPostTagService;
	use ProjectPostTag\ProjectTagDoesntExist;
	use ProjectPostTag\ProjectTagLabelEmpty;
	use ProjectPostTag\ProjectTagWithSameNameExists;
	use Roles\PermissionCategories;
	use Settings\Setting;
	use System\HttpHelper;
	use System\Layouts;
	use Uplift\Exceptions\IncompatiblePageType;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\Exceptions\NoObjectFound;
	use Uplift\Exceptions\ObjectAlreadyExists;
	use ValueError;

	#[Controller]
	#[RouteBase("/uplift/page-editor")]
	class PageEditorController extends BaseController{

		#[Route("GET", "/pages")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function getListOfPages(): JSONResult{
			// Will have string indices that are page types
			$pageResults = [
				PageType::General->name => [],
				PageType::Service->name => [],
				PageType::Blog->name => [],
				PageType::City->name => [],
				PageType::Project->name => [],
			];

			/** @var Page[] $systemPages */
			$systemPages = Page::query(
				resultOrder: (new ResultOrder())
					->by("pageName", "ASC"),
			);

			foreach($systemPages as $page){
				// We remove the pageBody and pageHead to make this API endpoint significantly faster,
				// as the result service doesn't use the body or head here, and we can save tons of byte transfer
				$page->pageBody = "";
				$page->pageHead = "";
				$pageResults[$page->pageType][] = $page;
			}

			return new JSONSuccess([
				"pages"=>$pageResults,
			]);
		}

		#[Route("GET", "/")]
		#[Route("GET", "@^/(?<pageID>\d+)$@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function pageEditorView(Request $request): string|Rewrite
		{
			$pageID = (int) $request->getParameter("pageID");

			$phoneNumbers = Setting::getSettingValue("companyPhoneNumbers");
			$phoneNumbersArray = [];
			$faxNumbers = Setting::getSettingValue("companyFaxNumbers");
			$faxNumbersArray = [];

			if ($phoneNumbers !== null){
				$phoneNumbersArray = json_decode($phoneNumbers, true);
			}

			if ($faxNumbers !== null){
				$faxNumbersArray = json_decode($faxNumbers, true);
			}

			/** @var ?Page $page */
			$page = null;
			$pageCustomData = [];
			$pageBreadcrumbs = [];
			$pageProjectTags = [];
			if ($pageID !== 0){
				/** @var Page $page */
				$page = Page::fetch($pageID);

				if ($page === null){
					return new Rewrite(
						path:"/404",
						statusCode: 404,
					);
				}

				$pageCustomData = json_decode($page->customData, true);
				if ($pageCustomData === null){
					$pageCustomData = [];
				}

				$pageBreadcrumbs = PageBreadcrumb::query(
					columnQuery: (new ColumnQuery())
						->where("pageID","=", $pageID),
				);
			}

			return Renderer::renderView(
				viewFileName:"page-editor/main.php",
				viewScope:[
					"viewingType"=>PageEditorViewType::STANDARD,
					"phoneNumbers"=>$phoneNumbersArray,
					"faxNumbers"=>$faxNumbersArray,
					"page"=>$page,
					"pageCustomData"=>$pageCustomData,
					"pageBreadcrumbs"=>$pageBreadcrumbs,
					"systemLayouts"=>Layouts::getAvailableLayouts(),
					"articleCategories"=>ArticleCategory::query(),
					"projectPostTags"=>ProjectPostTag::query(),
					"pageProjectTags"=>$pageProjectTags,
				],
			);
		}

		#[Route("GET", "@^/revisions/(?<pageID>\d+)$@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::VIEW_PAGE_HISTORY)]
		public function viewAllPageRevisions(Request $request): string
		{
			$pageID = (int) $request->getParameter("pageID");

			/** @var Page $page */
			$page = Page::fetch($pageID);

			return Renderer::renderView(
				viewFileName: "revisions/main.php",
				viewScope: [
					"page"=>$page,
					"revisions"=>PageEditorService::getAllDehydratedPageRevisions($pageID),
				],
			);
		}

		#[Route("GET", "@^/revisions/(?<pageID>\d+)/(?<revisionID>\d+)$@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::VIEW_PAGE_HISTORY)]
		public function pageRevisionView(): string
		{
			return Renderer::renderView(
				viewFileName:"page-editor/main.php",
				viewScope:[
                    "viewingType"=>PageEditorViewType::REVISION,
                    "phoneNumbers"=>[],
                    "faxNumbers"=>[],
                    "page"=>null,
                    "pageCustomData"=>null,
                    "pageBreadcrumbs"=>null,
                    "systemLayouts"=>null,
                    "articleCategories"=>null,
                    "projectPostTags"=>null,
                    "pageProjectTags"=>null,
				],
			);
		}

		#[Route("get", "/search-pages")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function searchPages(Request $request): JSONResult{
			$query = $request->getQueryValue("query");

			if ($query === null){
				http_response_code(500);
				return new JSONError("Missing query parameter.");
			}

			try {
				$pages = PageService::queryPagesByPageName($query);
			}catch(MalformedValue $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"pages"=>ObjectDehydrator::getDehydratedObjects($pages, ["pageName", "pageRoute"])
			]);
		}

		#[Route("post", "/page")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		#[ProcessRequestBody]
		public function newPage(Request $request): JSONResult{
			$account = Account::getCurrentUser();
			$payload = $request->getPayload();

			try{
				$newPageName = $payload->getTextPayload("page-name");
				$newPageType = $payload->getTextPayload("page-type");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newPage = PageEditorService::createPage(
					pageName: $newPageName->contents,
					pageType: $newPageType->contents,
				);

				// Archive the page
				match($newPageType->contents){
					PageType::General->name => PageArchivesService::archiveGeneralPage($newPage),
					PageType::Service->name => PageArchivesService::archiveServicePage($newPage),
					PageType::City->name => PageArchivesService::archiveCityPage($newPage),
					PageType::Blog->name => PageArchivesService::archiveBlogPage($newPage),
					PageType::Project->name => PageArchivesService::archiveProjectPage($newPage),
				};

			} catch (Exceptions\PageNameExists | \ValueError $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::PAGE_CREATION->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"pageID"=>$newPage->id,
					"pageName"=>$newPage->pageName,
				]),
			);

			return new JSONSuccess([
				"newPage"=>$newPage,
			]);
		}

		#[Route("get", "@^/page/(?<pageID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function getPage(Request $request): JSONResult{
			$pageID = (int) $request->getParameter("pageID");
			/** @var Page $page */
			$page = Page::fetch($pageID);

			if ($page === null){
				return new JSONError("No page with ID {$pageID}");
			}

			// Fetch the page's custom set data
			/** @var PageData[] $pageDatas */
			$pageDatas = PageData::query(
				columnQuery: (new ColumnQuery())
					->where("page_id","=",$pageID),
			);

			/** @var PageBreadcrumb[] $breadcrumbs */
			$breadcrumbs = PageBreadcrumb::query(
				columnQuery: (new ColumnQuery())
					->where("pageID","=",$pageID),
				resultOrder:(new ResultOrder())
					->by("position", "ASC"),
			);

			/** @var PageAttributePageValue[] $attributes */
			$attributes = PageAttributePageValue::query(
				columnQuery: (new ColumnQuery())
					->where("page_id","=", $pageID),
			);

			$attributeResponses = [];

			foreach($attributes as $pageAttributeValue){
				$attributeResponses[] = PageAttributeResponse::fromPageAttributePageValue($pageAttributeValue);
			}

			$pageDataKeyValue = [];
			foreach($pageDatas as $pageData){
				if (!array_key_exists(key: $pageData->name, array: $pageDataKeyValue)) {
					$pageDataKeyValue[$pageData->name] = $pageData->value;
				}else{
					// Convert it to an array and append
					// Multiple page datas with the same key name
					if (is_array($pageDataKeyValue[$pageData->name])){
						$pageDataKeyValue[$pageData->name][] = $pageData->value;
					}else{
						$pageDataKeyValue[$pageData->name] = [$pageDataKeyValue[$pageData->name], $pageData->value];
					}
				}
			}

			return new JSONSuccess([
				"page"=>$page,
				"breadcrumbs"=>$breadcrumbs,
				"data"=>$pageDataKeyValue,
				"attributes"=>$attributeResponses,
				"pageContentSections"=>$page->getAllContentSections(),
			]);
		}

		/**
		 * Gets the PageLayoutSectionsDefinition object for the provided page layout file name, if there is one.
		 * Accepts layoutFileName as a file name relative to the current theme's layout path but with no file extension.
		 *
		 * E.g. layoutFileName could be "General"
		 */
		#[Route("GET", "/page-layout-sections")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function getPageLayoutSectionInformation(Request $request): JSONResult{
			$fileName = $request->getQueryValue("layoutFileName") ?? "";
			try{
				$sectionDefinition = PageEditorService::getPageLayoutSectionsDefinitionFromLayoutFileName($fileName);
				return new JSONSuccess([
					"definition"=>$sectionDefinition
				]);
			}catch(\Exception $ex){
				return new JSONError($ex->getMessage());
			}
		}

		#[Route("get", "@^/page/(?<pageID>\d+)/revision/(?<revisionID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::VIEW_PAGE_HISTORY)]
		public function getPageRevisionArchive(Request $request): JSONResult{
			$revisionID = (int) $request->getParameter("revisionID");

			return new JSONSuccess(PageEditorService::getAllPageRevisionArchives($revisionID));
		}

		/**
		 * @throws IncompatiblePageType
		 */
		#[Route("post", "@^/revisions/(?<pageID>\d+)/(?<revisionID>\d+)/restore$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::VIEW_PAGE_HISTORY)]
		public function restorePageRevision(Request $request): JSONResult{
			$revisionID = (int) $request->getParameter("revisionID");

			try {
				PageEditorService::restoreRevision($revisionID);
			} catch (NoPrimaryKey $e) {
				return new JSONError($e->getMessage());
			} catch (Exceptions\PageNameExistsWithSamePageType|Exceptions\PageRouteInUse $e) {
				return new JSONError("Cannot restore revision. Revision conflicts with an existing, different page. Error: {$e->getMessage()}");
			} catch (Exceptions\PageLayoutDoesntExist|Exceptions\PageLayoutIsBlank $e) {
				return new JSONError("Revision has a layout error: {$e->getMessage()}");
			} catch (Exceptions\PageRouteIsBlank|Exceptions\PageNameIsBlank $e) {
				return new JSONError("Revision has a blank value error: {$e->getMessage()}");
			} catch (Exceptions\PublicationStatusDoesntExist $e) {
				return new JSONError("Publication status data error in revision: {$e->getMessage()}");
			}

			return new JSONSuccess([]);
		}

		/**
		 * This is the page-saving route for saving a page from the editor
		 * @param Request $request
		 * @return JSONResult
		 * @throws NoPrimaryKey
		 */
		#[Route("patch", "@^/page/(?<pageID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function savePage(Request $request): JSONResult{
			$account = Account::getCurrentUser();
			$pageID = (int) $request->getParameter("pageID");
			$payload = $request->getPayload();

			try{
				$pageLayout = $payload->getTextPayload("page-layout");
				$pageName = $payload->getTextPayload("page-name");
				$pageHead = $payload->getTextPayload("page-head");
				$pageBody = $payload->getTextPayload("page-body");
				$pageRoute = $payload->getTextPayload("page-url");
				$publicationStatus = $payload->getTextPayload("page-publication-status");
				$publicationTimestamp = $payload->getTextPayload("page-publication-timestamp");
				$breadcrumbsJSON = $payload->getTextPayload("breadcrumbs-json");
				$pageAttributesJSON = $payload->getTextPayload("page-attribute-values");
				$pageContentSectionsJSON = $payload->getTextPayload("page-content-sections");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			$breadcrumbs = json_decode($breadcrumbsJSON->contents, true);
			$pageAttributeValues = json_decode($pageAttributesJSON->contents, true);
			$pageContentSections = json_decode($pageContentSectionsJSON->contents, true);
			$excludeFromSitemap = $payload->getTextPayloadNullable("exclude-from-sitemap") !== null;
			$excludeFromSchema = $payload->getTextPayloadNullable("exclude-json-schema") !== null;
			$pageRouteIsRegex = $payload->getTextPayloadNullable("page-route-is-regex") !== null;

			try {
				$page = PageEditorService::getPageOrThrow($pageID);
			}catch(NoPageFoundWithID $e){
				return new JSONError($e->getMessage());
			}

			// To return back from the API
			$savedPageContentSections = null;

			// Check the page type and fetch, then save, the necessary payloads for that type
			if ($page->pageType === PageType::Blog->name){

				try {
					$featuredImageURI = $payload->getTextPayload("featured-image");
					$featuredImageThumbURI = $payload->getTextPayload("featured-image-thumb");
					$articleCategoryIDsJSON = $payload->getTextPayload("article-categories");
				}catch(NoPayloadFound $e){
					return new JSONError($e->getMessage());
				}

				$articleCategoryIDs = json_decode($articleCategoryIDsJSON->contents, true);

				try {
					PageEditorService::saveBlogPage(
						page: $page,
						pageName: $pageName->contents,
						pageRoute: $pageRoute->contents,
						pageRouteIsRegex: $pageRouteIsRegex,
						pageLayout: $pageLayout->contents,
						pageBody: $pageBody->contents,
						pageHead: $pageHead->contents,
						excludedFromSitemap: $excludeFromSitemap,
						excludedFromSchema: $excludeFromSchema,
						publicationStatus: (int) $publicationStatus->contents,
						publicationTimestamp: (int) $publicationTimestamp->contents,
						featuredImageURI: $featuredImageURI->contents,
						featuredImageThumbURI: $featuredImageThumbURI->contents,
						articleCategoryIDs: $articleCategoryIDs,
					);
					PageEditorService::saveBreadcrumbs(
						page: $page,
						breadcrumbs:$breadcrumbs,
					);
					PageEditorService::savePageAttributeValues($pageAttributeValues);
					$savedPageContentSections = PageEditorService::savePageContentSections($page, $pageContentSections);
					PageArchivesService::archiveBlogPage($page);
				} catch (Exceptions\PageLayoutDoesntExist|Exceptions\PageLayoutIsBlank|Exceptions\PageNameExistsWithSamePageType|Exceptions\PageNameIsBlank|Exceptions\PageRouteInUse|Exceptions\PageRouteIsBlank|Exceptions\PublicationStatusDoesntExist|IncompatiblePageType|ValueError $e) {
					return new JSONError($e->getMessage());
				}
			}elseif ($page->pageType === PageType::General->name) {
				try {
					PageEditorService::saveGeneralPage(
						page: $page,
						pageName: $pageName->contents,
						pageRoute: $pageRoute->contents,
						pageRouteIsRegex: $pageRouteIsRegex,
						pageLayout: $pageLayout->contents,
						pageBody: $pageBody->contents,
						pageHead: $pageHead->contents,
						excludedFromSitemap: $excludeFromSitemap,
						excludedFromSchema: $excludeFromSchema,
						publicationStatus: (int) $publicationStatus->contents,
						publicationTimestamp: (int) $publicationTimestamp->contents,
					);
					PageEditorService::saveBreadcrumbs(
						page: $page,
						breadcrumbs:$breadcrumbs,
					);
					PageEditorService::savePageAttributeValues($pageAttributeValues);
					$savedPageContentSections = PageEditorService::savePageContentSections($page, $pageContentSections);
					PageArchivesService::archiveGeneralPage($page);
				} catch (Exceptions\PageLayoutDoesntExist|Exceptions\PageLayoutIsBlank|Exceptions\PageNameExistsWithSamePageType|Exceptions\PageNameIsBlank|Exceptions\PageRouteInUse|Exceptions\PageRouteIsBlank|Exceptions\PublicationStatusDoesntExist|MalformedValue|IncompatiblePageType|ValueError $e) {
					return new JSONError($e->getMessage());
				}
			}elseif ($page->pageType === PageType::Service->name){

				try {
					$featuredImageURI = $payload->getTextPayload("featured-image");
					$featuredImageThumbURI = $payload->getTextPayload("featured-image-thumb");
				}catch(NoPayloadFound $e){
					return new JSONError($e->getMessage());
				}

				try {
					PageEditorService::saveServicePage(
						page: $page,
						pageName: $pageName->contents,
						pageRoute: $pageRoute->contents,
						pageRouteIsRegex: $pageRouteIsRegex,
						pageLayout: $pageLayout->contents,
						pageBody: $pageBody->contents,
						pageHead: $pageHead->contents,
						excludedFromSitemap: $excludeFromSitemap,
						excludedFromSchema: $excludeFromSchema,
						publicationStatus: (int) $publicationStatus->contents,
						publicationTimestamp: (int) $publicationTimestamp->contents,
						featuredImageURI:$featuredImageURI->contents,
						featuredImageThumbURI:$featuredImageThumbURI->contents,
					);
					PageEditorService::saveBreadcrumbs(
						page: $page,
						breadcrumbs:$breadcrumbs,
					);
					PageEditorService::savePageAttributeValues($pageAttributeValues);
					$savedPageContentSections = PageEditorService::savePageContentSections($page, $pageContentSections);
					PageArchivesService::archiveServicePage($page);
				} catch (Exceptions\PageLayoutDoesntExist|Exceptions\PageLayoutIsBlank|Exceptions\PageNameExistsWithSamePageType|Exceptions\PageNameIsBlank|Exceptions\PageRouteInUse|Exceptions\PageRouteIsBlank|Exceptions\PublicationStatusDoesntExist|MalformedValue|IncompatiblePageType|ValueError $e) {
					return new JSONError($e->getMessage());
				}
			}elseif ($page->pageType === PageType::Project->name){

				try {
					$cityName = $payload->getTextPayload("city-name");
					$stateName = $payload->getTextPayload("state-name");
					$stateNameShortHand = $payload->getTextPayload("state-name-shorthand");
					$brandsProducts = $payload->getTextPayload("brands-products");
					$customerDidLeaveReview = $payload->getTextPayloadNullable("customer-testimonial-check") !== null;
					$projectTagIDsJSON = $payload->getTextPayload("project-tag-ids");
					$customerReviewFirstName = $payload->getTextPayload("customer-review-first-name");
					$customerReviewLastName = $payload->getTextPayload("customer-review-last-name");
					$customerReviewTestimonial = $payload->getTextPayload("customer-testimonial-body");
					$featuredImageURI = $payload->getTextPayload("featured-image");
					$featuredImageThumbURI = $payload->getTextPayload("featured-image-thumb");
				}catch(NoPayloadFound $e){
					return new JSONError($e->getMessage());
				}

				$projectTagIDs = json_decode($projectTagIDsJSON->contents, true);

				try {
					PageEditorService::saveProjectPage(
						page: $page,
						pageName: $pageName->contents,
						pageRoute: $pageRoute->contents,
						pageRouteIsRegex: $pageRouteIsRegex,
						pageLayout: $pageLayout->contents,
						pageBody: $pageBody->contents,
						pageHead: $pageHead->contents,
						excludedFromSitemap: $excludeFromSitemap,
						excludedFromSchema: $excludeFromSchema,
						publicationStatus: (int) $publicationStatus->contents,
						publicationTimestamp: (int) $publicationTimestamp->contents,
						featuredImageURI:$featuredImageURI->contents,
						featuredImageThumbURI:$featuredImageThumbURI->contents,
						cityName: $cityName->contents,
						stateName:$stateName->contents,
						stateNameShorthand:$stateNameShortHand->contents,
						brandsProducts:$brandsProducts->contents,
						customerDidLeaveReview:$customerDidLeaveReview,
						customerReviewFirstName:$customerReviewFirstName->contents,
						customerReviewLastName:$customerReviewLastName->contents,
						customerReviewTestimonial:$customerReviewTestimonial->contents,
						projectTagIDs:$projectTagIDs,
					);
					PageEditorService::saveBreadcrumbs(
						page: $page,
						breadcrumbs:$breadcrumbs,
					);
					PageEditorService::savePageAttributeValues($pageAttributeValues);
					$savedPageContentSections = PageEditorService::savePageContentSections($page, $pageContentSections);
					PageArchivesService::archiveProjectPage($page);
				} catch (Exceptions\PageLayoutDoesntExist|Exceptions\PageLayoutIsBlank|Exceptions\PageNameExistsWithSamePageType|Exceptions\PageNameIsBlank|Exceptions\PageRouteInUse|Exceptions\PageRouteIsBlank|Exceptions\PublicationStatusDoesntExist|MalformedValue|IncompatiblePageType|ValueError $e) {
					return new JSONError($e->getMessage());
				}
			}elseif ($page->pageType === PageType::City->name){
				try{
					$cityName = $payload->getTextPayload('city-page-city-name');
					$stateName = $payload->getTextPayload('city-page-state-name');
					$stateNameShortHand = $payload->getTextPayload('city-page-state-name-shorthand');
					$officialCityURL = $payload->getTextPayload('city-page-city-url');
					$featuredImageURI = $payload->getTextPayload("featured-image");
					$featuredImageThumbURI = $payload->getTextPayload("featured-image-thumb");
				}catch(NoPayloadFound $e){
					return new JSONError($e->getMessage());
				}

				try {
					PageEditorService::saveCityPage(
						page: $page,
						pageName: $pageName->contents,
						pageRoute: $pageRoute->contents,
						pageRouteIsRegex: $pageRouteIsRegex,
						pageLayout: $pageLayout->contents,
						pageBody: $pageBody->contents,
						pageHead: $pageHead->contents,
						excludedFromSitemap: $excludeFromSitemap,
						excludedFromSchema: $excludeFromSchema,
						publicationStatus: (int) $publicationStatus->contents,
						publicationTimestamp: (int) $publicationTimestamp->contents,
						cityName: $cityName->contents,
						stateName: $stateName->contents,
						stateNameShorthand: $stateNameShortHand->contents,
						officialCityURL: $officialCityURL->contents,
						featuredImageURI: $featuredImageURI->contents,
						featuredImageThumbURI: $featuredImageThumbURI->contents,
					);
					PageEditorService::saveBreadcrumbs(
						page: $page,
						breadcrumbs:$breadcrumbs,
					);
					$savedPageContentSections = PageEditorService::savePageContentSections($page, $pageContentSections);
					PageArchivesService::archiveCityPage($page);
				} catch (Exceptions\PageLayoutDoesntExist|Exceptions\PageLayoutIsBlank|Exceptions\PageNameExistsWithSamePageType|Exceptions\PageNameIsBlank|Exceptions\PageRouteInUse|Exceptions\PageRouteIsBlank|Exceptions\PublicationStatusDoesntExist|MalformedValue|IncompatiblePageType|ValueError $e) {
					return new JSONError($e->getMessage());
				}
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_PAGE->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"pageID"=>$page->id,
					"pageName"=>$page->pageName,
				]),
			);

			return new JSONSuccess([
				"updatedPageContentSections"=>$savedPageContentSections
			]);
		}

		#[Route("patch", "/^\/page\/(?<pageID>\d+)\/type$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function convertPageType(Request $request): JSONResult{
			$payload = $request->getPayload();

			$pageID = (int) $request->getParameter("pageID");

			try{
				$newPageType = $payload->getTextPayload("new-page-type");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				PageEditorService::convertPage(
					pageID: $pageID,
					newPageType: $newPageType->contents,
				);
			} catch (NoPageFoundWithID|Exceptions\PageTypeNotFound $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess();
		}

		#[Route("post", "/^\/page\/(?<pageID>\d+)\/clone$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function clonePage(Request $request): JSONResult{
			$payload = $request->getPayload();
			$pageID = (int) $request->getParameter("pageID");

			try {
				$newPageName = $payload->getTextPayload("cloned-page-name");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$clonedPage = PageEditorService::clonePage(
					pageID: $pageID,
					newPageName: $newPageName->contents,
				);
			} catch (NoPageFoundWithID|Exceptions\PageNameExistsWithSamePageType|Exceptions\PageNameIsBlank $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"clonedPageID"=>$clonedPage->id,
			]);
		}

		#[Route("delete", "/^\/page\/(?<pageID>\d+)$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function deletePage(Request $request): JSONResult{
			$pageID = (int) $request->getParameter("pageID");

			try{
				PageEditorService::deletePage($pageID);
			} catch (NoPrimaryKey|NoPageFoundWithID $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess();
		}

		#[Route("post", "/article-category")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function createArticleCategory(Request $request): JSONResult{
			$payload = $request->getPayload();

			try {
				$categoryName = $payload->getTextPayload("new-article-category-name");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try{
				$newCategory = ArticleCategoryService::createCategory($categoryName->contents);
			} catch (CategoryNameIsBlank | CategoryWithSameNameExists $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"newCategoryID"=>$newCategory->id,
				"newCategoryName"=>$newCategory->name,
			]);
		}

		#[Route("patch", "/^\/article-category\/(?<articleCategoryID>\d+)$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function renameArticleCategory(Request $request): JSONResult{
			$account = Account::getCurrentUser();
			$articleCategoryID = (int) $request->getParameter('articleCategoryID');
			$payload = $request->getPayload();

			try {
				$newCategoryName = $payload->getTextPayload('new-category-name');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			/** @var ArticleCategory | null $thisCategory */
			$thisCategory = ArticleCategory::fetch(
				primaryKey: $articleCategoryID,
			);

			$oldCategoryName = $thisCategory->name;

			if ($thisCategory === null){
				return new JSONError("No category found with ID $articleCategoryID.");
			}

			try {
				ArticleCategoryService::renameCategory(
					articleCategory: $thisCategory,
					newName: $newCategoryName->contents,
				);
			} catch (CategoryNameIsBlank | CategoryWithSameNameExists $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::RENAME_ARTICLE_CATEGORY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"categoryID"=>$thisCategory->id,
					"oldName"=>$oldCategoryName,
					"newName"=>$newCategoryName->contents,
				]),
			);

			return new JSONSuccess([
				"newCategoryName"=>$newCategoryName->contents,
			]);
		}

		#[Route("delete", "/^\/article-category\/(?<articleCategoryID>\d+)$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function deleteArticleCategory(Request $request): JSONResult
		{
			$articleCategoryID = (int) $request->getParameter('articleCategoryID');

			try {
				ArticleCategoryService::deleteCategory($articleCategoryID);
			} catch (CategoryDoesNotExist $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess();
		}

		#[Route("post", "/project-tag")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function createProjectTag(Request $request): JSONResult
		{
			$account = Account::getCurrentUser();
			$payload = $request->getPayload();

			try {
				$newTagLabel = $payload->getTextPayload("new-project-tag-label");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}


			try {
				$newTag = ProjectPostTagService::createTag(
					newTagLabel: $newTagLabel->contents,
				);
			} catch (ProjectTagLabelEmpty | ProjectTagWithSameNameExists $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CREATE_PROJECT_POST_TAG->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"newProjectPostTagID" => $newTag->id,
					"newProjectPostTagLabel" => $newTag->label,
				]),
			);

			return new JSONSuccess([
				"projectTagID" => $newTag->id,
				"projectTagLabel" => $newTag->label,
			]);
		}

		#[Route("delete", "/^\/project-tag\/(?<projectTagID>\d+)$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function deleteProjectTag(Request $request): JSONResult
		{
			$projectTagID = (int) $request->getParameter('projectTagID');

			try {
				ProjectPostTagService::deleteTag($projectTagID);
			} catch (ProjectTagDoesntExist $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess();
		}

		#[Route("patch", "/^\/project-tag\/(?<projectTagID>\d+)$/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function renameProjectTag(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$projectTagID = (int) $request->getParameter('projectTagID');
			try {
				$newLabel = $payload->getTextPayload('renamed-project-tag-label');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$projectTag = ProjectPostTagService::renameTag(
					tagID:$projectTagID,
					newLabel:$newLabel->contents,
				);
			} catch (ProjectTagDoesntExist|ProjectTagLabelEmpty|ProjectTagWithSameNameExists $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"newTagLabel"=>$projectTag->label,
			]);
		}

		#[Route("get", "/article-categories")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function getArticleCategories(): JSONResult
		{
			return new JSONSuccess([
				"categories"=>ArticleCategory::query(),
			]);
		}

		#[Route("get", "/project-tags")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function getProjectTags(): JSONResult
		{
			return new JSONSuccess([
				"tags"=>ProjectPostTag::query(),
			]);
		}

		#[Route("get", "/\/page-attributes\/(?<pageID>\d+)/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function getPageAttributesAvailableForPage(Request $request): JSONResult
		{
			$pageID = $request->getParameter("pageID");

			/** @var ?Page $page */
			$page = Page::fetch($pageID);

			if ($page === null){
				http_response_code(400);
				return new JSONError("No page found with Page ID $pageID.");
			}

			$pageType = PageType::fromName($page->pageType);
			$pageAttributesAvailable = PageAttributeService::getAvailableAttributesForPageType($pageType);

			return new JSONSuccess([
				"availableAttributes"=>$pageAttributesAvailable,
			]);
		}

		#[Route("put", "/\/page-attributes\/(?<pageID>\d+)/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function addAttributeToPage(Request $request): JSONResult
		{
			$pageID = $request->getParameter("pageID");
			$payload = $request->getPayload();

			try{
				$attributeID = $payload->getTextPayload("attributeID");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			try {
				$newAttributeValue = PageAttributeService::addAttributeToPage($attributeID->contents, $pageID);
				$queryResponseValue = PageAttributeResponse::fromPageAttributePageValue($newAttributeValue);
			} catch (ObjectAlreadyExists $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"newPageAttribute"=>$queryResponseValue,
			]);
		}

		#[Route("DELETE", "/\/attributes\/page-attribute-value\/(?<pageAttributeValueID>\d+)/", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::EDIT_PAGES)]
		public function deletePageAttributeValueFromPage(Request $request): JSONResult
		{
			$pageAttributeValueID = $request->getParameter("pageAttributeValueID");

			try {
				PageAttributeService::deletePageAttributeValue($pageAttributeValueID);
			} catch (NoPrimaryKey|NoObjectFound $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([]);
		}
	}