<?php
	namespace PageArchives;

	use Nox\ORM\ColumnQuery;
	use Page\Page;
	use Page\PageBreadcrumb;
	use Page\PageContentSection;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Uplift\Exceptions\IncompatiblePageType;

	class PageArchivesService{

		private static function archiveStandardPageProperties(Page $page): PageArchive{
			$archive = new PageArchive();
			$archive->pageID = $page->id;
			$archive->pageName = $page->pageName;
			$archive->pageType = $page->pageType;
			$archive->pageRoute = $page->pageRoute;
			$archive->pageRouteIsRegex = $page->pageRouteIsRegex;
			$archive->pageLayout = $page->pageLayout;
			$archive->pageHead = $page->pageHead;
			$archive->pageBody = $page->pageBody;
			$archive->creationTime = $page->creationTime;
			$archive->excludeFromSitemap = $page->excludeFromSitemap;
			$archive->publicationStatus = $page->publicationStatus;
			$archive->publicationTimestamp = $page->publicationTimestamp;

			$archive->save();

			return $archive;
		}

		private static function archivePageBreadcrumbs(PageArchive $pageArchive, array $crumbs): array{
			$archives = [];

			foreach($crumbs as $crumb){
				$breadcrumbArchive = new PageBreadrumbArchive();
				$breadcrumbArchive->pageArchiveID = $pageArchive->id;
				$breadcrumbArchive->pageID = $crumb->pageID;
				$breadcrumbArchive->uri = $crumb->uri;
				$breadcrumbArchive->position = $crumb->position;
				$breadcrumbArchive->label = $crumb->label;
				$breadcrumbArchive->save();
				$archives[] = $breadcrumbArchive;
			}

			return $archives;
		}

		private static function archivePageData(PageArchive $pageArchive, PageData $data): PageDataArchive{
			$archive = new PageDataArchive();
			$archive->pageArchiveID = $pageArchive->id;
			$archive->name = $data->name;
			$archive->value = $data->value;
			$archive->pageID = $pageArchive->pageID;
			$archive->save();

			return $archive;
		}

		/**
		 * Archives all page content sections belonging to a page
		 * @param PageArchive $pageArchive
		 * @param int $pageId
		 * @return PageContentSectionArchive[]
		 */
		private static function archiveAllPageSections(PageArchive $pageArchive, int $pageId): array{
			$pageContentSectionArchives = [];
			// If this page has any content sections associated with it, archive them
			// Do not check if the current layout does - always archive existing page content sections
			/** @var PageContentSection[] $contentSections */
			$contentSections = PageContentSection::query(
				columnQuery: (new ColumnQuery())
					->where("page_id", "=", $pageId)
			);

			foreach ($contentSections as $section){
				// Create an archive for it
				$archiveSection = new PageContentSectionArchive();
				$archiveSection->archiveId = $pageArchive->id;
				$archiveSection->pageId = $pageId;
				$archiveSection->content = $section->content;
				$archiveSection->sectionName = $section->sectionName;
				$archiveSection->save();
				$pageContentSectionArchives[] = $archiveSection;
			}

			return $pageContentSectionArchives;
		}

		/**
		 * @throws IncompatiblePageType
		 */
		public static function archiveGeneralPage(Page $page): PageArchive{
			if ($page->pageType !== PageType::General->name){
				throw new IncompatiblePageType("Wrong page type for archive function.");
			}

			$archive = self::archiveStandardPageProperties($page);
			self::archiveAllPageSections($archive, $page->id);
			self::archivePageBreadcrumbs($archive, $page->getBreadcrumbs());
			return $archive;
		}

		/**
		 * @throws IncompatiblePageType
		 */
		public static function archiveServicePage(Page $page): PageArchive{
			if ($page->pageType !== PageType::Service->name){
				throw new IncompatiblePageType("Wrong page type for archive function.");
			}

			$archive = self::archiveStandardPageProperties($page);
			self::archiveAllPageSections($archive, $page->id);
			self::archivePageBreadcrumbs($archive, $page->getBreadcrumbs());

			$featuredImages = $page->getPageDatas(PageDatas::FEATURED_IMAGE);
			$featuredImageThumbs = $page->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);

			if (!empty($featuredImages)){
				self::archivePageData($archive, $featuredImages[0]);
			}

			if (!empty($featuredImageThumbs)){
				self::archivePageData($archive, $featuredImageThumbs[0]);
			}

			return $archive;
		}

		/**
		 * @throws IncompatiblePageType
		 */
		public static function archiveBlogPage(Page $page): PageArchive{
			if ($page->pageType !== PageType::Blog->name){
				throw new IncompatiblePageType("Wrong page type for archive function.");
			}

			$pageArchive = self::archiveStandardPageProperties($page);
			self::archiveAllPageSections($pageArchive, $page->id);
			self::archivePageBreadcrumbs($pageArchive, $page->getBreadcrumbs());

			// Archive blog data
			$categories = $page->getPageDatas(PageDatas::ARTICLE_CATEGORY);
			$featuredImages = $page->getPageDatas(PageDatas::FEATURED_IMAGE);
			$featuredImageThumbs = $page->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);

			foreach($categories as $pageDataCategory) {
				self::archivePageData($pageArchive, $pageDataCategory);
			}

			if (!empty($featuredImages)){
				self::archivePageData($pageArchive, $featuredImages[0]);
			}

			if (!empty($featuredImageThumbs)){
				self::archivePageData($pageArchive, $featuredImageThumbs[0]);
			}

			return $pageArchive;
		}

		/**
		 * @throws IncompatiblePageType
		 */
		public static function archiveCityPage(Page $page): PageArchive{
			if ($page->pageType !== PageType::City->name){
				throw new IncompatiblePageType("Wrong page type for archive function.");
			}

			$pageArchive = self::archiveStandardPageProperties($page);
			self::archiveAllPageSections($pageArchive, $page->id);
			self::archivePageBreadcrumbs($pageArchive, $page->getBreadcrumbs());

			// Archive blog data
			$cityNames = $page->getPageDatas(PageDatas::CITY_NAME);
			$stateNames = $page->getPageDatas(PageDatas::STATE_NAME);
			$stateNamesShorthand = $page->getPageDatas(PageDatas::STATE_NAME_SHORTHAND);
			$cityURLs = $page->getPageDatas(PageDatas::OFFICIAL_CITY_URL);
			$featuredImages = $page->getPageDatas(PageDatas::FEATURED_IMAGE);
			$featuredImageThumbs = $page->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);

			if (!empty($cityNames)){
				self::archivePageData($pageArchive, $cityNames[0]);
			}

			if (!empty($stateNames)){
				self::archivePageData($pageArchive, $stateNames[0]);
			}

			if (!empty($stateNamesShorthand)){
				self::archivePageData($pageArchive, $stateNamesShorthand[0]);
			}

			if (!empty($cityURLs)){
				self::archivePageData($pageArchive, $cityURLs[0]);
			}

			if (!empty($featuredImages)){
				self::archivePageData($pageArchive, $featuredImages[0]);
			}

			if (!empty($featuredImageThumbs)){
				self::archivePageData($pageArchive, $featuredImageThumbs[0]);
			}

			return $pageArchive;
		}

		/**
		 * @throws IncompatiblePageType
		 */
		public static function archiveProjectPage(Page $page): PageArchive{
			if ($page->pageType !== PageType::Project->name){
				throw new IncompatiblePageType("Wrong page type for archive function.");
			}

			$pageArchive = self::archiveStandardPageProperties($page);
			self::archiveAllPageSections($pageArchive, $page->id);
			self::archivePageBreadcrumbs($pageArchive, $page->getBreadcrumbs());

			// Archive project data
			$projectTags = $page->getPageDatas(PageDatas::PROJECT_TAG);
			foreach($projectTags as $projectTag){
				self::archivePageData($pageArchive, $projectTag);
			}

			$cityNames = $page->getPageDatas(PageDatas::CITY_NAME);
			$stateNames = $page->getPageDatas(PageDatas::STATE_NAME);
			$stateNamesShorthand = $page->getPageDatas(PageDatas::STATE_NAME_SHORTHAND);
			$projectBrands = $page->getPageDatas(PageDatas::PROJECT_BRANDS_PRODUCTS);

			if (!empty($cityNames)){
				self::archivePageData($pageArchive, $cityNames[0]);
			}

			if (!empty($stateNames)){
				self::archivePageData($pageArchive, $stateNames[0]);
			}

			if (!empty($stateNamesShorthand)){
				self::archivePageData($pageArchive, $stateNamesShorthand[0]);
			}

			if (!empty($projectBrands)){
				self::archivePageData($pageArchive, $projectBrands[0]);
			}

			$didLeaveReview = $page->getPageDatas(PageDatas::CUSTOMER_DID_LEAVE_REVIEW);
			if (!empty($didLeaveReview)){
				$didLeaveReview = $didLeaveReview[0];
				if ($didLeaveReview->value === "1"){
					$customerFirstNames = $page->getPageDatas(PageDatas::CUSTOMER_REVIEW_FIRST_NAME);
					$customerLastNames = $page->getPageDatas(PageDatas::CUSTOMER_REVIEW_LAST_NAME);
					$customerTestimonials = $page->getPageDatas(PageDatas::CUSTOMER_REVIEW_TESTIMONIAL);

					if (!empty($customerFirstNames)){
						self::archivePageData($pageArchive, $customerFirstNames[0]);
					}

					if (!empty($customerLastNames)){
						self::archivePageData($pageArchive, $customerLastNames[0]);
					}

					if (!empty($customerTestimonials)){
						self::archivePageData($pageArchive, $customerTestimonials[0]);
					}
				}
			}

			return $pageArchive;
		}

		/**
		 * Archives a page by determining what page type it is, then calling the correct PageArchivesService method
		 * to archive that particular page type.
		 * @throws IncompatiblePageType
		 */
		public static function archivePage(Page $page): PageArchive{
			if ($page->pageType === PageType::General->name){
				return self::archiveGeneralPage($page);
			}
			elseif ($page->pageType === PageType::Service->name){
				return self::archiveServicePage($page);
			}
			elseif ($page->pageType === PageType::Blog->name){
				return self::archiveBlogPage($page);
			}
			elseif ($page->pageType === PageType::Project->name){
				return self::archiveProjectPage($page);
			}
			elseif ($page->pageType === PageType::City->name){
				return self::archiveCityPage($page);
			}
		}
	}