<?php

	namespace Page\PageAttributes;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Page\PageAttribute\PageAttributeService;
	use Roles\PermissionCategories;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\Exceptions\NoObjectFound;
	use Uplift\Exceptions\ObjectAlreadyExists;

	#[Controller]
	#[RouteBase("/uplift/page-attributes")]
	class PageAttributeController extends BaseController
	{

		#[Route("GET", "/")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_PAGE_ATTRIBUTES)]
		public function pageAttributeManagementView(Request $request): string
		{
			return Renderer::renderView(
				viewFileName: "page-attributes/main.php",
			);
		}

		#[Route("GET", "/query")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_PAGE_ATTRIBUTES)]
		public function queryAttributes(Request $request): JSONResult
		{
			$query = $request->getQueryValue("query");
			$pageNumber = $request->getQueryValue("pageNumber");

			$resultArray = PageAttributeService::query(
				query: $query,
				pageNumber: $pageNumber,
			);

			return new JSONSuccess([
				"attributes"=>$resultArray['attributes'],
				"totalPages"=>$resultArray['totalPages'],
			]);
		}

		#[Route("PUT", "/attribute")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_PAGE_ATTRIBUTES)]
		public function createAttribute(Request $request): JSONResult
		{
			$payload = $request->getPayload();

			try{
				$name = $payload->getTextPayload("name");
				$pageTypes = $payload->getArrayPayload("pageTypes");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			try {
				$pageAttribute = PageAttributeService::createAttribute(
					name: $name->contents,
					pageTypes: $pageTypes->contents,
				);
			} catch (EmptyValue|MalformedValue|ObjectAlreadyExists $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::NEW_PAGE_ATTRIBUTE->value,
				accountID: Account::getCurrentUser()->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"name"=>$name->contents,
					"id"=>$pageAttribute->id,
				]),
			);

			return new JSONSuccess([]);
		}

		#[Route("PATCH", "/\/attribute\/(?<attributeID>\d+)/", true)]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_PAGE_ATTRIBUTES)]
		public function saveAttribute(Request $request): JSONResult
		{
			$attributeID = $request->getParameter("attributeID");
			$payload = $request->getPayload();

			try{
				$name = $payload->getTextPayload("name");
				$pageTypes = $payload->getArrayPayload("pageTypes");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			try {
				$pageAttribute = PageAttributeService::saveAttribute(
					attributeID: $attributeID,
					name: $name->contents,
					pageTypes: $pageTypes->contents,
				);
			} catch (EmptyValue|MalformedValue|ObjectAlreadyExists|NoObjectFound $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_PAGE_ATTRIBUTE->value,
				accountID: Account::getCurrentUser()->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"pageTypes"=>json_encode($pageTypes->contents),
					"name"=>$name->contents,
					"id"=>$pageAttribute->id,
				]),
			);

			return new JSONSuccess([]);
		}

		#[Route("DELETE", "/\/attribute\/(?<attributeID>\d+)/", true)]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_PAGE_ATTRIBUTES)]
		public function deleteAttribute(Request $request): JSONResult
		{
			$attributeID = $request->getParameter("attributeID");
			$payload = $request->getPayload();

			try {
				PageAttributeService::deleteAttribute(
					attributeID: $attributeID,
				);
			} catch (NoObjectFound $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_PAGE_ATTRIBUTE->value,
				accountID: Account::getCurrentUser()->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"id"=>$attributeID,
				]),
			);

			return new JSONSuccess([]);
		}

	}