<?php
	namespace Page;

	use Accounts\Account;
	use DOMElement;
	use DOMException;
	use DOMNode;
	use Exception;
	use ExternalSiteImporter\ExternalSiteImporterService;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Interfaces\ModelInstance;
	use Nox\ORM\Interfaces\MySQLModelInterface;
	use Nox\ORM\ModelClass;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\PageAttribute\PageAttributePageValue;
	use PageArchives\PageArchive;
	use PageEditor\PageEditorService;
	use SectionParser\SectionParser;
	use Settings\CustomSettings;
	use Settings\CustomSettingService;
	use Settings\Setting;
	use Settings\Settings;
	use ShortcodeParser\ShortcodeParser;
	use System\HttpHelper;
	use System\Layouts;
	use System\Themes;
	use TemplateManager\PageLayouts\PageLayoutSection;
	use TemplateManager\PageLayouts\PageLayoutSectionsDefinition;
	use TemplateManager\PageLayouts\PageLayoutSectionsProvider;
	use Uplift\Exceptions\NoObjectFound;

	class Page extends ModelClass implements ModelInstance
	{

		/**
		 * The current page that was matched with the current route. This will be null if no page was found.
		 * Otherwise, the routing process in nox-request.php will set this value if a page was found.
		 */
		private static ?Page $currentPage = null;

		public ?int $id = null;
		public string $pageName;
		public string $pageType;
		public string $pageRoute;
		public int $pageRouteIsRegex;
		public string $pageLayout;
		public string $pageHead;
		public string $pageBody;
		public int $creationTime;
		public int $excludeFromSitemap;
		public int $excludeSchemaInjection;
		public string $customData;
		public int $publicationStatus;
		public int $publicationTimestamp;
		public int $isDeleted;

		public static function getModel(): MySQLModelInterface
		{
			return new PagesModel();
		}

		public function __construct()
		{
			parent::__construct($this);
		}

		public static function getCurrentPage(): Page | null{
			return self::$currentPage;
		}

		public static function setCurrentPage(Page $page): void{
			self::$currentPage = $page;
		}

		public static function fromArray(array $a): Page{
			$obj = new Page();
			$obj->id = $a["id"];
			$obj->pageName = $a["pageName"];
			$obj->pageType = $a["pageType"];
			$obj->pageRoute = $a["pageRoute"];
			$obj->pageRouteIsRegex = $a["pageRouteIsRegex"];
			$obj->pageLayout = $a["pageLayout"];
			$obj->pageHead = $a["pageHead"];
			$obj->pageBody = $a["pageBody"];
			$obj->creationTime = $a["creationTime"];
			$obj->excludeFromSitemap = $a["excludeFromSitemap"];
			$obj->excludeSchemaInjection = $a["excludeSchemaInjection"];
			$obj->customData = $a["customData"];
			$obj->publicationStatus = $a["publicationStatus"];
			$obj->publicationTimestamp = $a["publicationTimestamp"];
			$obj->isDeleted = $a["isDeleted"];

			return $obj;
		}

		/**
		 * Can return an array of PageData
		 * @param PageDatas $pageDataEnum
		 * @return PageData[]
		 */
		public function getPageDatas(PageDatas $pageDataEnum): array{
			/** @var PageData[] $pageDatas */
			$pageDatas = PageData::query(
				columnQuery: (new ColumnQuery())
					->where("page_id","=",$this->id)
					->and()
					->where("name","=",$pageDataEnum->name)
			);

			return $pageDatas;
		}

		/**
		 * Returns the first matching PageData for the page
		 */
		public function getPageData(PageDatas $pageDataEnum): ?PageData{
			/** @var ?PageData $pageData */
			$pageData = PageData::queryOne(
				columnQuery: (new ColumnQuery())
					->where("page_id","=",$this->id)
					->and()
					->where("name","=",$pageDataEnum->name)
			);

			return $pageData;
		}

		/**
		 * Gets a list of all the page content sections that this page has.
		 * @return PageContentSection[]
		 */
		public function getAllContentSections(): array{
			/** @var PageContentSection[] $sections */
			$sections = PageContentSection::query(
				columnQuery: (new ColumnQuery())
					->where("page_id", "=", $this->id)
			);

			return $sections;
		}

		/**
		 * Retrieves an individual page content section identified by the section name.
		 */
		public function getContentSectionByName(string $sectionName): ?PageContentSection{
			/** @var ?PageContentSection $section */
			$section = PageContentSection::queryOne(
				columnQuery: (new ColumnQuery())
					->where("page_id", "=", $this->id)
					->and()
					->where("section_name", "=", $sectionName)
			);

			return $section;
		}

		/**
		 * Returns an array of breadcrumbs for the page
		 * @return PageBreadcrumb[]|null
		 */
		public function getBreadcrumbs(): array | null{
			/** @var PageBreadcrumb[] $breadcrumbs */
			$breadcrumbs = PageBreadcrumb::query(
				columnQuery: (new ColumnQuery())
					->where("pageID","=",$this->id),
				resultOrder: (new ResultOrder())
					->by("position", "ASC"),
			);

			return $breadcrumbs;
		}

		/**
		 * Uses the pageHead, pageBody, and the page's layout to fetch a rendered HTML of
		 * the page
		 * @throws Exception
		 */
		public function getRenderedHTML(): string
		{
			$pageLayoutName = $this->pageLayout;
			$availableLayouts = Layouts::getAvailableLayouts();

			/**
			 * To be reverse-compatible with CMS v4, the following variables must exist
			 * string $schemaType, string $breadcrumbs, string $headContents, string $bodyContents
			 * when rendering the page upon a matching layout.
			 * Additionally, the following variables must be available for a given page type:
			 * string $cityName
			 * string $stateName
			 * string $stateNameShorthand
			 * string $cityUrl
			 * string $mapKey
			 * string $featuredImage
			 * string $featuredImageThumb
			 * string $brandProducts;
			 * string $customerFirstName;
			 * string $customerLastName;
			 * string $customerTestimonialBody;
			 * int $customerTestimonialCheck; Tiny integer. 1 = yes review and 0 = no review
			 * int $disableGeneratedMap; Deprecated, just set as 0
			 */
			foreach ($availableLayouts as $fsFileLayout) {
				if (strtolower($pageLayoutName) === strtolower($fsFileLayout->fileNameWithoutExtension)) {
					// Matching layout

					// Begin output buffering
					ob_start();
					// Create this for backwards compatibility
					// Older layout files (and maybe current ones?)

					$this->processOGTitle();
					$this->processOGImage();
					$this->processOGUrl();

					$shortcodeParser = new ShortcodeParser();
					$ShortcodeParser = $shortcodeParser;
					$headContents = $shortcodeParser->parse($this->pageHead);
					$bodyContents = $shortcodeParser->parse($this->pageBody);

					$schemaType = Setting::getSettingValue(Settings::SCHEMA_TYPE->value);
					$breadcrumbs = $shortcodeParser->parse("{{ breadcrumbs }}");
					$disableGeneratedMap = 0;
					$baseURL = HttpHelper::getWebsiteBaseURL();
					$baseUrl = $baseURL;

					// Define company variables
					$companyName = Setting::getSettingValue(Settings::COMPANY_NAME->value);
					$companyStreet = Setting::getSettingValue(Settings::COMPANY_STREET->value);
					$companyCity = Setting::getSettingValue(Settings::COMPANY_CITY->value);
					$companyState = Setting::getSettingValue(Settings::COMPANY_STATE->value);
					$companyPostal = Setting::getSettingValue(Settings::COMPANY_POSTAL->value);
					$companyLogo = Setting::getSettingValue(Settings::COMPANY_LOGO->value);
					$phoneNumbers = json_decode((string) Setting::getSettingValue(Settings::COMPANY_PHONE_NUMBERS->value)) ?? [];
					$faxNumbers = json_decode((string) Setting::getSettingValue(Settings::COMPANY_FAX_NUMBERS->value)) ?? [];
					$mapKey = \NoxEnv::GOOGLE_MAPS_API_KEY;

					// Handle page type variables that always existed in previous
					// CMS versions. For backwards compatibility.
					// Moving forward, layouts should just query the Page object directly.
					$cityNameData = $this->getPageDatas(PageDatas::CITY_NAME);
					if (!empty($cityNameData)){
						$cityName = $cityNameData[0]->value;
					}

					$stateNameData = $this->getPageDatas(PageDatas::STATE_NAME);
					if (!empty($stateNameData)){
						$stateName = $stateNameData[0]->value;
					}

					$stateNameShorthandData = $this->getPageDatas(PageDatas::STATE_NAME_SHORTHAND);
					if (!empty($stateNameShorthandData)){
						$stateNameShorthand = $stateNameShorthandData[0]->value;
					}

					$cityURLData = $this->getPageDatas(PageDatas::OFFICIAL_CITY_URL);
					if (!empty($cityURLData)){
						$cityUrl = $cityURLData[0]->value;
					}

					$featuredImageData = $this->getPageDatas(PageDatas::FEATURED_IMAGE);
					if (!empty($featuredImageData)){
						$featuredImage = $featuredImageData[0]->value;
					}

					$featuredImageThumbData = $this->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);
					if (!empty($featuredImageThumbData)){
						$featuredImageThumb = $featuredImageThumbData[0]->value;
					}

					$brandProductsData = $this->getPageDatas(PageDatas::PROJECT_BRANDS_PRODUCTS);
					if (!empty($brandProductsData)){
						$brandProducts = $brandProductsData[0]->value;
					}

					$customerFirstNameData = $this->getPageDatas(PageDatas::CUSTOMER_REVIEW_FIRST_NAME);
					if (!empty($customerFirstNameData)){
						$customerFirstName = $customerFirstNameData[0]->value;
					}

					$customerLastNameData = $this->getPageDatas(PageDatas::CUSTOMER_REVIEW_LAST_NAME);
					if (!empty($customerLastNameData)){
						$customerLastName = $customerLastNameData[0]->value;
					}

					$customerTestimonialBodyData = $this->getPageDatas(PageDatas::CUSTOMER_REVIEW_TESTIMONIAL);
					if (!empty($customerTestimonialBodyData)){
						$customerTestimonialBody = $customerTestimonialBodyData[0]->value;
					}

					$customerTestimonialCheckData = $this->getPageDatas(PageDatas::CUSTOMER_DID_LEAVE_REVIEW);
					if (!empty($customerTestimonialCheckData)){
						$customerTestimonialCheck = (int) $customerTestimonialCheckData[0]->value;
					}

					/**
					 * Handle _legacy_ sectionized content parsing
					 * As of April 25, 2024, there is now a rigidly defined per-layout content section system
					 * supported in the CMS. This is a legacy parsing done with shortcode-like tags in the page
					 * content that should no longer be used.
					 */

					/** @var array $sections Keys will be the section name, values will be the section content */
					$sections = [];
					$sectionParser = new SectionParser();
					$sectionObjects = $sectionParser->parseSections($bodyContents);
					foreach($sectionObjects as $sectionObject){
						$sections[$sectionObject->name] = $sectionObject->content;
					}

					include $fsFileLayout->fullFilePath;
					$renderedPage = ob_get_clean();

					// Inject administrative front-ends if there is a user
					if (Account::getCurrentUser() !== null) {
						// Inject the public front end JS
						$frontEndJSScript = '<script type="module" src="/uplift-assets/js/public-front-end/PublicFrontEnd.js"></script>';

						$renderedPage = str_replace("</head>", "\n$frontEndJSScript\n</head>", $renderedPage);

						// If there is not a "noadmin" GET parameter then inject the admin bar too
						if (!isset($_GET['noadmin'])) {
							ob_start();
							include __DIR__ . "/../../resources/views/_partials/public-front-end/admin-bar.php";
							$publicAdminBar = ob_get_clean();
							$renderedPage = str_replace("</body>", $publicAdminBar . "</body>", $renderedPage);
						}
					}

					// Inject a noindex robots meta if the site has indexing disabled
					if (Setting::getSettingValue(Settings::ENTIRE_SITE_NO_INDEX->value) === "1"){
						$renderedPage = str_replace("</head>", str_replace("\t", "", <<<HTML
							<meta name="robots" content="noindex">
							</head>
						HTML), $renderedPage);
					}

					// Inject the Lightbox component if this site hasn't opted out of the injection
					$optOutLightboxSettingValue = Setting::getSettingValue(Settings::OPT_OUT_LIGHTBOX_INJECTION->value);
					if ($optOutLightboxSettingValue === null || $optOutLightboxSettingValue === "0"){
						$renderedPage = str_replace("</head>", str_replace("\t", "", <<<HTML
							<link rel="stylesheet" href="/uplift-assets/front-end-injections/css/simple-lightbox.css">
							<script type="module" src="/uplift-assets/front-end-injections/js/simple-lightbox.js"></script>
							<script type="module" src="/uplift-assets/front-end-injections/js/init-simple-lightbox.js"></script>
							</head>
						HTML), $renderedPage);
					}

					// Inject schema, if the page isn't opting out of it
					if ($this->excludeSchemaInjection === 0){
						$schemaArray = $this->getSchemaForPage();
						$schemaAsJSON = json_encode($schemaArray, JSON_PRETTY_PRINT);
						$schemaJSONLDElement = '<script type="application/ld+json">' . $schemaAsJSON . '</script>';

						// Add WebSite structured data, if a business name is set
						$companyName = Setting::getSettingValue(Settings::COMPANY_NAME->value);
						$structuredData = [
							"@context"=>"https://schema.org",
							"@type"=>"WebSite",
							"name"=>$companyName,
							"url"=>HttpHelper::getWebsiteBaseURL(),
						];

						$websiteSchemaJSONLDElement = '<script type="application/ld+json">' . json_encode($structuredData, JSON_PRETTY_PRINT) . '</script>';

						$renderedPage = str_replace(
							"<head>",
							"<head>\n" . $schemaJSONLDElement . "\n" . $websiteSchemaJSONLDElement,
							$renderedPage
						);
					}

					// Inject Uplift meta elements
					$renderedPage = str_replace("<head>", trim(<<<HTML
						<head>
						<meta name="powered-by" content="Uplift">
						<meta name="p-type" content="{$this->pageType}">
					HTML), $renderedPage);

					return $renderedPage;
				}
			}

			return sprintf("No layout file found for page ID %d", $this->id);
		}

		/**
		 * Finds the first H1 in the default/first content section or page body.
		 * If found, injects it as an og element into the page head.
		 * @throws DOMException
		 */
		public function processOGTitle(): void{
			if (!$this->pageHeadHasOpenGraphProperty("title")){
				// Inject one by finding page's first H1 and getting the text content, then trimming it.
				// Additionally, remove all tabs and newlines
				try {
					$h1Content = $this->getFirstH1ElementContent();
					$blankDoc = new \DOMDocument();
					$metaElement = $blankDoc->createElement("meta");
					$metaElement->setAttribute("property", "og:title");
					$metaElement->setAttribute("content", strip_tags($h1Content));
					$this->injectNodeIntoPageHead($metaElement);
				} catch (NoH1InBodyContent) {
					// Do nothing
				}
			}
		}

		/**
		 * Finds the first image in the default/first content section or page body.
		 * If found, injects it as an og element into the page head.
		 * @throws DOMException
		 */
		public function processOGImage(): void{
			if (!$this->pageHeadHasOpenGraphProperty("image")){
				try {
					$firstImgSrc = $this->getFirstImgElementSource();

					// Check if it is an absolute URL already
					$host = parse_url($firstImgSrc, PHP_URL_HOST);
					if ($host === null){
						// Add the base URL
						if (str_starts_with($firstImgSrc, "/")) {
							$firstImgSrc = HttpHelper::getWebsiteBaseURL() . $firstImgSrc;
						}else{
							$firstImgSrc = HttpHelper::getWebsiteBaseURL() . "/" . $firstImgSrc;
						}
					}

					$blankDoc = new \DOMDocument();
					$metaElement = $blankDoc->createElement("meta");
					$metaElement->setAttribute("property", "og:image");
					$metaElement->setAttribute("content", $firstImgSrc);
					$this->injectNodeIntoPageHead($metaElement);
				} catch (NoImgInBodyContent) {
					// Do nothing
				}
			}
		}

		/**
		 * Injects a meta element with og:url as the property and the current URL as the content. Does not
		 * inject the element if one already exists on the page.
		 * @return void
		 * @throws DOMException
		 */
		public function processOGUrl(): void{
			if (!$this->pageHeadHasOpenGraphProperty("url")){
				$baseUrl = HttpHelper::getWebsiteBaseURL();
				$path = $_SERVER['REQUEST_URI'];
				$currentUrl = $baseUrl . $path;
				$blankDoc = new \DOMDocument();
				$metaElement = $blankDoc->createElement("meta");
				$metaElement->setAttribute("property", "og:url");
				$metaElement->setAttribute("content", $currentUrl);
				$this->injectNodeIntoPageHead($metaElement);
			}
		}

		/**
		 * Checks if the pageHead contains a meta element with a property attribute matching the provided OG property
		 * named.
		 */
		public function pageHeadHasOpenGraphProperty(string $ogProperty): bool{
			// Wrap the content so that DOMDocument parses it correctly
			$pageHeadWrapped = "<html><head><meta charset=\"utf-8\">{$this->pageHead}</head></html>";
			libxml_use_internal_errors(true);
			$document = new \DOMDocument;
			$document->loadHTML($pageHeadWrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

			$metaElements = $document->getElementsByTagName("meta");

			/** @var DOMElement $node */
			foreach($metaElements as $node){
				if (strtolower($node->getAttribute("property")) === "og:{$ogProperty}"){
					return true;
				}
			}

			return false;
		}

		/**
		 * Attempts to find an H1 in the page's default/first content section or the page body.
		 * If there is one, will return the text content of that element with newlines and tabs removes then trimmed.
		 * @throws NoH1InBodyContent
		 */
		public function getFirstH1ElementContent(): string{
			// Wrap the content so that DOMDocument parses it correctly
			$defaultOrFirstContentSection = $this->getDefaultContentSectionOrFirstSection();
			$bodyContent = $defaultOrFirstContentSection?->content ?? $this->pageBody;
			$pageBodyWrapped = "<html><head><meta charset=\"utf-8\"></head><body>{$bodyContent}</body></html>";
			libxml_use_internal_errors(true);
			$document = new \DOMDocument;
			$document->loadHTML($pageBodyWrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

			$h1Elements = $document->getElementsByTagName("h1");
			if ($h1Elements->count() > 0){
				/** @var DOMElement $firstH1 */
				$firstH1 = $h1Elements->item(0);
				$innerContent = ExternalSiteImporterService::getInnerHTMLOfDOM($firstH1);
				$innerContent = preg_replace("/[\n\r\t]/", "", $innerContent);
				return trim($innerContent);
			}

			throw new NoH1InBodyContent();
		}

		/**
		 * Attempts to find the first img in the default/first content section or the pageBody if no sections.
		 * If there is one, will return the text content of that element with newlines and tabs removes then trimmed.
		 * @throws NoImgInBodyContent
		 */
		public function getFirstImgElementSource(): string{
			// Wrap the content so that DOMDocument parses it correctly
			$defaultOrFirstContentSection = $this->getDefaultContentSectionOrFirstSection();
			$bodyContent = $defaultOrFirstContentSection?->content ?? $this->pageBody;
			$pageBodyWrapped = "<html><head><meta charset=\"utf-8\"></head><body>{$bodyContent}</body></html>";
			libxml_use_internal_errors(true);
			$document = new \DOMDocument;
			$document->loadHTML($pageBodyWrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

			$imgElements = $document->getElementsByTagName("img");
			if ($imgElements->count() > 0){
				/** @var DOMElement $firstImg */
				$firstImg = $imgElements->item(0);
				return $firstImg->getAttribute("src");
			} else {
				/** @var ?PageData $featuredImage */
				$featuredImage = $this->getPageData(PageDatas::FEATURED_IMAGE);

				if (!empty($featuredImage)){
					return $featuredImage->value;
				}
			}

			throw new NoImgInBodyContent();
		}

		/**
		 * Attempts to find a <meta> element with the description name.
		 * @throws NoMetaDescriptionInHead
		 */
		public function getMetaDescription(): string{
			// Wrap the content so that DOMDocument parses it correctly
			$pageHeadWrapped = "<html><head><meta charset=\"utf-8\">{$this->pageHead}</head><body></body></html>";
			libxml_use_internal_errors(true);
			$document = new \DOMDocument;
			$document->loadHTML($pageHeadWrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

			$metaElements = $document->getElementsByTagName("meta");
			if ($metaElements->count() > 0){
				/** @var DOMElement $metaElement */
				foreach($metaElements->getIterator() as $metaElement){
					if (strtolower($metaElement->getAttribute("name") === "description")){
						return $metaElement->getAttribute("content");
					}
				}
			}

			throw new NoMetaDescriptionInHead();
		}

		/**
		 * Adds a DOMNode to the pageHead
		 */
		public function injectNodeIntoPageHead(DOMNode $node){
			// Wrap the content so that DOMDocument parses it correctly
			$pageHeadWrapped = "<html><head><meta charset=\"utf-8\">{$this->pageHead}</head></html>";
			libxml_use_internal_errors(true);
			$document = new \DOMDocument;
			$document->loadHTML($pageHeadWrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

			// Import the node into this document
			$node = $document->importNode($node);

			/** @var DOMElement $headElement */
			$headElement = $document->getElementsByTagName("head")->item(0);
			$headElement->append($node);

			$headInnerHTML = ExternalSiteImporterService::getInnerHTMLOfDOM($headElement);

			// Remove the meta charset that was added for UTF-8 parsing
			$headInnerHTML = str_replace("<meta charset=\"utf-8\">", "", $headInnerHTML);

			$this->pageHead = $headInnerHTML;
		}

		public function getLastEditTime(): int{
			/** @var ?PageArchive $editArchive */
			$editArchive = PageArchive::queryOne(
				columnQuery: (new ColumnQuery())
					->where("page_id","=",$this->id),
				resultOrder: (new ResultOrder())
					->by("archived_timestamp", "DESC"),
				pager: (new Pager(pageNumber: 1, limit: 1))
			);

			if ($editArchive === null){
				return $this->publicationTimestamp;
			}

			return $editArchive->archivedTimestamp;
		}

		/**
		 * @throws NoObjectFound
		 */
		public function getAttributeValue(int $pageAttributePageValueID): PageAttributePageValue{
			/** @var ?PageAttributePageValue $pageAttributePageValue */
			$pageAttributePageValue = PageAttributePageValue::fetch($pageAttributePageValueID);

			if ($pageAttributePageValue === null){
				throw new NoObjectFound("No PageAttributePageValue object with ID $pageAttributePageValueID");
			}else{
				return $pageAttributePageValue;
			}
		}

		/**
		 * @throws NoObjectFound
		 */
		public function getAttributeValueFromAttributeID(int $pageAttributeID): PageAttributePageValue{
			/** @var ?PageAttributePageValue $pageAttributePageValue */
			$pageAttributePageValue = PageAttributePageValue::queryOne(
				columnQuery: (new ColumnQuery())
					->where("page_attribute_id","=",$pageAttributeID)
					->and()
					->where("page_id","=",$this->id)
			);

			if ($pageAttributePageValue === null){
				throw new NoObjectFound("No PageAttributePageValue object with Attribute ID $pageAttributeID and Page ID " . $this->id);
			}else{
				return $pageAttributePageValue;
			}
		}

		/**
		 * Generates a page-aware JSON+LD compatible array that can be converted to a JSON string
		 * and injected onto a page.
		 * @return array
		 * @throws Exception
		 */
		public function getSchemaForPage(): array{
			$schema = [];
			$baseURL = HttpHelper::getWebsiteBaseURL();
			$context = "http://schema.org";
			$companySchemaType = Setting::getSettingValue(Settings::SCHEMA_TYPE->value);
			$companyName = Setting::getSettingValue(Settings::COMPANY_NAME->value);
			$logoRelativePath = Setting::getSettingValue(Settings::COMPANY_LOGO->value);
			$phoneNumbers = json_decode(Setting::getSettingValue(Settings::COMPANY_PHONE_NUMBERS->value) ?? "[]", true);
			$fullLogoURL = $baseURL . $logoRelativePath;
			$mainPhoneNumber = "";

			if (!empty($phoneNumbers)){
				$mainPhoneNumber = $phoneNumbers[0];
			}

			$schema['@context'] = $context;
			$schema['@type'] = $companySchemaType;
			$schema['name'] = $companyName;
			$schema['url'] = $baseURL;
			$schema['logo'] = $fullLogoURL;
			$schema['image'] = $fullLogoURL;
			$schema['telephone'] = $mainPhoneNumber;

			// Inject all addresses
			$addresses = [
				[
					"@type"=>"PostalAddress",
					"streetAddress"=>Setting::getSettingValue(Settings::COMPANY_STREET->value),
					"addressLocality"=>Setting::getSettingValue(Settings::COMPANY_CITY->value),
					"addressRegion"=>Setting::getSettingValue(Settings::COMPANY_STATE->value),
					"postalCode"=>Setting::getSettingValue(Settings::COMPANY_POSTAL->value),
				]
			];

			$additionalAddressCustomSettings = CustomSettingService::getAllCustomSettings(CustomSettings::ADDITIONAL_ADDRESS);
			if (!empty($additionalAddressCustomSettings)){
				foreach($additionalAddressCustomSettings as $additionalAddressCustomSetting){
					$valueAsJSON = $additionalAddressCustomSetting->value;
					/** @var array{street: string, city: string, state: string, postal: string, country: string} $addressArray */
					$addressArray = json_decode($valueAsJSON, true);
					$addresses[] = [
						"@type"=>"PostalAddress",
						"streetAddress"=>$addressArray['street'],
						"addressLocality"=>$addressArray['city'],
						"addressRegion"=>$addressArray['state'],
						"postalCode"=>$addressArray['postal'],
					];
				}
			}

			$schema['address'] = $addresses;


			return $schema;
		}

		/**
		 * Generates a URL-safe URL segment given a page name.
		 * E.g. "Landscaping in Dallas, TX" would return "landscaping-in-dallas-tx"
		 * @return string
		 */
		public function generateURLSlugFromPageName(): string{
			$urlSlug = strtolower($this->pageName); // Lower entire string

			// Handle when a comma doesn't have a space after it
			$urlSlug = preg_replace("/,\S/", ", ", $urlSlug);

			$urlSlug = preg_replace("/\s/", "-", $urlSlug); // Remove whitespace and use dashes
			$urlSlug = preg_replace("/[^a-zA-Z0-9\-]/", "", $urlSlug); // Remove all characters except letters, dashes, and numbers
			return preg_replace("/-{2,}/", "-", $urlSlug);
		}

		/**
		 * Returns the PageLayoutSection for this page's layout which is either
		 * - The default content section
		 * - The first section
		 * This will return null if this page's layout has no sections defined.
		 */
		public function getLayoutDefaultOrFirstSection(): ?PageLayoutSection{
			// Get the sections definition for this layout

			if (empty($this->pageLayout)){
				// Weird, page doesn't have a layout. Could be a new build that isn't fully set up
				return null;
			}

			$sectionsDefinition = PageEditorService::getPageLayoutSectionsDefinitionFromLayoutFileName($this->pageLayout);

			if ($sectionsDefinition !== null) {
				// Find the default content section for the layout definition, or the first section if there
				// is no default
				/** @var ?PageLayoutSection $defaultOrFirstSection */
				$defaultOrFirstSection = PageLayoutSection::queryOne(
					columnQuery: (new ColumnQuery())
						->where("page_layout_section_definition_id", "=", $sectionsDefinition->id),
					resultOrder: (new ResultOrder())
						->by("is_default_content_section", "DESC")
						->by("id", "ASC")
				);

				return $defaultOrFirstSection;
			}

			return null;
		}

		/**
		 * Returns the default content section or the first layout section. If this page's layout
		 * has no defined sections, then null is returned.
		 */
		public function getDefaultContentSectionOrFirstSection(): ?PageContentSection{
			$defaultOrFirstLayoutSection = $this->getLayoutDefaultOrFirstSection();
			if ($defaultOrFirstLayoutSection !== null){
				// Now, get the actual page content section
				/** @var ?PageContentSection $existingPageSectionWithSectionName */
				$existingPageSectionWithSectionName = PageContentSection::queryOne(
					columnQuery: (new ColumnQuery())
						->where("page_id", "=", $this->id)
						->and()
						->where("section_name", "=", $defaultOrFirstLayoutSection->sectionName)
				);

				return $existingPageSectionWithSectionName;
			}

			return null;
		}

		/**
		 * Saves the provided string content to the default layout section for this page, if there is one.
		 * If there is no default content section defined for the page's layout sections, then the first
		 * layout section is used.
		 * If the layout this page uses has no sectioned content, then this method will do nothing.
		 */
		public function saveToDefaultContentSectionOrFirstSection(string $content): void{
			$defaultOrFirstLayoutSection = $this->getLayoutDefaultOrFirstSection();
			if ($defaultOrFirstLayoutSection !== null){
				// Now, try to get an existing page section for that layout section.
				// It is okay if this is null, as savePageContentSections will create the page layout section
				// if we provide a null Id below.
				$defaultOrFirstPageContentSection = $this->getDefaultContentSectionOrFirstSection();

				// Save the content sections
				// This will not remove sections that are not in the array. This function simply adds or edits
				// sections that are provided in the array
				PageEditorService::savePageContentSections($this, [
					[
						"id"=>$defaultOrFirstPageContentSection?->id ?? null,
						"sectionName"=>$defaultOrFirstLayoutSection->sectionName,
						"content"=>$content
					],
				]);
			}

		}
	}