<?php
	namespace Installation;

	use GuzzleHttp\Exception\GuzzleException;
	use MonologWrapper\MonologWrapper;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\ImageManager\ImageFileAlreadyExists;

	#[Controller]
	#[RouteBase("/uplift/install")]
	class InstallationController extends BaseController
	{

		#[Route("GET", "/")]
		public function installView(): string
		{
			return Renderer::renderView(
				viewFileName: "installation/main.php",
				viewScope:[
					"parsedInitialPages"=>InstallationService::getInitialPages(),
				],
			);
		}

		#[Route("POST", "/")]
		#[UseJSON]
		#[ProcessRequestBody]
		public function initiateInstallationProcess(Request $request): JSONResult
		{
			$logger = MonologWrapper::getLogger();
			$payload = $request->getPayload();

			try{
				$schemaType = $payload->getTextPayload("schema-type");
				$acceloCompanyID = $payload->getTextPayload("company-accelo-id");
				$companyName = $payload->getTextPayload("company-name");
				$companyStreet = $payload->getTextPayload("company-street");
				$companyCity = $payload->getTextPayload("company-city");
				$companyState = $payload->getTextPayload("company-state");
				$companyPostal = $payload->getTextPayload("company-postal");
				$companyPhoneNumber = $payload->getTextPayload("company-phone-number");
				$companyFaxNumber = $payload->getTextPayload("company-fax-number");
				$masterAccountUsername = $payload->getTextPayload("master-account-username");
				$masterAccountFirstName = $payload->getTextPayload("master-account-first-name");
				$masterAccountLastName = $payload->getTextPayload("master-account-last-name");
				$masterAccountPassword = $payload->getTextPayload("master-account-password");
				$clientAccountUsername = $payload->getTextPayload("client-account-username");
				$clientAccountFirstName = $payload->getTextPayload("client-account-first-name");
				$clientAccountLastName = $payload->getTextPayload("client-account-last-name");
				$clientAccountPassword = $payload->getTextPayload("client-account-password");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			// These are okay to not be reported as missing
			try{
				$companyLogo = $payload->getFileUploadPayload("company-logo");
				$companyFavicon = $payload->getFileUploadPayload("company-favicon");
			}catch(NoPayloadFound){}

			// Handle initial page payload
			$initialPageFileNamesJSON = $payload->getTextPayloadNullable("initial-page-filenames");
			$isTestBuild = $payload->getTextPayloadNullable("is-test-build") !== null;

			try {
				InstallationService::install(
					schemaType: $schemaType->contents,
					acceloCompanyID: (int)$acceloCompanyID->contents,
					companyName: $companyName->contents,
					companyStreet: $companyStreet->contents,
					companyCity: $companyCity->contents,
					companyState: $companyState->contents,
					companyPostal: $companyPostal->contents,
					companyPhoneNumber: $companyPhoneNumber->contents,
					companyFaxNumber: $companyFaxNumber->contents,
					logo: $companyLogo ?? null,
					favicon: $companyFavicon ?? null,
					masterAccountUsername: $masterAccountUsername->contents,
					masterAccountFirstName: $masterAccountFirstName->contents,
					masterAccountLastName: $masterAccountLastName->contents,
					masterAccountPassword: $masterAccountPassword->contents,
					clientAccountUsername: $clientAccountUsername->contents,
					clientAccountFirstName: $clientAccountFirstName->contents,
					clientAccountLastName: $clientAccountLastName->contents,
					clientAccountPassword: $clientAccountPassword->contents,
					initialPageFileNamesToCreateJSON: $initialPageFileNamesJSON?->contents,
					isTestBuild: $isTestBuild,
				);

				$logger->info("Successfully installed Uplift.");
			}catch(GuzzleException $e){
				$logger->error("Guzzle exception during system installation: {$e->getMessage()}");
				http_response_code(500);
				return new JSONError($e->getMessage());
			} catch (InvalidInstallationState|EmptyValue|ImageFileAlreadyExists|MalformedValue $e) {
				$logger->error($e->getMessage());
				http_response_code(500);
				return new JSONError($e->getMessage());
			}catch(ParseError $e){
				$logger->error($e->getMessage());
				http_response_code(500);
				return new JSONError(sprintf("The system was installed, but a Nox ParseError was received when trying to create the initial pages. Error: %s", $e->getMessage()));
			}

			return new JSONSuccess();
		}
	}