<?php
	namespace Uplift\ImageProcessing;

	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\GuzzleException;
	use GuzzleHttp\Exception\RequestException;
	use GuzzleHttp\RequestOptions;
	use MonologWrapper\MonologWrapper;
	use Uplift\ImageProcessing\Exceptions\ImageProcessingException;

	/**
	 * Utility class for interfacing with a remote image processing server's APIs that allow us to manipulate images
	 * of varying image types.
	 */
	class ImageProcessing{
		public const IMAGE_EXTENSION_TO_MIME_MAP = [
			"jpg" => "image/jpeg",
			"jpeg" => "image/jpeg",
			"png" => "image/png",
			"gif" => "image/gif",
			"webp" => "image/webp",
			"heic" => "image/heic",
		];
		// private const IMAGE_PROCESSING_SERVER_BASE = "https://localhost:7202";
		private const IMAGE_PROCESSING_SERVER_BASE = "https://image-server-1.fbdash.com";

		/**
		 * Sends a request to the image processing server to information about the image from the provided
		 * binary of that image.
		 * @throws ImageProcessingException
		 */
		public function getImageInfo(
			string $imageBasename,
			string $imageBinary,
		): ?ImageDataPayload{
			$logger = MonologWrapper::getLogger();
			$guzzleClient = new Client([
				"verify" => false,
			]);

			try{
				$parameters = [];
				$parameters[] = ["name"=>"imageFile", "contents"=>$imageBinary, "filename"=>$imageBasename];

				$response = $guzzleClient->request(
					"POST",
					self::IMAGE_PROCESSING_SERVER_BASE . "/api/v1/info",
					[
						RequestOptions::MULTIPART => $parameters,
					],
				);
			}catch(GuzzleException $ex){
				$exceptionClassName = $ex::class;
				if ($ex instanceof RequestException) {
					$bodyContents = $ex->getResponse()?->getBody()?->getContents();
					$logger->error("Image processing error. {$exceptionClassName} message: " . $bodyContents);
					throw new ImageProcessingException($bodyContents);
				}else{
					$logger->error("Image processing error. {$exceptionClassName} message: " . $ex->getMessage());
					throw new ImageProcessingException($ex->getMessage());
				}
			}

			if ($response->getStatusCode() === 200){
				$jsonContents = $response->getBody()->getContents();
				/** @var {MIME: string, Width: int, Height: int, FileSizeInBytes} $imageDataRawObject */
				$imageDataRawObject = json_decode($jsonContents);
				$imageData = new ImageDataPayload();
				$imageData->height = $imageDataRawObject->height;
				$imageData->width = $imageDataRawObject->width;
				$imageData->fileSizeInBytes = $imageDataRawObject->fileSizeInBytes;
				$imageData->mime = $imageDataRawObject->mime;
				return $imageData;
			}else{
				$logger->error("Image processing error. HTTP status code: " . $response->getStatusCode());
				throw new ImageProcessingException("Image processing error. HTTP status code: " . $response->getStatusCode());
			}
		}

		/**
		 * Sends a request to the image processing server to resize the provided image to the provided dimensions
		 * Returns the resized image as a binary string. It will return null if something failed, and the problem
		 * will be logged to the standard logging file with Monolog.
		 * @throws ImageProcessingException
		 */
		public function resize(
			string $imageBasename,
			string $imageBinary,
			?int $width,
			?int $height
		): ?string{
			$logger = MonologWrapper::getLogger();
			$guzzleClient = new Client([
				"verify" => false,
			]);

			try{
				$parameters = [];
				$parameters[] = ["name"=>"imageFile", "contents"=>$imageBinary, "filename"=>$imageBasename];

				if ($width !== null){
					$parameters[] = ["name"=>"width", "contents"=>$width];
				}

				if ($height !== null){
					$parameters[] = ["name"=>"height", "contents"=>$height];
				}

				$response = $guzzleClient->request(
					"PATCH",
					self::IMAGE_PROCESSING_SERVER_BASE . "/api/v1/resize",
					[
						RequestOptions::MULTIPART => $parameters,
					],
				);
			}catch(GuzzleException $ex){
				$exceptionClassName = $ex::class;
				if ($ex instanceof RequestException) {
					$bodyContents = $ex->getResponse()?->getBody()?->getContents();
					$logger->error("Image processing error. {$exceptionClassName} message: " . $bodyContents);
					throw new ImageProcessingException($bodyContents);
				}else{
					$logger->error("Image processing error. {$exceptionClassName} message: " . $ex->getMessage());
					throw new ImageProcessingException($ex->getMessage());
				}
			}

			if ($response->getStatusCode() === 200){
				return $response->getBody()->getContents();
			}else{
				$logger->error("Image processing error. HTTP status code: " . $response->getStatusCode());
				throw new ImageProcessingException("Image processing error. HTTP status code: " . $response->getStatusCode());
			}
		}

		/**
		 * Sends a request to the image processing server to crop the provided image from the center
		 * with a given rectangular size (width and height)
		 * Returns the cropped image as a binary string. It will return null if something failed, and the problem
		 * will be logged to the standard logging file with Monolog.
		 * @throws ImageProcessingException
		 */
		public function cropFromCenter(
			string $imageBasename,
			string $imageBinary,
			?int $cropFromCenterWidth,
			?int $cropFromCenterHeight
		): ?string{
			$logger = MonologWrapper::getLogger();
			$guzzleClient = new Client([
				"verify" => false,
			]);

			try{
				$parameters = [];
				$parameters[] = ["name"=>"imageFile", "contents"=>$imageBinary, "filename"=>$imageBasename];

				if ($cropFromCenterWidth !== null){
					$parameters[] = ["name"=>"width", "contents"=>$cropFromCenterWidth];
				}

				if ($cropFromCenterHeight !== null){
					$parameters[] = ["name"=>"height", "contents"=>$cropFromCenterHeight];
				}

				$response = $guzzleClient->request(
					"PATCH",
					self::IMAGE_PROCESSING_SERVER_BASE . "/api/v1/crop-from-center",
					[
						RequestOptions::MULTIPART => $parameters,
					],
				);
			}catch(GuzzleException $ex){
				$exceptionClassName = $ex::class;
				if ($ex instanceof RequestException) {
					$bodyContents = $ex->getResponse()?->getBody()?->getContents();
					$logger->error("Image processing error. {$exceptionClassName} message: " . $bodyContents);
					throw new ImageProcessingException($bodyContents);
				}else{
					$logger->error("Image processing error. Exception message: " . $ex->getMessage());
					throw new ImageProcessingException($ex->getMessage());
				}
			}

			if ($response->getStatusCode() === 200){
				return $response->getBody()->getContents();
			}else{
				$logger->error("Image processing error. HTTP status code: " . $response->getStatusCode());
				throw new ImageProcessingException("Image processing error. HTTP status code: " . $response->getStatusCode());
			}
		}

		/**
		 * Sends a request to the image processing server to crop the provided image using a provided
		 * rectangle's dimensions.
		 * Returns the cropped image as a binary string. It will return null if something failed, and the problem
		 * will be logged to the standard logging file with Monolog.
		 * @throws ImageProcessingException
		 */
		public function crop(
			string $imageBasename,
			string $imageBinary,
			int $topLeftX,
			int $topLeftY,
			int $bottomRightX,
			int $bottomRightY,
		): ?string{
			$logger = MonologWrapper::getLogger();
			$guzzleClient = new Client([
				"verify" => false,
			]);

			try{
				$parameters = [];
				$parameters[] = ["name"=>"imageFile", "contents"=>$imageBinary, "filename"=>$imageBasename];
				$parameters[] = ["name"=>"topLeftX", "contents"=>$topLeftX];
				$parameters[] = ["name"=>"topLeftY", "contents"=>$topLeftY];
				$parameters[] = ["name"=>"bottomRightX", "contents"=>$bottomRightX];
				$parameters[] = ["name"=>"bottomRightY", "contents"=>$bottomRightY];

				$response = $guzzleClient->request(
					"PATCH",
					self::IMAGE_PROCESSING_SERVER_BASE . "/api/v1/crop",
					[
						RequestOptions::MULTIPART => $parameters,
					],
				);
			}catch(GuzzleException $ex){
				$exceptionClassName = $ex::class;
				if ($ex instanceof RequestException) {
					$bodyContents = $ex->getResponse()?->getBody()?->getContents();
					$logger->error("Image processing error. {$exceptionClassName} message: " . $bodyContents);
					throw new ImageProcessingException($bodyContents);
				}else{
					$logger->error("Image processing error. Exception message: " . $ex->getMessage());
					throw new ImageProcessingException($ex->getMessage());
				}
			}

			if ($response->getStatusCode() === 200){
				return $response->getBody()->getContents();
			}else{
				$logger->error("Image processing error. HTTP status code: " . $response->getStatusCode());
				throw new ImageProcessingException("Image processing error. HTTP status code: " . $response->getStatusCode());
			}
		}

		/**
		 * Sends a request to the image processing server to convert the image's current image file type
		 * to a new file type that is identified by the new file type's MIME.
		 * Returns the resized image as a binary string. It will return null if something failed, and the problem
		 * will be logged to the standard logging file with Monolog.
		 * @throws ImageProcessingException
		 */
		public function convertImageType(
			string $imageBasename,
			string $imageBinary,
			string $newImageMimeType,
		): ?string{
			$logger = MonologWrapper::getLogger();
			$guzzleClient = new Client([
				"verify" => false,
			]);

			try{
				$parameters = [];
				$parameters[] = ["name"=>"imageFile", "contents"=>$imageBinary, "filename"=>$imageBasename];
				$parameters[] = ["name"=>"desiredMimeType", "contents"=>$newImageMimeType];

				$response = $guzzleClient->request(
					"PATCH",
					self::IMAGE_PROCESSING_SERVER_BASE . "/api/v1/convert",
					[
						RequestOptions::MULTIPART => $parameters,
					],
				);
			}catch(GuzzleException $ex){
				$exceptionClassName = $ex::class;
				if ($ex instanceof RequestException) {
					$bodyContents = $ex->getResponse()?->getBody()?->getContents();
					$logger->error("Image processing error. {$exceptionClassName} message: " . $bodyContents);
					throw new ImageProcessingException($bodyContents);
				}else{
					$logger->error("Image processing error. Exception message: " . $ex->getMessage());
					throw new ImageProcessingException($ex->getMessage());
				}
			}

			if ($response->getStatusCode() === 200){
				return $response->getBody()->getContents();
			}else{
				$logger->error("Image processing error. HTTP status code: " . $response->getStatusCode());
				throw new ImageProcessingException("Image processing error. HTTP status code: " . $response->getStatusCode());
			}
		}
	}