<?php

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use ImageGallery\ImageGallery;
	use ImageGallery\ImageGalleryMember;
	use ImageGallery\ImageGalleryNameEmpty;
	use ImageGallery\ImageGalleryService;
	use ImageGallery\ImageGalleryWithSameNameExists;
	use Nox\Http\ArrayPayload;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use System\HttpHelper;
	use Uplift\ImageManager\ImageMissingThumb;

	#[Controller]
	#[RouteBase("/uplift")]
	class ImageGalleryController extends BaseController{

		/**
		 * @throws \Nox\RenderEngine\Exceptions\ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/gallery-manager")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "gallery-manager/main.php",
			);
		}

		#[Route("GET", "/gallery-manager/galleries")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function getGalleries(): JSONResult
		{
			return new JSONSuccess([
				"galleries"=>ImageGalleryService::getAllGalleries(),
			]);
		}

		#[Route("PUT", "/gallery-manager/gallery")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function createGallery(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$name = $payload->getTextPayload("gallery-name");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$gallery = ImageGalleryService::createNewGallery($name->contents);
			} catch (ImageGalleryNameEmpty|ImageGalleryWithSameNameExists $e) {
				return new JSONError($e->getMessage());
			}

			// Populates the numImagesInGallery property
			$gallery->fetchNumImagesInGallery();

			ActivityLog::log(
				categoryID: ActivityLogCategories::CREATE_IMAGE_GALLERY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"galleryName"=>$name,
				]),
			);

			return new JSONSuccess([
				"newGallery"=>$gallery,
			]);
		}

		#[Route("GET", "@/gallery-manager/gallery/(?<galleryID>\d+)/members$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function getGalleryMembers(Request $request): JSONResult
		{
			$galleryID = (int) $request->getParameter("galleryID");

			$gallery = ImageGallery::fetch($galleryID);
			if ($gallery === null){
				return new JSONError(sprintf("Gallery with ID %d doesn't exist.", $galleryID));
			}

			return new JSONSuccess([
				"members"=>ImageGalleryService::getGalleryMembers($galleryID),
			]);
		}

		#[Route("PUT", "@/gallery-manager/gallery/(?<galleryID>\d+)/members$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function addMembersToGallery(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$galleryID = (int) $request->getParameter("galleryID");
			$account = Account::getCurrentUser();

			try {
				$images = $payload->getArrayPayload("images");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			/** @var ImageGallery $gallery */
			$gallery = ImageGallery::fetch($galleryID);
			if ($gallery === null){
				return new JSONError(sprintf("Gallery with ID %d doesn't exist.", $galleryID));
			}

			if (empty($images)){
				return new JSONError("Missing images payload.");
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::ADD_IMAGES_TO_GALLERY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"galleryName"=>$gallery->name,
				]),
			);

			try {
				$newGalleryMembers = ImageGalleryService::addMembersToGallery($galleryID, $images->contents);
				return new JSONSuccess([
					"newMembers"=>$newGalleryMembers
				]);
			}catch(ImageMissingThumb $ex){
				http_response_code(400);
				return new JSONError($ex->getMessage());
			}

		}

		#[Route("PATCH", "@/gallery-manager/gallery/(?<galleryID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function updateMemberPositionsInGallery(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$galleryID = (int) $request->getParameter("galleryID");
			$membersPayloads = $payload->getAllPayloads();
			$account = Account::getCurrentUser();

			/** @var ImageGallery $gallery */
			$gallery = ImageGallery::fetch($galleryID);

			if ($gallery === null){
				return new JSONError(sprintf("Gallery with ID %d doesn't exist.", $galleryID));
			}

			$members = [];

			/** @var ArrayPayload[] $membersPayload */
			foreach($membersPayloads as $membersPayload){
				$members[] = $membersPayload->contents;
			}

			if (empty($members)){
				return new JSONError("Member payload is empty.");
			}

			ImageGalleryService::updateMembersPositionInGallery($galleryID, $members);

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_IMAGES_IN_GALLERY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"galleryName"=>$gallery->name,
					"editType"=>"Adjusted image positioning in gallery",
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "@/gallery-manager/gallery/(?<galleryID>\d+)/name$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function renameGallery(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$galleryID = (int) $request->getParameter("galleryID");
			$account = Account::getCurrentUser();

			try {
				$newName = $payload->getTextPayload("gallery-name");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			/** @var ImageGallery $gallery */
			$gallery = ImageGallery::fetch($galleryID);
			if ($gallery === null){
				return new JSONError(sprintf("Gallery with ID %d doesn't exist.", $galleryID));
			}

			if (empty($newName)){
				return new JSONError("New gallery name can't be empty.");
			}

			try {
				ImageGalleryService::updateGalleryName($gallery, $newName->contents);
			} catch (ImageGalleryWithSameNameExists) {
				return new JSONError("Another gallery with that name already exists.");
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::RENAME_IMAGE_GALLERY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"oldName"=>$gallery->name,
					"newName"=>$newName->contents,
				]),
			);

			return new JSONSuccess([
				"newName"=>$newName->contents,
			]);
		}

		#[Route("PATCH", "@/gallery-manager/gallery/(?<galleryID>\d+)/members/(?<galleryMemberID>\d+)/alt-text$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function updateGalleryMemberAltText(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$galleryID = (int) $request->getParameter("galleryID");
			$galleryMemberID = (int) $request->getParameter("galleryMemberID");
			$account = Account::getCurrentUser();

			try {
				$altText = $payload->getTextPayload("alt-text");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			/** @var ImageGallery $gallery */
			$gallery = ImageGallery::fetch($galleryID);

			/** @var ImageGalleryMember $galleryMember */
			$galleryMember = ImageGalleryMember::fetch($galleryMemberID);

			if ($gallery === null){
				return new JSONError(sprintf("Gallery with ID %d doesn't exist.", $galleryID));
			}

			if ($galleryMember === null){
				return new JSONError(sprintf("Gallery member with ID %d doesn't exist.", $galleryID));
			}

			ImageGalleryService::updateMemberAltText($galleryMember, $altText->contents);

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_IMAGES_IN_GALLERY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"galleryName"=>$gallery->name,
					"editType"=>"Adjusted image alt text in gallery",
				]),
			);

			return new JSONSuccess();
		}

		#[Route("DELETE", "@/gallery-manager/gallery/(?<galleryID>\d+)/members/(?<galleryMemberID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function deleteGalleryMemberFromGallery(Request $request): JSONResult
		{
			$galleryID = (int) $request->getParameter("galleryID");
			$galleryMemberID = (int) $request->getParameter("galleryMemberID");
			$account = Account::getCurrentUser();

			ImageGalleryService::removeGalleryMember($galleryMemberID);

			ActivityLog::log(
				categoryID: ActivityLogCategories::REMOVE_IMAGES_FROM_GALLERY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"galleryID"=>$galleryID,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("DELETE", "@/gallery-manager/gallery/(?<galleryID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_IMAGE_GALLERIES)]
		public function deleteGallery(Request $request): JSONResult
		{
			$galleryID = (int) $request->getParameter("galleryID");
			$account = Account::getCurrentUser();

			/** @var ImageGallery $gallery */
			$gallery = ImageGallery::fetch($galleryID);

			if ($gallery === null){
				return new JSONError(sprintf("Gallery with ID %d doesn't exist.", $galleryID));
			}

			ImageGalleryService::deleteGallery($gallery);

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_IMAGE_GALLERY->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"galleryName"=>$gallery->name,
				]),
			);

			return new JSONSuccess();
		}
	}