<?php
	/**
	 * @author Garet C. Green
	 */

	namespace GDHelper;

	use GDHelper\Exceptions\AnimatedWebPNotSupported;
	use GDHelper\Exceptions\FileNotFound;
	use GDHelper\Exceptions\ImageTypesAreTheSame;
	use GDHelper\Exceptions\InvalidImage;
	use GDHelper\Exceptions\UnrecognizedImageExtension;
	use GDHelper\Exceptions\UnsupportedImageTypeInteger;
	use GDHelper\Implementations\CropImage;
	use GDHelper\Implementations\ResizeImage;
	use GDHelper\Implementations\RotateImage;
	use Monolog\Level;
	use MonologWrapper\MonologWrapper;

	class GDHelper implements \JsonSerializable
	{

		public string $binary;
		public int $width;
		public int $height;
		public int $imageType;
		/**
		 * @var int Orientation obtained by reading the exif data of the file, if any. This is in degrees - it is
		 * not the flag value itself (e.g., it's not the 3, 6, 8 that Exif stores but is the degree conversion of those).
		 */
		public int $exifOrientation = 0;

		/** @var resource $resource Cannot type-hint Resources as of PHP 8. Is the image resource */
		public $resource;

		/**
		 * Constructs an instance with a file's binary
		 * contents
		 * @param string $binary
		 * @return GDHelper
		 */
		public function __construct(string $binary)
		{
			$this->binary = $binary;

			// Read the first 16 bytes to handle VP8X fatal error catching
			// Animated WebP with VP8X will throw an uncatchable error.
			// So we must try to identify it first
			$firstBytes = substr($binary, 12, 4);
			if ($firstBytes === "VP8X") {
				$animFlag = substr($binary, 20, 1);
				$isAnimated = (ord($animFlag) >> 1) & 1;
				if ($isAnimated) {
					throw new AnimatedWebPNotSupported("Animated WebP currently not supported by the PHP GD library.");
				}
			}

			// Get the image's info
			$imageInfo = @getimagesizefromstring($this->binary);
			if ($imageInfo === false) {
				throw new InvalidImage;
			} else {
				$this->width = $imageInfo[0];
				$this->height = $imageInfo[1];
				$this->imageType = $imageInfo[2];
			}
		}

		/**
		 * Allocates the memory and resource for this GDHelper image
		 */
		public function allocateResource()
		{
			$this->resource = @imagecreatefromstring($this->binary);

			// Convert all to RGB and save alpha channels
			imagepalettetotruecolor($this->resource);
			imagealphablending($this->resource, true);
			imagesavealpha($this->resource, true);

			// Was the image parsable?
			if ($this->resource === false) {
				throw new InvalidImage("The binary passed is not a valid image type.");
			}
		}

		/**
		 * Rotates an image in degrees
		 * @param float $angleInDegrees
		 * @param int $backgroundFillColor (Optional) To fill space now unused by the image
		 * @return GDHelper A new instance of GDHelper with the cropped image
		 */
		public function rotate(float $angleInDegrees, int $backgroundFillColor = 0)
		{
			$rotateImage = new RotateImage($this);
			return $rotateImage->rotate($angleInDegrees, $backgroundFillColor);
		}

		/**
		 * Resizes an image with pixel interpolation
		 * @return GDHelper A new instance of GDHelper with the cropped image
		 */
		public function resize(int $x, int $y)
		{
			$resizeImage = new ResizeImage($this);
			return $resizeImage->resize($x, $y);
		}

		/**
		 * Clears the stored GD resource
		 */
		public function clearResource()
		{
			if ($this->resource !== null) {
				imagedestroy($this->resource);
			}
		}

		/**
		 * Crops an image at the given boundaries
		 * @return GDHelper new instance
		 */
		public function crop(
			int $topX,
			int $topY,
			int $bottomX,
			int $bottomY
		): GDHelper
		{
			$cropImage = new CropImage($this);
			return $cropImage->crop($topX, $topY, $bottomX, $bottomY);
		}

		/**
		 * Crops an image from the center
		 * @return GDHelper new instance
		 */
		public function cropFromCenter(int $sizeX, int $sizeY): GDHelper
		{
			$cropImage = new CropImage($this);
			return $cropImage->cropFromCenter($sizeX, $sizeY);
		}

		/**
		 * Converts the image into a different image type. Will return
		 * a new GDHelper object.
		 * @throws ImageTypesAreTheSame
		 */
		public function convert(int $toImageType): GDHelper
		{
			if ($this->imageType === $toImageType) {
				throw new ImageTypesAreTheSame();
			}

			ob_start();
			switch ($toImageType) {
				case IMAGETYPE_JPEG:
					imagejpeg($this->resource);
					$newBinary = ob_get_contents();
					break;
				case IMAGETYPE_PNG:
					imagepng($this->resource);
					$newBinary = ob_get_contents();
					break;
				case IMAGETYPE_GIF:
					imagegif($this->resource);
					$newBinary = ob_get_contents();
					break;
				case IMAGETYPE_WEBP:
					imagewebp($this->resource);
					$newBinary = ob_get_contents();
					break;
				default:
					throw new UnsupportedImageTypeInteger();
			}

			ob_end_clean();

			$newGDHelper = new GDHelper($newBinary);

			// Handle exif rotation
			if ($this->exifOrientation !== 0){
				$newGDHelper->allocateResource();
				$rotatedNewGDHelper = $newGDHelper->rotate($this->exifOrientation);
				$newGDHelper->clearResource();
				$newGDHelper = $rotatedNewGDHelper;
			}

			return $newGDHelper;
		}

		/**
		 * Gets a base64 data string ready to be served
		 * over an HTTP stream. As a URL, image source, etc
		 * @return string
		 */
		public function toBase64DataString(): string
		{
			$base64Image = base64_encode($this->binary);

			switch ($this->imageType) {
				case IMAGETYPE_JPEG:
					return "data:image/jpeg;base64,$base64Image";
					break;
				case IMAGETYPE_PNG:
					return "data:image/png;base64,$base64Image";
					break;
				case IMAGETYPE_GIF:
					return "data:image/gif;base64,$base64Image";
					break;
				case IMAGETYPE_WEBP:
					return "data:image/webp;base64,$base64Image";
					break;
				default:
					break;
			}

			return "";
		}

		/**
		 * @param string $filePath
		 * @return GDHelper
		 * @throws AnimatedWebPNotSupported
		 * @throws FileNotFound
		 * @throws InvalidImage
		 */
		public static function fromFilePath(string $filePath): GDHelper
		{
			$filePath = realpath($filePath);
			if (!$filePath) {
				throw new FileNotFound();
			}

			$logger = MonologWrapper::getLogger();
			$binary = file_get_contents($filePath);

			try {
				$gdHelper = new GDHelper($binary);
			} catch (InvalidImage $e) {
				$logger->error($e->getMessage());
				throw new InvalidImage(sprintf("%s - is not a valid image file.", $filePath));
			}

			// Try to read the orientation
			if (function_exists("exif_read_data")){
				$exifData = @exif_read_data($filePath);
				if ($exifData !== false) {
					if (array_key_exists("Orientation", $exifData)) {
						$orientationFlag = (int) $exifData['Orientation'];
						$gdHelper->exifOrientation = match ($orientationFlag) {
							3 => 180,
							6 => -90,
							8 => 90,
							default => 0,
						};
					}
				}
			}

			return $gdHelper;
		}

		/**
		 * Converts an image file extension to the GDImage type
		 */
		public static function extensionToGDType(string $extension): int
		{
			$extension = strtolower($extension);
			return match ($extension) {
				"jpg", "jpeg" => IMAGETYPE_JPEG,
				"gif" => IMAGETYPE_GIF,
				"png" => IMAGETYPE_PNG,
				"webp" => IMAGETYPE_WEBP,
				default => throw new UnrecognizedImageExtension(),
			};
		}

		public function jsonSerialize(): array
		{
			return [
				"width" => $this->width,
				"height" => $this->height,
				"imageType" => $this->imageType,
			];
		}
	}
