<?php

	namespace ExternalSiteImporter;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use GuzzleHttp\Exception\GuzzleException;
	use MonologWrapper\MonologWrapper;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Redirect;
	use Nox\Http\Request;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use PageEditor\Exceptions\PageLayoutDoesntExist;
	use PageEditor\Exceptions\PageLayoutIsBlank;
	use PageEditor\Exceptions\PageNameExists;
	use PageEditor\Exceptions\PageNameExistsWithSamePageType;
	use PageEditor\Exceptions\PageNameIsBlank;
	use PageEditor\Exceptions\PageRouteInUse;
	use PageEditor\Exceptions\PageRouteIsBlank;
	use PageEditor\Exceptions\PublicationStatusDoesntExist;
	use Roles\PermissionCategories;
	use System\HttpHelper;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\MalformedValue;

	#[Controller]
	#[RouteBase("/uplift/external-importer")]
	class ExternalSiteImporterController extends BaseController{

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::WEBSITE_IMPORTER)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName:"external-site-importer/main.php",
			);
		}

		#[Route("GET", "/fetch")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::WEBSITE_IMPORTER)]
		public function fetchURLContent(Request $request): JSONResult
		{
			$url = $request->getQueryValue("url");
			$logger = MonologWrapper::getLogger();

			if ($url === null){
				http_response_code(500);
				return new JSONError("Missing url parameter in request.");
			}

			try {
				$result = ExternalSiteImporterService::fetch($url);
			} catch (NoBodyElement|NoHeadElement|GuzzleException|EmptyValue $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			$logger->info("Fetched remote URL for external site importer: $url");

			ActivityLog::log(
				categoryID: ActivityLogCategories::EXTERNAL_SITE_IMPORTER_URL_FETCH->value,
				accountID: (Account::getCurrentUser())->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"url"=>$url,
				]),
			);

			return new JSONSuccess([
				"head"=>$result->head,
				"body"=>$result->body,
			]);
		}

		#[Route("POST", "/build")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::WEBSITE_IMPORTER)]
		public function buildNewPage(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			try{
				$importedFromURL = $payload->getTextPayload("imported-from-url");
				$pageName = $payload->getTextPayload("page-name");
				$pageType = $payload->getTextPayload("page-type");
				$cityName = $payload->getTextPayload("page-city-name");
				$stateName = $payload->getTextPayload("page-state-name");
				$stateNameShorthand = $payload->getTextPayload("page-state-name-shorthand");
				$cityURL = $payload->getTextPayload("page-city-url");
				$pageRoute = $payload->getTextPayload("page-route");
				$pageLayout = $payload->getTextPayload("page-layout");
				$breadcrumbsJSON = $payload->getTextPayload("breadcrumbs");
				$pageHead = $payload->getTextPayload("page-head");
				$pageBody = $payload->getTextPayload("page-body");
			}catch(NoPayloadFound $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			try {
				$newPage = ExternalSiteImporterService::buildPageFromImport(
					urlImportedFrom: $importedFromURL->contents,
					pageName: $pageName->contents,
					pageType: $pageType->contents,
					pageHead: $pageHead->contents,
					pageBody: $pageBody->contents,
					pageRoute: $pageRoute->contents,
					pageLayout: $pageLayout->contents,
					cityName: $cityName->contents,
					stateName: $stateName->contents,
					stateNameShorthand: $stateNameShorthand->contents,
					cityURL: $cityURL->contents,
					breadcrumbsJSON: $breadcrumbsJSON->contents,
				);
			} catch (\Exception $e) {
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::EXTERNAL_SITE_IMPORTER_BUILD_PAGE->value,
				accountID: (Account::getCurrentUser())->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"pageID"=>$newPage->id,
					"pageName"=>$newPage->pageName,
				]),
			);

			return new JSONSuccess([
				"newPageURL"=>"/uplift/page-editor/{$newPage->id}",
			]);
		}
	}