<?php
	namespace Crons;

	// Required to import here on some systems as they load out of order
	require_once __DIR__ . "/Jobs/Job.php";

	use Crons\Jobs\Job;
	use Crons\Jobs\SendUpdateToControlPanel;
	use MonologWrapper\MonologWrapper;
	use SebastianBergmann\CodeCoverage\NoCodeCoverageDriverAvailableException;

	class CronsService{

		private static function getLastCronRunTimestamp(string $cronClassName): int{
			/** @var ?CronRunHistory $runHistory */
			$runHistory = CronRunHistory::fetch($cronClassName);
			if ($runHistory === null){
				return 0;
			}else{
				return $runHistory->lastRunTimestamp;
			}
		}

		/**
		 * Statically defined registered crons. To add a new cron, add your cron into the return array.
		 * @return Job[]
		 */
		private static function getRegisteredCrons(): array{
			return [
				(new SendUpdateToControlPanel(self::getLastCronRunTimestamp(SendUpdateToControlPanel::class))),
			];
		}

		/**
		 * Runs registered crons.
		 */
		public static function runCrons(): void{
			$logger = MonologWrapper::getLogger();
			$cronsRan = 0;
			$cronsSucceeded = 0;
			$cronsFailed = 0;
			$logger->debug("Running crons.");
			foreach(self::getRegisteredCrons() as $job){
				if ($job->isDue()){
					++$cronsRan;
					$jobClassName = $job::class;

					/** @var ?CronRunHistory $runHistory */
					$runHistory = CronRunHistory::fetch($jobClassName);
					if ($runHistory === null) {
						// Create one
						$runHistory = new CronRunHistory();
						$runHistory->name = $jobClassName;
						$runHistory->lastRunTimestamp = time();
					}else{
						$runHistory->lastRunTimestamp = time();
					}

					try {
						$job->run();
						$runHistory->didSucceed = 1;
						++$cronsSucceeded;
					}catch(\Exception $e){
						++$cronsFailed;
						$logger->error($job::class . " failed to finish its run. Error: {$e->getMessage()}");
						$runHistory->didSucceed = 0;
						continue;
					}

					$runHistory->save();
				}else{
					$nextDueTime = $job->getCronTimer()->getDueTimestamp($job->lastRunTimestamp);
					$logger->debug($job::class . " is not due yet. Next due timestamp is {$nextDueTime} which is " . date("M d, Y H:i:s", $nextDueTime));
				}
			}

			if ($cronsRan > 0){
				$logger->info("Ran $cronsRan cron jobs. $cronsSucceeded succeeded. $cronsFailed failed.");
			}
		}
	}