<?php
	namespace ClientProjectForm;

	use ClientProjectForm\Exceptions\ProjectValidationError;
	use ContentHelper\ContentHelper;
	use FileSystemUtilities\Exceptions\InvalidFileName;
	use FileSystemUtilities\FileSystemUtilities;
	use FileSystemUtilities\FSImageFile;
	use GDHelper\Exceptions\AnimatedWebPNotSupported;
	use GDHelper\exceptions\FileNotFound;
	use GDHelper\Exceptions\InvalidImage;
	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\GuzzleException;
	use GuzzleHttp\RequestOptions;
	use ImageGallery\ImageGalleryNameEmpty;
	use ImageGallery\ImageGalleryService;
	use ImageGallery\ImageGalleryWithSameNameExists;
	use ImageManager\Exception\InvalidDirectory;
	use ImageManager\ImageManagerService;
	use MonologWrapper\MonologWrapper;
	use Nox\Http\FileUploadPayload;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Page\Page;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageService;
	use Page\PageType;
	use Page\PublicationStatus;
	use PageEditor\Exceptions\NoPageFoundWithID;
	use PageEditor\PageEditorService;
	use ProjectPostTag\ProjectPostTag;
	use Settings\Setting;
	use Settings\Settings;
	use System\Layouts;
	use System\System;
	use System\Themes;
	use Uplift\Exceptions\MalformedValue;
	use Uplift\Exceptions\NoObjectFound;
	use Uplift\ImageManager\ImageFileAlreadyExists;
	use Uplift\ImageManager\ImageIsAThumb;
	use Uplift\ImageManager\ImageManager;
	use Uplift\ImageManager\ImageMissingThumb;
	use Uplift\ImageProcessing\Exceptions\ImageProcessingException;
	use Uplift\ImageProcessing\ImageProcessing;

	class ClientProjectFormService{
		/**
		 * The threshold, in bytes, of an image the client sends via the IPP
		 * form that would trigger us to resize it on the remote image processing server.
		 */
		public const CLIENT_UPLOADED_IMAGE_FILE_SIZE_THRESHOLD = 1000000;

		const COVER_PHOTO_PLACEHOLDER = "__COVER_PHOTO_PLACEHOLDER__";

		/**
		 * @throws ProjectValidationError
		 */
		private static function validateRequiredProjectData(
			string $projectTitle,
			string $projectDescription,
			int $projectCityPageID,
			string $projectLocationCity,
			string $projectLocationStateShorthand,
			string $projectLocationState,
		): void{

			$projectTitle = trim($projectTitle);
			$projectDescription = trim($projectDescription);
			$projectLocationCity = trim($projectLocationCity);
			$projectLocationStateShorthand = trim($projectLocationStateShorthand);
			$projectLocationState = trim($projectLocationState);

			if (empty($projectTitle)){
				throw new ProjectValidationError("Your project title cannot be empty.");
			}

			if (strlen($projectTitle) < 10){
				throw new ProjectValidationError("Your project title is too short.");
			}

			if (strlen($projectTitle) > 255){
				throw new ProjectValidationError("Your project title is too long.");
			}

			$wordsInTitle = explode(" ", $projectTitle);
			if (count($wordsInTitle) <= 3){
				throw new ProjectValidationError("Your project title should contain more than three (3) words.");
			}

			if (empty($projectDescription)){
				throw new ProjectValidationError("Your project description cannot be empty.");
			}

			if (strlen($projectDescription) < 30){
				throw new ProjectValidationError("Your project description is too short.");
			}

			$wordsInDescription = explode(" ", $projectDescription);
			if (count($wordsInDescription) <= 20){
				throw new ProjectValidationError("Your project description should contain more than 20 words.");
			}

			// Only check custom locations if a city page ID isn't set
			if ($projectCityPageID === 0){
				throw new ProjectValidationError("Please select a location option from the dropdown menu.");
			}elseif ($projectCityPageID === -1) {
				if (empty($projectLocationCity)) {
					throw new ProjectValidationError("You must fill out the project's city location.");
				}

				if (empty($projectLocationStateShorthand)) {
					throw new ProjectValidationError("You must fill out the project's state location.");
				}

				if (strlen($projectLocationStateShorthand) !== 2) {
					throw new ProjectValidationError("The project's state location should only be two letters - like TX and not Texas.");
				}

				if (empty($projectLocationState)) {
					throw new ProjectValidationError("You must fill out the project's full state location.");
				}

				if (strlen($projectLocationState) === 2) {
					throw new ProjectValidationError("The state's full name should not be an abbreviation. Please check you have entered the state's full state name in the proper field and not in the abbreviation field.");
				}
			}else{
				// Check if the page exists
				/** @var ?Page $cityPage */
				$cityPage = Page::fetch($projectCityPageID);
				if ($cityPage === null){
					throw new ProjectValidationError("The city page location you have chosen from the dropdown doesn't exist in the system. This is an error that you should report to support@footbridgemedia.com");
				}
			}

		}

		/**
		 * The projectDescription should be sanitized before calling this function
		 */
		private static function getHTMLFormattedProjectContent(
			string $projectDescription,
			string $projectBudget,
		): string{
			$projectBudget = trim($projectBudget);

			// Break up all newlines into paragraphs
			$paragraphs = explode("\n", $projectDescription);
			$finalHTML = "";
			foreach($paragraphs as $index=>$paragraph){
				$paragraph = trim($paragraph);
				if (!empty($paragraph)){
					$finalHTML .= sprintf("\t<p>\n\t\t%s\n\t</p>\n", $paragraph);
				}
			}

			if (strlen($projectBudget) > 0) {
				$finalHTML .= sprintf("\t<p>\n\t\t<strong>Budget: </strong><span>%s</span>\n\t</p>", $projectBudget);
			}

			return $finalHTML;
		}

		/**
		 * Adds a new paragraph and city page link to the project content. The modified string is then returned.
		 */
		private static function addCityPageParagraphAndLinkToContent(string $projectContent, Page $cityPage): string{
			return sprintf(
				"%s\n\t<p>\n\t\t<strong>Location: </strong><a href=\"%s\">%s</a>\n\t</p>",
				$projectContent,
				$cityPage->pageRoute,
				$cityPage->pageName
			);
		}

		/**
		 * @throws ProjectValidationError
		 * @throws NoObjectFound
		 */
		public static function createProjectPost(
			string $projectTitle,
			string $projectDescription,
			int $projectCityPageID,
			string $projectLocationCity,
			string $projectLocationStateShorthand,
			string $projectLocationState,
			int $projectPrimaryServicePageID,
			string $projectBudget,
			string $projectBrandsMaterials,
			bool $didLeaveCustomerReview,
			string $customerReviewFirstName,
			string $customerReviewLastName,
			string $customerReviewTestimonial,
			array $projectTagIds,
		): Page{
			self::validateRequiredProjectData(
				projectTitle: $projectTitle,
				projectDescription: $projectDescription,
				projectCityPageID: $projectCityPageID,
				projectLocationCity: $projectLocationCity,
				projectLocationStateShorthand: $projectLocationStateShorthand,
				projectLocationState: $projectLocationState,
			);

			// Strip HTML characters from raw content
			$projectDescription = htmlspecialchars($projectDescription);

			// Find the service page provided by the user if provided
			$primaryServicePage = null;
			if ($projectPrimaryServicePageID > 0){
				/** @var ?Page $primaryServicePage */
				$primaryServicePage = Page::fetch($projectPrimaryServicePageID);
				if ($primaryServicePage === null){
					throw new NoObjectFound("The primary service selected doesn't have a page associated with it. Please contact support@footbridgemedia.com with this error message for them to resolve it for you in order to post projects.");
				}else{
					$serviceLinkedProjectDescription = self::addServiceLinkToContent(
						projectPostBody: $projectDescription,
						serviceName: $primaryServicePage->pageName,
						servicePageLink: $primaryServicePage->pageRoute,
					);

					$htmlDescription = self::getHTMLFormattedProjectContent($serviceLinkedProjectDescription, $projectBudget);
				}
			}else{
				$htmlDescription = self::getHTMLFormattedProjectContent($projectDescription, $projectBudget);
			}

			$htmlTitle = sprintf("<h1>%s</h1>", htmlspecialchars($projectTitle));

			// Avoid duplicate page names
			$attempt = 0;
			$pageTitle = $projectTitle;
			do {
				/** @var ?Page $existingPageWithProjectTile */
				$existingPageWithProjectTile = Page::queryOne(
					columnQuery: (new ColumnQuery())
						->where("pageName", "=", $pageTitle)
				);

				if ($existingPageWithProjectTile !== null){
					++$attempt;

					$pageTitle = $projectTitle . " ($attempt)";
				}
			}while($existingPageWithProjectTile !== null);

			$metaDescriptionContent = htmlspecialchars(self::getMetaDescriptionFromProjectDescription($projectDescription));

			// Find the project reel index page
			$projectsIndexPage = PageService::getProjectsIndexPage();
			if ($projectsIndexPage === null){
				// Can't continue, we'll have a bad time
				throw new NoObjectFound("This CMS is missing a project's index page. Please contact support@footbridgemedia.com with this error message for them to resolve it for you in order to post projects.");
			}

			// If the city page ID is > 0, then add a city page link to the content
			$cityPage = null;
			if ($projectCityPageID > 0){
				/** @var ?Page $cityPage */
				$cityPage = Page::fetch($projectCityPageID);
				$htmlDescription = self::addCityPageParagraphAndLinkToContent($htmlDescription, $cityPage);

				// Fill in the project location variables using the selected city page
				$projectLocationCity = $cityPage->getPageData(PageDatas::CITY_NAME)->value;
				$projectLocationState = $cityPage->getPageData(PageDatas::STATE_NAME)->value;
				$projectLocationStateShorthand = $cityPage->getPageData(PageDatas::STATE_NAME_SHORTHAND)->value;
			}

			$formattedBody = sprintf("%s\n<div class=\"clearfix\">\n%s\n%s\n</div>", $htmlTitle, self::COVER_PHOTO_PLACEHOLDER, $htmlDescription);

			// Find a Project layout
			$fsFiles = Layouts::getAvailableLayouts();
			$layoutName = null;

			foreach($fsFiles as $fsFile){
				if (str_contains(strtolower($fsFile->fileNameWithoutExtension), "project")){
					$layoutName = $fsFile->fileNameWithoutExtension;
					break;
				}
			}

			if ($layoutName === null){
				throw new NoObjectFound("This system has no layout file that contains Project in its name. Please report this error to support@footbridgemedia.com so they can resolve it and allow you to publish project posts.");
			}

			$projectPage = new Page();
			$projectPage->pageName = $pageTitle;
			$projectPage->pageBody = $formattedBody;
			$projectPage->pageHead = sprintf("<title>%s</title>\n", htmlspecialchars($projectTitle));
			$projectPage->pageHead .= sprintf("<meta name=\"description\" property=\"og:description\" content=\"%s\">\n", $metaDescriptionContent);
			$projectPage->pageLayout = $layoutName;
			$projectPage->publicationStatus = PublicationStatus::Published->value;
			$projectPage->publicationTimestamp = time();
			$projectPage->pageType = PageType::Project->name;

			$projectURL = sprintf("%s/%s", $projectsIndexPage->pageRoute, $projectPage->generateURLSlugFromPageName());
			$projectPage->pageRoute = $projectURL;

			$projectPage->save();

			// Save the page body content to the first layout section
			// if this page has one
			$projectPage->saveToDefaultContentSectionOrFirstSection($formattedBody);

			// Save the budget, and the brands/materials
			$projectBrandsMaterialsData = new PageData();
			$projectBrandsMaterialsData->pageID = $projectPage->id;
			$projectBrandsMaterialsData->name = PageDatas::PROJECT_BRANDS_PRODUCTS->name;
			$projectBrandsMaterialsData->value = $projectBrandsMaterials;
			$projectBrandsMaterialsData->save();

			// Set the project location
			$projectLocationCityData = new PageData();
			$projectLocationCityData->pageID = $projectPage->id;
			$projectLocationCityData->name = PageDatas::CITY_NAME->name;
			$projectLocationCityData->value = $projectLocationCity;
			$projectLocationCityData->save();

			$projectLocationStateNameShorthandData = new PageData();
			$projectLocationStateNameShorthandData->pageID = $projectPage->id;
			$projectLocationStateNameShorthandData->name = PageDatas::STATE_NAME_SHORTHAND->name;
			$projectLocationStateNameShorthandData->value = $projectLocationStateShorthand;
			$projectLocationStateNameShorthandData->save();

			$projectLocationStateNameShorthandData = new PageData();
			$projectLocationStateNameShorthandData->pageID = $projectPage->id;
			$projectLocationStateNameShorthandData->name = PageDatas::STATE_NAME->name;
			$projectLocationStateNameShorthandData->value = $projectLocationState;
			$projectLocationStateNameShorthandData->save();

			$customerLeftReviewData = new PageData();
			$customerLeftReviewData->pageID = $projectPage->id;
			$customerLeftReviewData->name = PageDatas::CUSTOMER_DID_LEAVE_REVIEW->name;
			$customerLeftReviewData->value = $didLeaveCustomerReview ? 1 : 0;
			$customerLeftReviewData->save();

			// Create breadcrumbs for this page
			$crumbs = [
				[
					"label"=>"Home",
					"uri"=>"/",
				],
				[
					"label"=>$projectsIndexPage->pageName,
					"uri"=>$projectsIndexPage->pageRoute,
				],
				[
					"label"=>$projectTitle,
					"uri"=>$projectURL,
				],
			];

			PageEditorService::saveBreadcrumbs($projectPage, $crumbs);

			if ($didLeaveCustomerReview === true){
				// Save the customer name and customer testimonial
				$customerReviewFirstNameData = new PageData();
				$customerReviewFirstNameData->pageID = $projectPage->id;
				$customerReviewFirstNameData->name = PageDatas::CUSTOMER_REVIEW_FIRST_NAME->name;
				$customerReviewFirstNameData->value = $customerReviewFirstName;
				$customerReviewFirstNameData->save();

				$customerReviewLastNameData = new PageData();
				$customerReviewLastNameData->pageID = $projectPage->id;
				$customerReviewLastNameData->name = PageDatas::CUSTOMER_REVIEW_LAST_NAME->name;
				$customerReviewLastNameData->value = $customerReviewLastName;
				$customerReviewLastNameData->save();

				$customerReviewTestimonialData = new PageData();
				$customerReviewTestimonialData->pageID = $projectPage->id;
				$customerReviewTestimonialData->name = PageDatas::CUSTOMER_REVIEW_TESTIMONIAL->name;
				$customerReviewTestimonialData->value = $customerReviewTestimonial;
				$customerReviewTestimonialData->save();
			}

			// Tag the project with a service tag and a location tag

			if ($projectPrimaryServicePageID > 0) {
				/** @var ProjectPostTag $serviceTag */
				$serviceTag = ProjectPostTag::queryOne(
					columnQuery: (new ColumnQuery())
						->where("label", "=", $primaryServicePage->pageName)
				);

				// Create the tags if they don't exist
				if ($serviceTag === null) {
					$serviceTag = new ProjectPostTag();
					$serviceTag->label = $primaryServicePage->pageName;
					$serviceTag->save();
				}

				// Add recent projects shortcode to the service page
				if ($primaryServicePage !== null) {
					self::addRecentProjectsShortcodeToServicePage($serviceTag->id, $primaryServicePage);
				}

				$serviceProjectTagPageData = new PageData();
				$serviceProjectTagPageData->name = PageDatas::PROJECT_TAG->name;
				$serviceProjectTagPageData->pageID = $projectPage->id;
				$serviceProjectTagPageData->value = $serviceTag->id;
				$serviceProjectTagPageData->save();
			}

			$cityState = sprintf("%s, %s", $projectLocationCity, $projectLocationStateShorthand);

			/** @var ProjectPostTag $locationTag */
			$locationTag = ProjectPostTag::queryOne(
				columnQuery: (new ColumnQuery())
					->where("label", "=", $cityState)
			);

			if ($locationTag === null){
				$locationTag = new ProjectPostTag();
				$locationTag->label = $cityState;
				$locationTag->save();
			}

			// Add a recent projects shortcode to the city page
			if ($cityPage !== null){
				self::addRecentProjectsShortcodeToCityPage($locationTag->id, $cityPage);
			}

			$locationProjectTagPageData = new PageData();
			$locationProjectTagPageData->name = PageDatas::PROJECT_TAG->name;
			$locationProjectTagPageData->pageID = $projectPage->id;
			$locationProjectTagPageData->value = $locationTag->id;
			$locationProjectTagPageData->save();

			// Individual project tags
			// Do not add the tag if it is equal to the local tag Id or the service tag Id
			foreach($projectTagIds as $tagId){
				if ($tagId === $locationTag->id || (isset($serviceTag) && $tagId === $serviceTag->id)){
					continue;
				}

				$projectTagData = new PageData();
				$projectTagData->name = PageDatas::PROJECT_TAG->name;
				$projectTagData->value = $tagId;
				$projectTagData->pageID = $projectPage->id;
				$projectTagData->save();
			}

			return $projectPage;
		}

		/**
		 * Creates a new directory in the theme's image directory for this project title.
		 * Will return the new, full path of the image directory.
		 *
		 * All created directories will be put under a IMAGES_DIRECTORY/projects directory. If there is no
		 * directory for /projects in the IMAGES_DIRECTORY then one will be created first.
		 * @throws \ImageManager\Exception\MissingParameter
		 * @throws ProjectValidationError
		 * @throws \FileSystemUtilities\Exceptions\PathDoesntExist
		 * @throws \FileSystemUtilities\Exceptions\InvalidFolderName
		 * @throws \FileSystemUtilities\exceptions\MaximumNewFolderDepthExceeded
		 * @throws InvalidDirectory
		 * @throws \FileSystemUtilities\Exceptions\NewDirectoryWithSameNameExists
		 */
		public static function createImageDirectoryForProject(
			string $projectTitle,
		): string{
			$rootImagesDirectory = realpath(ImageManager::IMAGES_DIRECTORY);
			$projectsImagesDirectory = sprintf("%s/projects", $rootImagesDirectory);

			if (!file_exists($projectsImagesDirectory)){
				$newFolder = ImageManagerService::newDirectory($rootImagesDirectory);
				$projectsImagesDirectory = ImageManagerService::renameDirectory($newFolder, "projects");
			}

			// Fetch all characters that are only dashes, underscores, numbers, or letters. Any spaces
			// replace with a dash. This will generate a directory for this project name
			$newProjectDirectoryName = "";
			foreach(str_split($projectTitle) as $char){
				if ($char === " "){
					$newProjectDirectoryName .= "-";
				}else{
					if (preg_match("/[\d_\-a-z]/i", $char) === 1){
						$newProjectDirectoryName .= $char;
					}
				}

				// Don't let the directory name get too long
				if (strlen($newProjectDirectoryName) > 25){
					break;
				}
			}

			// Handle when more than one dash or underscore happens consecutively
			$newProjectDirectoryName = preg_replace(
				pattern: "/[_-]{2,}/",
				replacement: "",
				subject: $newProjectDirectoryName,
			);

			// Lowercase the entire new directory name
			$newProjectDirectoryName = strtolower($newProjectDirectoryName);

			if (empty($newProjectDirectoryName)){
				throw new ProjectValidationError("The project title contains too many invalid characters and no directory can be made with that project title. The resulting folder name is empty.");
			}

			// Check if a directory with the possible new name exists.
			$newDirectoryFullPath = sprintf("%s/%s", $projectsImagesDirectory, $newProjectDirectoryName);
			if (file_exists($newDirectoryFullPath)){
				// Add the current unix time to the directory path
				if (str_ends_with(haystack: $newProjectDirectoryName, needle:"-")){
					$newProjectDirectoryName .= time();
				}else {
					$newProjectDirectoryName .= "-" . time();
				}
			}

			$newProjectFolder = ImageManagerService::newDirectory($projectsImagesDirectory);
			return ImageManagerService::renameDirectory($newProjectFolder, $newProjectDirectoryName);
		}

		/**
		 * Uploads the cover photo and places it in the project posts' content. The project post must have a
		 * placeholder text defined by self::COVER_PHOTO_PLACEHOLDER in its pageBody
		 * @param FileUploadPayload $coverPhoto
		 * @param int $projectPageID
		 * @param string $projectImagesDirectory
		 * @return void
		 * @throws InvalidDirectory
		 * @throws InvalidFileName
		 * @throws AnimatedWebPNotSupported
		 * @throws InvalidImage
		 * @throws FileNotFound
		 * @throws ImageFileAlreadyExists
		 * @throws ImageIsAThumb|ImageProcessingException
		 */
		public static function uploadCoverPhotoAndPlaceInContent(
			FileUploadPayload $coverPhoto,
			int $projectPageID,
			string $projectImagesDirectory,
		): void{
			$fileName = ContentHelper::stripAccents($coverPhoto->fileName);
			$fullNewFilePath = sprintf("%s/%s", $projectImagesDirectory, $fileName);
			$imageProcessing = new ImageProcessing();

			// We need to check that the file size. If it is too large, then we'll ship it off
			// to be resized.
			if ($coverPhoto->fileSize > self::CLIENT_UPLOADED_IMAGE_FILE_SIZE_THRESHOLD){
				// The cover photo needs to be resized
				$coverPhoto->contents = $imageProcessing->resize(
					basename($fullNewFilePath),
					$coverPhoto->contents,
					1080,
					null
				);

				// If the cover photo was an HEIC, our image processing server will have sent it back as
				// a JPEG
				if (strtolower(pathinfo($fullNewFilePath, PATHINFO_EXTENSION)) === "heic") {
					$coverPhoto->fileName = pathinfo($fullNewFilePath, PATHINFO_FILENAME) . ".jpg";
					$fullNewFilePath = dirname($fullNewFilePath) . "/" . $coverPhoto->fileName;
				}
			}

			// If the image is in HEIC format, then we need to convert it to JPEG
			if (strtolower(pathinfo($fullNewFilePath, PATHINFO_EXTENSION)) === "heic"){
				$coverPhoto->contents = $imageProcessing->convertImageType(
					basename($fullNewFilePath),
					$coverPhoto->contents,
					ImageProcessing::IMAGE_EXTENSION_TO_MIME_MAP["jpeg"]
				);

				// Set the name file name and path
				$coverPhoto->fileName = pathinfo($fullNewFilePath, PATHINFO_FILENAME) . ".jpg";
				$fullNewFilePath = dirname($fullNewFilePath) . "/" . $coverPhoto->fileName;
			}

			$fsImage = ImageManagerService::uploadImage(
				filePath: $fullNewFilePath,
				contents:$coverPhoto->contents,
				overrideExistingFile: true,
			);

			// Place the image in the content
			/** @var Page $projectPost */
			$projectPost = Page::fetch($projectPageID);

			// Get a safe alt attribute to use from the project title. Remove newlines, tabs, quotes
			$altAttribute = preg_replace("/[\n\t\r\"'<>\[\]\(\)]/", "", $projectPost->pageName);

			$coverPhotoHTML = sprintf('%s<img alt="%s" src="%s" class="img-r-dynamic" width="450">', "\t", $altAttribute, str_replace(" ", "%20", $fsImage->uri));

			$projectPost->pageBody = str_replace(
				search: self::COVER_PHOTO_PLACEHOLDER,
				replace:$coverPhotoHTML,
				subject: $projectPost->pageBody
			);
			$projectPost->save();

			// Add the featured image data
			$featuredImageData = new PageData();
			$featuredImageData->pageID = $projectPageID;
			$featuredImageData->name = PageDatas::FEATURED_IMAGE->name;
			$featuredImageData->value = $fsImage->uri;
			$featuredImageData->save();

			$featuredImageThumbData = new PageData();
			$featuredImageThumbData->pageID = $projectPageID;
			$featuredImageThumbData->name = PageDatas::FEATURED_IMAGE_THUMB->name;
			$featuredImageThumbData->value = $fsImage->thumbURI;
			$featuredImageThumbData->save();
		}

		/**
		 * Uploads an image to the directory provided. Returns the file name that was accepted for uploading
		 * as it could be modified in this method.
		 * @param FileUploadPayload $image
		 * @param string $projectImagesDirectory
		 * @return string
		 * @throws InvalidDirectory
		 * @throws InvalidFileName
		 * @throws AnimatedWebPNotSupported
		 * @throws InvalidImage
		 * @throws FileNotFound
		 * @throws ImageFileAlreadyExists
		 * @throws ImageIsAThumb|ImageProcessingException
		 */
		public static function uploadProjectImage(
			FileUploadPayload $image,
			string $projectImagesDirectory,
		): string{
			$fileName = ContentHelper::stripAccents($image->fileName);
			$fullNewFilePath = sprintf("%s/%s", $projectImagesDirectory, $fileName);
			$imageProcessing = new ImageProcessing();

			// We need to check that the file size. If it is too large, then we'll ship it off
			// to be resized.
			if ($image->fileSize > self::CLIENT_UPLOADED_IMAGE_FILE_SIZE_THRESHOLD){
				// The photo needs to be resized
				$image->contents = $imageProcessing->resize(
					basename($fullNewFilePath),
					$image->contents,
					1080,
					null
				);

				// If the photo was an HEIC, our image processing server will have sent it back as
				// a JPEG
				if (strtolower(pathinfo($fullNewFilePath, PATHINFO_EXTENSION)) === "heic") {
					$image->fileName = pathinfo($fullNewFilePath, PATHINFO_FILENAME) . ".jpg";
					$fullNewFilePath = dirname($fullNewFilePath) . "/" . $image->fileName;
				}
			}

			// If the image is in HEIC format, then we need to convert it to JPEG
			if (strtolower(pathinfo($fullNewFilePath, PATHINFO_EXTENSION)) === "heic"){
				$image->contents = $imageProcessing->convertImageType(
					basename($fullNewFilePath),
					$image->contents,
					ImageProcessing::IMAGE_EXTENSION_TO_MIME_MAP["jpeg"]
				);

				// Set the name file name and path
				$image->fileName = pathinfo($fullNewFilePath, PATHINFO_FILENAME) . ".jpg";
				$fullNewFilePath = dirname($fullNewFilePath) . "/" . $image->fileName;
			}

			ImageManagerService::uploadImage(
				filePath: $fullNewFilePath,
				contents:$image->contents,
				overrideExistingFile: true,
			);

			return basename($fullNewFilePath);
		}

		/**
		 * @param int $projectPostID
		 * @param string $projectImagesDirectory
		 * @param string[] $listOfImageFileNames Ordered list of string image file names to put into the gallery
		 * @throws ImageGalleryNameEmpty
		 * @throws ImageGalleryWithSameNameExists
		 * @throws AnimatedWebPNotSupported
		 * @throws FileNotFound
		 * @throws InvalidImage|ImageMissingThumb
		 */
		public static function addProjectGalleryToPost(
			int $projectPostID,
			string $projectImagesDirectory,
			array $listOfImageFileNames
		): void{
			/** @var Page $projectPost */
			$projectPost = Page::fetch($projectPostID);

			// Create a gallery for this IPP
			$galleryName = $projectPost->pageName;

			try {
				$gallery = ImageGalleryService::createNewGallery(
					galleryName: $galleryName,
				);
			} catch (ImageGalleryWithSameNameExists $e) {
				// If this happens, add the UNIX timestamp to the gallery name
				$galleryName .= "-" . time();
				$gallery = ImageGalleryService::createNewGallery(
					galleryName: $galleryName,
				);
			}

			// Scan the images directory for any files and add them to the gallery
			$files = array_diff(scandir($projectImagesDirectory), [".",".."]);
			$imageArraysToAddToGallery = [];
			$position = 0;
			foreach ($listOfImageFileNames as $imageFileName){
				// Find it in the $files array
				$index = array_search($imageFileName, $files);
				if ($index !== false){
					// Found the image in the directory
					// Add it to the gallery
					$fullPath = realpath(sprintf("%s/%s", $projectImagesDirectory, $imageFileName));
					if (!is_dir($fullPath)){
						$fsImage = new FSImageFile(
							fileName: $imageFileName,
							fullFilePath:$fullPath,
							uri:ImageManager::getImageURIFromFilePath($fullPath),
							acceptSVG:false,
						);

						$imageArraysToAddToGallery[] = [
							"imageURI"=>$fsImage->uri,
							"thumbURI"=>$fsImage->thumbURI,
							"altText"=>"",
							"position"=>$position,
						];
					}
					++$position;
				}
			}

			ImageGalleryService::addMembersToGallery(
				galleryID: $gallery->id,
				images:$imageArraysToAddToGallery,
			);

			// Add this gallery to the project content
			$projectPost->pageBody .= sprintf(
				"\n\n<h2>Project Image Gallery</h2>\n{{ gallery id=\"%d\" gallery-name=\"%s\" }}",
				$gallery->id,
				$gallery->getShortcodeAttributeSafeGalleryName()
			);
			$projectPost->save();
		}

		/**
		 * @throws GuzzleException
		 * @throws MalformedValue
		 */
		public static function submitProjectPostSubmissionToMasterServer(
			int $projectPostID
		): void{
			/** @var Page $projectPage */
			$projectPage = Page::fetch($projectPostID);
			$client = new Client();
			$uuid = Setting::getSettingValue(Settings::BUILD_UUID->value);
			$apiKey = Setting::getSettingValue(Settings::UPLIFT_CONTROL_PANEL_API_KEY->value);
			$panelHost = System::getUpliftControlPanelHost();
			$endpoint = sprintf("%s/uplift/build/ipp/queue", $panelHost);

			$client->request(
				method:"POST",
				uri:$endpoint,
				options:[
					RequestOptions::HEADERS => [
						"Uplift-UUID"=>$uuid,
						"Uplift-API-Key"=>$apiKey,
					],
					RequestOptions::MULTIPART=>[
						[
							"name"=>"pageID",
							"contents"=>$projectPostID,
						],
						[
							"name"=>"pageName",
							"contents"=>$projectPage->pageName,
						],
					],
				],
			);
		}

		/**
		 * @throws NoPrimaryKey
		 * @throws NoPageFoundWithID
		 */
		public static function undoProjectPostSubmission(
			int $projectPostID,
		): void{
			PageEditorService::deletePage($projectPostID);
		}

		/**
		 * Extracts a meta description from the project body by taking X amount of characters from the description
		 * after newlines, tabs, and returns have been removed. Additionally, double quotes are replaced with single
		 * quotes so to not interfere with the HTML quotes wrapping the meta description content attribute value.
		 * @param string $projectDescription
		 * @return string
		 */
		private static function getMetaDescriptionFromProjectDescription(
			string $projectDescription,
		): string{
			$projectDescriptionWhitespaceStripped = str_replace('"', "'", $projectDescription);
			$projectDescriptionWhitespaceStripped = str_replace("\t", "", $projectDescriptionWhitespaceStripped);
			$projectDescriptionWhitespaceStripped = str_replace("\n", " ", $projectDescriptionWhitespaceStripped);
			$projectDescriptionWhitespaceStripped = str_replace("\r", "", $projectDescriptionWhitespaceStripped);

			// Max length is currently 155 characters
			return substr($projectDescriptionWhitespaceStripped, 0, 154);
		}

		/**
		 * Returns a new string that has taken the provided projectPostBody and added a service link
		 * to the content in some form. The link is added by finding the service page name in the content
		 * or adding a new paragraph as necessary.
		 */
		public static function addServiceLinkToContent(
			string $projectPostBody,
			string $serviceName,
			string $servicePageLink,
		): string{

			$replacementLink = sprintf("<a href=\"%s\">$1</a>", $servicePageLink);

			// Escape the serviceName for injection into a regular expression
			$regexEscapedServiceName = preg_quote($serviceName);

			// Replace instances of the delimiter "/" of the regex in the serviceName with an escape sequence
			$regexEscapedServiceName = str_replace("/", "\\/", $regexEscapedServiceName);

			$linkedContent = preg_replace(
				pattern: sprintf("/(%s)/i", $regexEscapedServiceName),
				replacement:$replacementLink,
				subject: $projectPostBody,
				limit: 1
			);

			// Did a replacement happen?
			if ($linkedContent === $projectPostBody){
				// No replacement happened
				// Append it to the bottom

				$appendedContent = sprintf(
					"<strong>Service provided:</strong> <a href=\"%s\">%s</a>",
					$servicePageLink,
					$serviceName,
				);

				return sprintf("%s\n\n%s", $projectPostBody, $appendedContent);
			}else{
				return $linkedContent;
			}
		}

		/**
		 * Checks if a "get-recent-projects" shortcode exists in the body of the cityPage. If it doesn't,
		 * then injects one and an H2 at the bottom of the content.
		 *
		 * If a shortcode already exists, will attempt to find a match for `included-project-tags="[]"` in some form,
		 * and inject the new project tag into the JSON array. If it cannot find a match for included-project-tags,
		 * then this function will do nothing.
		 *
		 * This process will also check the contents of the page's layout for any indication of the shortcode.
		 */
		private static function addRecentProjectsShortcodeToCityPage(
			int $projectTagID,
			Page $cityPage
		): void{
			$logger = MonologWrapper::getLogger();
			$pageBody = $cityPage->pageBody;

			// Check if the layout has the get-recent-projects shortcode
			try {
				$layoutBody = Layouts::getLayoutFileContentsFromPageLayoutName($cityPage->pageLayout);
				if (str_contains($layoutBody, "{{ get-recent-projects")){
					// The layout contains the shortcode, do nothing in this case as the layout is a PHP file and not necessarily
					// easy to edit without damaging syntax.
					return;
				}
			}catch(NoObjectFound){
				// No layout? Weird
				$logger->warning("$cityPage->pageName (ID: $cityPage->id) has layout set to $cityPage->pageLayout, but that layout doesn't exist.");
			}

			if (str_contains($pageBody, "{{ get-recent-projects")){
				// Try to find and replace the attribute defining project tags
				$pageBody = self::addProjectTagToRecentProjectsShortcode($projectTagID, $pageBody);
			}else{
				// Neither the page nor the layout contains the shortcode. Inject it at the bottom of the page body after
				// and H2
				$pageBody .= sprintf("\n<h2>Recent Projects in %s</h2>\n", $cityPage->pageName);
				$pageBody .= sprintf(
					'{{ get-recent-projects num-projects="2" columns="1" included-project-tags="[%d]" autofill="0" }}',
					$projectTagID
				);
			}

			$cityPage->pageBody = $pageBody;
			$cityPage->save();
		}

		/**
		 * Checks if a "get-recent-projects" shortcode exists in the body of the servicePage. If it doesn't,
		 * then injects one and an H2 at the bottom of the content.
		 *
		 * If a shortcode already exists, will attempt to find a match for `included-project-tags="[]"` in some form,
		 * and inject the new project tag into the JSON array. If it cannot find a match for included-project-tags,
		 * then this function will do nothing.
		 *
		 * This process will also check the contents of the page's layout for any indication of the shortcode.
		 */
		private static function addRecentProjectsShortcodeToServicePage(
			int $projectTagID,
			Page $servicePage
		): void{
			$logger = MonologWrapper::getLogger();
			$pageBody = $servicePage->pageBody;

			// Check if the layout has the get-recent-projects shortcode
			try {
				$layoutBody = Layouts::getLayoutFileContentsFromPageLayoutName($servicePage->pageLayout);
				if (str_contains($layoutBody, "{{ get-recent-projects")){
					// The layout contains the shortcode, do nothing in this case as the layout is a PHP file and not necessarily
					// easy to edit without damaging syntax.
					return;
				}
			}catch(NoObjectFound){
				// No layout? Weird
				$logger->warning("$servicePage->pageName (ID: $servicePage->id) has layout set to $servicePage->pageLayout, but that layout doesn't exist.");
			}

			if (str_contains($pageBody, "{{ get-recent-projects")){
				// Try to find and replace the attribute defining project tags
				$pageBody = self::addProjectTagToRecentProjectsShortcode($projectTagID, $pageBody);
			}else{
				// Neither the page nor the layout contains the shortcode. Inject it at the bottom of the page body after
				// and H2
				$pageBody .= sprintf("\n<h2>Recent %s Projects</h2>\n", $servicePage->pageName);
				$pageBody .= sprintf(
					'{{ get-recent-projects num-projects="2" columns="1" included-project-tags="[%d]" autofill="0" }}',
					$projectTagID
				);
			}

			$servicePage->pageBody = $pageBody;
			$servicePage->save();
		}

		/**
		 * Uses PCRE to find the get-recent-projects shortcode and then find the included-project-tags attribute. Then,
		 * replaces the included-project-tags with a new set of IDs that include the $projectTagID - only if
		 * it isn't already present.
		 */
		public static function addProjectTagToRecentProjectsShortcode(int $projectTagID, string $pageContent): string{
			return preg_replace_callback(
				'/{{ get-recent-projects.*included-project-tags="(?<projectTagIDs>[^\"]+)".*?}}/i',
				function(array $matches) use ($projectTagID){
					$fullMatch = $matches[0];

					$projectTags = json_decode($matches['projectTagIDs'], true);

					if ($projectTags === null){
						$projectTags = [];
					}

					if (!in_array($projectTagID, $projectTags)){
						// Add it
						$projectTags[] = $projectTagID;

						// Replace the content and specifically replace the included-project-tags
						return preg_replace(
							"/included-project-tags=\"[^\"]*\"/i",
							sprintf('included-project-tags="%s"', json_encode($projectTags)),
							$fullMatch
						);
					}

					return $fullMatch;
				},
				$pageContent
			);
		}

		/**
		 * @throws InvalidFileName
		 */
		public static function validateProjectPhotoFileName(FileUploadPayload $image): void{
			$fileNameBase = basename($image->fileName);

			if (!FileSystemUtilities::isNameFileSafe($fileNameBase)){
				throw new InvalidFileName("The file $image->fileName contains invalid characters in the file name. You may only have numbers, letters, underscores, or dashes in the file name before the file extension.");
			}
		}

	}