<?php

	namespace ArticleCategory;

	use Accounts\Account;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\ORM\ColumnQuery;
	use Page\ArticleCategory;
	use Page\PageData;
	use Page\PageDatas;
	use System\HttpHelper;

	class ArticleCategoryService
	{
		/**
		 * @throws CategoryWithSameNameExists
		 * @throws CategoryNameIsBlank
		 */
		public static function createCategory(
			string $categoryName,
		): ArticleCategory{

			$categoryName = trim($categoryName);

			if (strlen($categoryName) === 0){
				throw new CategoryNameIsBlank("New article category cannot be blank.");
			}

			/** @var ArticleCategory | null $existingCategoryWithSameName */
			$existingCategoryWithSameName = ArticleCategory::queryOne(
				columnQuery: (new ColumnQuery())
					->where("name", "=", $categoryName),
			);

			if ($existingCategoryWithSameName !== null) {
				throw new CategoryWithSameNameExists("{$categoryName} already exists as an article category.");
			}

			$newCategory = new ArticleCategory();
			$newCategory->name = $categoryName;
			$newCategory->save();

			ActivityLog::log(
				categoryID: ActivityLogCategories::CREATE_ARTICLE_CATEGORY->value,
				accountID: (Account::getCurrentUser())->id,
				ip: HttpHelper::getIPFromRequest(),
				jsonData: json_encode([
					"newCategoryID"=>$newCategory->id,
					"newCategoryName"=>$categoryName,
				]),
			);

			return $newCategory;
		}

		/**
		 * @throws CategoryWithSameNameExists
		 * @throws CategoryNameIsBlank
		 */
		public static function renameCategory(
			ArticleCategory $articleCategory,
			string $newName,
		): void{

			if (strlen($newName) === 0){
				throw new CategoryNameIsBlank("New article category cannot be blank.");
			}

			/** @var ArticleCategory | null $existingCategoryWithSameName */
			$existingCategoryWithSameName = ArticleCategory::queryOne(
				columnQuery: (new ColumnQuery())
					->where("name", "=", $newName)
					->and()
					->where("id","!=",$articleCategory->id),
			);

			if ($existingCategoryWithSameName !== null) {
				throw new CategoryWithSameNameExists("{$newName} already exists as an article category.");
			}

			$articleCategory->name = $newName;
			$articleCategory->save();
		}

		public static function deleteCategory(
			int $categoryID,
		): void{
			/** @var ArticleCategory | null $articleCategory */
			$articleCategory = ArticleCategory::fetch($categoryID);

			if ($articleCategory === null){
				throw new CategoryDoesNotExist("No article category found with ID {$categoryID}");
			}

			// Delete any page data that has this category
			/** @var PageData[] $pageDatas */
			$pageDatas = PageData::query(
				columnQuery: (new ColumnQuery())
					->where("name","=",PageDatas::ARTICLE_CATEGORY->name)
					->and()
					->where("value","=",$categoryID)
			);

			foreach($pageDatas as $pageData){
				$pageData->delete();
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETE_ARTICLE_CATEGORY->value,
				accountID: (Account::getCurrentUser())->id,
				ip: HttpHelper::getIPFromRequest(),
				jsonData: json_encode([
					"categoryName"=>$articleCategory->name,
					"categoryID"=>$articleCategory->id,
				]),
			);

			$articleCategory->delete();
		}
	}