<?php

	namespace Apache;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use System\HttpHelper;

	#[Controller]
	#[RouteBase("/uplift")]
	class ApacheController extends BaseController
	{
		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/manage-apache")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_APACHE)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "manage-apache/main.php",
			);
		}

		#[Route("GET", "/htaccess")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_APACHE)]
		public function getHtaccess(): JSONResult
		{
			return new JSONSuccess([
				"htaccess"=>file_get_contents(__DIR__ . "/../../../.htaccess"),
			]);
		}

		#[Route("PATCH", "/htaccess")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_APACHE)]
		public function editHtaccess(Request $request): JSONResult
		{
			$payload = $request->getPayload();

			try {
				$roleName = $payload->getTextPayload("htaccess-content");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			file_put_contents(
				filename: __DIR__ . "/../../../.htaccess",
				data: $roleName->contents
			);

			ActivityLog::log(
				categoryID: ActivityLogCategories::SAVE_APACHE_CONFIG_FILE->value,
				accountID: (Account::getCurrentUser())->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}
	}