<?php

	namespace ActivityLogs;

	use Accounts\Account;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use System\System;

	class ActivityLogsService
	{
		/**
		 * @param int $category
		 * @param int|null $user
		 * @return ColumnQuery
		 */
		private static function getActivityLogColumnQuery(
			int $category,
			int|null $user = null
		): ColumnQuery{
			$columnQuery = new ColumnQuery();

			if (!empty($category) && !empty($user)){
				$columnQuery
					->where("category", "=", $category)
					->and()
					->where("userID", "=", $user);
			} elseif(!empty($category)) {
				$columnQuery
					->where("category", "=", $category);
			} elseif(!empty($user)) {
				$columnQuery
					->where("userID", "=", $user);
			}

			return $columnQuery;
		}

		/**
		 * @param array $activityLogs
		 * @return ActivityLogDto[]
		 */
		public static function convertActivityLogsToDtos(array $activityLogs): array{
			$users = Account::query();
			/**
			 * @var ActivityLogDto[]
			 */
			$dtos = [];

			foreach($activityLogs as $activityLog) {
				$dto = new ActivityLogDto($activityLog);
				foreach($users as $user) {
					if($user->id === $activityLog->userID) {
						$dto->username = $user->username;
					}
				}
				$dto->categoryDescription = ActivityLogCategories::ACTION_VERBIAGE[$activityLog->categoryID];
				$dto->timeAgo = System::getTimeAgo($activityLog->timestamp);
				$dtos[] = $dto;
			}

			return $dtos;
		}

		/**
		 * @param int $page
		 * @param int $limit
		 * @param int $category
		 * @param string $order
		 * @param int|null $user
		 * @return ActivityLogDto[]
		 */
		public static function getActivityLogs(
			int $page,
			int $limit,
			int $category,
			string $order,
			int|null $user = null
		): array{
			$activityLogs = ActivityLog::query(
				columnQuery: self::getActivityLogColumnQuery($category, $user),
				resultOrder: (new ResultOrder())
					->by("timestamp", $order),
				pager: new Pager(pageNumber: $page, limit: $limit),
			);

			return self::convertActivityLogsToDtos($activityLogs);
		}

		/**
		 * @param int $category
		 * @param int|null $user
		 * @return int
		 */
		public static function getTotalActivityLogs(
			int $category,
			int|null $user = null,
		): int{
			return ActivityLog::count(
				columnQuery: self::getActivityLogColumnQuery($category, $user),
			);
		}

		/**
		 * @param int $userID
		 * @return ActivityLog|null
		 */
		public static function getLastActivityLogForUser(
			int $userID
		): ActivityLog|null{
			/** @var ?ActivityLog $activityLog */
			$activityLog = ActivityLog::queryOne(
				columnQuery: (new ColumnQuery())
					->where("userID", "=", $userID),
			);

			return $activityLog;
		}
	}