<?php

namespace Accounts;

use Nox\ORM\ColumnQuery;
use Nox\ORM\Exceptions\NoPrimaryKey;
use Nox\ORM\Pager;
use Roles\PermissionCategories;
use Roles\Role;
use ValueError;

class AccountsService
{

  /**
   * @param string $query
   * @return ColumnQuery
   */
  private static function getColumnQueryForSearch(string $query): ColumnQuery
  {
    $columnQuery = new ColumnQuery();
    if (!empty($query)) {
      $columnQuery->where("username", "LIKE", sprintf("%%%s%%", $query))
        ->or()
        ->where("firstName", "LIKE", sprintf("%%%s%%", $query))
        ->or()
        ->where("CONCAT(`{firstName}`, ' ', `{lastName}`)", "LIKE", sprintf("%%%s%%", $query))
        ->or()
        ->where("lastName", "LIKE", sprintf("%%%s%%", $query))
        ->or()
        ->where("email", "LIKE", sprintf("%%%s%%", $query));
    }

    return $columnQuery;
  }

  public static function logUserIn(
    Account $account,
    ?int $expiratoryTime = null,
  ): AccountCookieSession {

    if ($expiratoryTime === null) {
      $expiratoryTime = time() + (86400 * Account::TOKEN_EXPIRATORY_IN_DAYS);
    }

    $cookieSession = new AccountCookieSession();
    $cookieSession->cookie = AccountCookieSession::generateTokenHash($account->username);
    $cookieSession->userID = $account->id;
    $cookieSession->expires = $expiratoryTime;
    $cookieSession->save();

    return $cookieSession;
  }

  /**
   * @param int $page
   * @param int $limit
   * @param string $query
   * @return Account[]
   */
  public static function getAccounts(
    int $page,
    int $limit,
    string $query,
  ): array {
    return Account::query(
      columnQuery: self::getColumnQueryForSearch($query),
      pager: new Pager(pageNumber: $page, limit: $limit),
    );
  }

  /**
   * @param string $query
   * @return int
   */
  public static function getTotalAccountsForQuery(
    string $query,
  ): int {
    return Account::count(
      columnQuery: self::getColumnQueryForSearch($query),
    );
  }

  /**
   * @param string $username
   * @param string $firstName
   * @param string $lastName
   * @param string $email
   * @param string $passwordPlainText
   * @param int $roleID
   * @return Account
   * @throws UsernameInUse
   * @throws ValueError
   */
  public static function createAccount(
    string $username,
    string $firstName,
    string $lastName,
    string $email,
    string $passwordPlainText,
    int $roleID,
  ): Account {

    $username = trim($username);

    // Validations
    if (empty($firstName)) {
      throw new ValueError("A first name must be provided.");
    }

    if (empty($username)) {
      throw new ValueError("A username must be provided.");
    }

    if (empty($passwordPlainText)) {
      throw new ValueError("A username must be provided.");
    }

    // Validate email format if it is set
    $email = trim($email);
    if (strlen($email) > 0) {
      $email = filter_var(
        value: $email,
        filter: FILTER_VALIDATE_EMAIL,
      );

      if ($email === false) {
        throw new ValueError("Email is in an invalid format.");
      }
    }

    // Does account with that username exist?
    $existingAccount = Account::queryOne(
      columnQuery: (new ColumnQuery())
        ->where("username", "=", $username),
    );

    if ($existingAccount !== null) {
      throw new UsernameInUse("Username is already in use by another account.");
    }

    $newAccount = new Account();
    $newAccount->username = $username;
    $newAccount->firstName = $firstName;
    $newAccount->lastName = $lastName;
    $newAccount->roleID = $roleID;
    $newAccount->email = $email;
    $newAccount->password = password_hash(
      password: $passwordPlainText,
      algo: PASSWORD_DEFAULT,
    );
    $newAccount->canBeDeleted = 1;
    $newAccount->save();

    return $newAccount;
  }

  /**
   * @param Account $account
   * @param string $firstName
   * @param string $lastName
   * @param string $email
   * @param int $roleID
   * @return void
   * @throws ValueError
   */
  public static function saveAccountEdits(
    Account $account,
    string $firstName,
    string $lastName,
    string $email,
    int $roleID,
  ): void {

    // Validations
    if (empty($firstName)) {
      throw new ValueError("A first name must be provided.");
    }

    // Validate email format if it is set
    $email = trim($email);
    if (strlen($email) > 0) {
      $email = filter_var(
        value: $email,
        filter: FILTER_VALIDATE_EMAIL,
      );

      if ($email === false) {
        throw new ValueError("Email is in an invalid format.");
      }
    }

    $account->firstName = $firstName;
    $account->lastName = $lastName;
    $account->roleID = $roleID;
    $account->email = $email;
    $account->save();
  }

  /**
   * @param Account $account
   * @param string $newPassword
   * @return void
   * @throws ValueError
   */
  public static function changePassword(
    Account $account,
    string $newPassword,
  ): void {

    // Validations
    if (empty($newPassword)) {
      throw new ValueError("Passwords cannot be empty.");
    }

    $account->password = password_hash(
      password: $newPassword,
      algo: PASSWORD_DEFAULT,
    );

    $account->save();

    // TODO
    // Send password notification email
  }

  /**
   * @param Account $account
   * @param int $isDisabled
   * @return void
   * @throws ValueError
   */
  public static function changeDisabledStatus(
    Account $account,
    int $isDisabled,
  ): void {

    if ($isDisabled !== 0 && $isDisabled !== 1) {
      throw new ValueError("isDisabled parameter must be either 0 or 1.");
    }

    $account->disabled = $isDisabled;
    $account->save();
  }

  /**
   * @param Role $role
   * @return void
   */
  public static function changeRole(
    Role $role
  ): void {
    $accounts = Account::query(
      columnQuery: (new ColumnQuery())
        ->where("role_id", "=", $role->id),
    );

    foreach ($accounts as $account) {
      $account->roleID = 0;
      $account->save();
    }
  }

  /**
   * Attempts to fetch an account that has all permissions.
   * @throws NoAccountWithAllPermissionsFound
   */
  public static function getAccountWithAllPermissions(): Account
  {
    // First, find a Role that has all permissions enabled
    /** @var Role[] $allRoles */
    $allRoles = Role::query();

    $rolesWithAllPermissions = [];
    foreach ($allRoles as $role) {
      foreach (PermissionCategories::cases() as $permissionCategory) {
        if (!$role->hasEnabledPermission($permissionCategory)) {
          continue 2;
        }
      }

      // If we get here, then this role has all permissions
      $rolesWithAllPermissions[] = $role;
    }

    // Find any account that has a roleID of the roles in the $rolesWithAllPermissions array
    // The first match is acceptable to use
    $account = null;
    foreach ($rolesWithAllPermissions as $role) {
      /** @var ?Account $accountWithRole */
      $accountWithRole = Account::queryOne(
        columnQuery: (new ColumnQuery())
          ->where("role_id", "=", $role->id)
      );

      if ($accountWithRole !== null) {
        $account = $accountWithRole;
        break;
      }
    }

    if ($account === null) {
      throw new NoAccountWithAllPermissionsFound("No account found in the system that has all permissions.");
    }

    return $account;
  }

  /**
   * Logs out the current session user
   * @return void
   * @throws NoPrimaryKey
   */
  public static function logoutCurrentSession(): void
  {
    $account = Account::getCurrentUser();
    if ($account !== null) {
      // Get the token row and delete it
      $token = AccountCookieSession::queryOne(
        columnQuery: (new ColumnQuery())
          ->where("cookie", "=", $_COOKIE[Account::LOGIN_TOKEN_NAME]),
      );
      if ($token !== null) {
        $token->delete();
      }

      setcookie(Account::LOGIN_TOKEN_NAME, "", -1, "/");
    }
  }

  /**
   * Queries for the first account with the provided first and last names
   */
  public static function getAccountByName(string $firstName, string $lastName): ?Account
  {
    /** @var ?Account $account */
    $account = Account::queryOne(
      columnQuery: (new ColumnQuery())
        ->where("firstName", "=", $firstName)
        ->and()
        ->where("lastName", "=", $lastName)
    );

    return $account;
  }

  /**
   * Get an account by the account Id
   */
  public static function getAccountById(int $accountId): ?Account
  {
    /** @var ?Account $account */
    $account = Account::fetch($accountId);

    return $account;
  }
}