<?php

	namespace Accounts;

	use Accounts\Attributes\RequireNoAccount;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use ActivityLogs\ActivityLogsService;
	use APIPublic\AuthorizeByAPIKey;
	use MonologWrapper\MonologWrapper;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\Http\Response;
	use Nox\Http\Rewrite;
	use Nox\ORM\ColumnQuery;
	use Nox\RenderEngine\Exceptions\LayoutDoesNotExist;
	use Nox\RenderEngine\Exceptions\ParseError;
	use Nox\RenderEngine\Exceptions\ViewFileDoesNotExist;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Accounts\Attributes\RequireLogin;
	use Roles\PermissionCategories;
	use Roles\Role;
	use Roles\RolesService;
	use System\HttpHelper;
	use System\System;

	#[Controller]
	#[RouteBase("/uplift")]
	class AccountsController extends BaseController{

		#[Route("POST", "/api/private/v1/login")]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequireNoAccount(rewriteRequestIfAccountPresent: false, responseCode: 403, newRoute: null)]
		public function loginAccount(Request $request): JSONResult
		{
			// Fetch client IP
			$clientIP = $request->getIP();

			// Deny blank IPs
			if (empty($clientIP)) {
				http_response_code(403);
				return new JSONError("Your request must include an IP address in the request headers.");
			}

			try{
				$username = $request->getPayload()->getTextPayload("username");
				$password = $request->getPayload()->getTextPayload("password");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			// Find the user with the parameters
			/** @var ?Account $account */
			$account = Account::queryOne(
				columnQuery: (new ColumnQuery())
					->where("username", "=", $username->contents)
			);

			if ($account === null) {
				http_response_code(400);
				return new JSONError("Invalid account credentials.");
			}

			// Verify the password
			if (!password_verify($password->contents, $account->password)) {
				return new JSONError("Invalid account credentials.");
			}

			// Create a login token
			$cookieSession = new AccountCookieSession();
			$cookieSession->cookie = AccountCookieSession::generateTokenHash($username->contents);
			$cookieSession->userID = $account->id;
			$cookieSession->expires = time() + (86400 * Account::TOKEN_EXPIRATORY_IN_DAYS);
			$cookieSession->save();

			ActivityLog::log(
				categoryID: ActivityLogCategories::SIGNED_IN->value,
				accountID: $account->id,
				ip: $clientIP,
				jsonData: json_encode([]),
			);

			return new JSONSuccess([
				"tokenName" => Account::LOGIN_TOKEN_NAME,
				"token" => $cookieSession->cookie,
				"expiresInDays" => Account::TOKEN_EXPIRATORY_IN_DAYS,
			]);
		}

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "/manage-user-accounts")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_USERS)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "user-accounts/main.php",
				viewScope: [
					"roles"=>Role::query(),
				],
			);
		}

		/**
		 * @throws ParseError
		 * @throws ViewFileDoesNotExist
		 * @throws LayoutDoesNotExist
		 */
		#[Route("GET", "@/manage-user-accounts/(?<userID>\d+)$@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_USERS)]
		public function manageUserView(Request $request): string | Rewrite
		{
			$userID = (int) $request->getParameter("userID");

			/** @var Account | null $account */
			$account = Account::fetch($userID);

			if ($account === null){
				return new Rewrite(
					path: "/uplift/404",
					statusCode:404,
				);
			}

			return Renderer::renderView(
				viewFileName: "user-accounts/manage-account.php",
				viewScope: [
					"managedAccount"=>$account,
					"roles"=>Role::query(),
					"activityLogCategories"=>ActivityLogCategories::cases(),
				],
			);
		}

		#[Route("GET", "/user-accounts")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_USERS)]
		public function getAccounts(Request $request): JSONResult
		{
			$limit = $request->getQueryValue("limit");
			$page = $request->getQueryValue("page");
			$query = $request->getQueryValue("query");

			if ($limit === null){
				return new JSONError("Missing 'limit' GET parameter.");
			}

			if ($page === null){
				return new JSONError("Missing 'page' GET parameter.");
			}

			if ($query === null){
				return new JSONError("Missing 'query' GET parameter.");
			}

			$totalAccountsForQuery = AccountsService::getTotalAccountsForQuery($query);

			$accounts = AccountsService::getAccounts(
				page: $page,
				limit:$limit,
				query:$query,
			);

			foreach($accounts as $account){
				// Hydrate the account by adjusting the "role" key to reflect the actual role.
				// "role" is an old column that isn't supported anymore and is outdated. Use the
				// role_id column and get the actual role name
				$roleID = $account->roleID;
				/** @var Role $role */
				$role = Role::fetch($roleID);
				$account->role = $role->name;

				// Add the most recent activity log timestamp to account
				$lastActivityLog = ActivityLogsService::getLastActivityLogForUser($account->id);

				if($lastActivityLog !== null) {
					$account->lastActivity = $lastActivityLog->timestamp;
				}
			}

			//var_dump($lastActivityLog);

			return new JSONSuccess([
				"accounts"=>$accounts,
				"totalAccounts"=>$totalAccountsForQuery,
				"totalPages"=>ceil($totalAccountsForQuery / $limit),
			]);
		}

		#[Route("PUT", "/user-accounts")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_USERS)]
		public function createAccount(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			try{
				$username = $payload->getTextPayload("username");
				$firstName = $payload->getTextPayload("first-name");
				$lastName = $payload->getTextPayload("last-name");
				$email = $payload->getTextPayload("email");
				$passwordPlainText = $payload->getTextPayload("password");
				$roleID = $payload->getTextPayload("role");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			$currentAccount = Account::getCurrentUser();

			try {
				$newAccount = AccountsService::createAccount(
					username: $username->contents,
					firstName: $firstName->contents,
					lastName: $lastName->contents,
					email: $email?->contents ?? "",
					passwordPlainText: $passwordPlainText->contents,
					roleID: (int) $roleID->contents,
				);
			} catch (UsernameInUse|\ValueError $e ) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::CREATE_NEW_ACCOUNT->value,
				accountID: $currentAccount->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess([
				"newAccount"=>$newAccount,
			]);
		}

		#[Route("PATCH", "@/user-accounts/(?<userID>\d+)$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_USERS)]
		public function editAccount(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$accountID = (int) $request->getParameter('userID');

			try {
				$firstName = $payload->getTextPayload("first-name");
				$lastName = $payload->getTextPayload("last-name");
				$email = $payload->getTextPayload("email");
				$roleID = $payload->getTextPayload("role-id");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			$currentAccount = Account::getCurrentUser();

			/** @var Account | null $account */
			$account = Account::fetch($accountID);

			if ($account === null){
				return new JSONError("No account with ID {$accountID}");
			}

			try {
				 AccountsService::saveAccountEdits(
					account:$account,
					firstName: $firstName->contents,
					lastName: $lastName->contents,
					email: $email->contents,
					roleID: $roleID->contents,
				);
			} catch (\ValueError $e ) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::EDIT_ACCOUNT_INFORMATION->value,
				accountID: $currentAccount->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"accountEdited"=>$account->username,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "@/user-accounts/(?<userID>\d+)/password$@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_USERS)]
		public function changeAccountPassword(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$accountID = (int) $request->getParameter("userID");
			$currentAccount = Account::getCurrentUser();

			try {
				$passwordPlainText = $payload->getTextPayload("new-password");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			/** @var Account | null $account */
			$account = Account::fetch($accountID);

			if ($account === null){
				return new JSONError("No account with ID {$accountID}");
			}

			try {
				AccountsService::changePassword(
					account:$account,
					newPassword: $passwordPlainText->contents,
				);
			} catch (\ValueError $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				ActivityLogCategories::CHANGED_PASSWORD->value,
				$currentAccount->id,
				$request->getIP(),
				jsonData: json_encode([
					"accountEdited"=>$account->username,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "@/user-accounts/(?<userID>\d+)/disabled@", true)]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_USERS)]
		public function changeAccountDisabledStatus(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$accountID = (int) $request->getParameter('userID');
			$currentAccount = Account::getCurrentUser();
			try {
				$isDisabled = $payload->getTextPayload("is-disabled");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			/** @var Account | null $account */
			$account = Account::fetch($accountID);

			if ($account === null){
				return new JSONError("No account with ID {$accountID}");
			}

			$isDisabledValue = (int) $isDisabled->contents;

			try {
				AccountsService::changeDisabledStatus(
					account:$account,
					isDisabled: $isDisabledValue,
				);
			} catch (\ValueError $e ) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				ActivityLogCategories::CHANGE_ACCOUNT_DISABLE_STATUS->value,
				$currentAccount->id,
				$request->getIP(),
				jsonData: json_encode([
					"status"=>$isDisabledValue,
					"accountEdited"=>$account->username,
				]),
			);

			return new JSONSuccess();
		}

		/**
		 * This route allows a user to create a general account with the assumption it is missing,
		 * so long as a valid Uplift API key is present as a GET query parameter. This is almost always from an external
		 * software integration that is trying to log a user in, but the account with the provided firstName and lastName
		 * doesn't yet exist in the system.
		 */
		#[Route("GET", "/accounts/create-account-from-client-dashboard")]
		public function createGeneralAccountFromDashboardView(Request $request): string{
			$logger = MonologWrapper::getLogger();
			$firstName = $request->getQueryValue("firstName");
			$lastName = $request->getQueryValue("lastName");
			$upliftApiKey = $request->getQueryValue("uplift-api-key");

			// Verify the API key
			if (empty($upliftApiKey)){
				$logger->warning("No API key provided to create missing general account.");
				header("X-Forbidden-Reason: No API key provided.");
				http_response_code(403);
				return "";
			}else{
				// Check if it is valid
				if (System::isAPIKeyValid($upliftApiKey)){

					// Verify a firstName and lastName is present
					if (empty($firstName) || empty($lastName)){
						$logger->warning("Missing either firstName or lastName to create missing general account.");
						http_response_code(400);
						return "";
					}

					// All checks out, render the view
					return Renderer::renderView(
						viewFileName: "user-accounts/create-general-account-by-api.php",
						viewScope: [
							"upliftApiKey" => $upliftApiKey,
							"firstName" => $firstName,
							"lastName" => $lastName,
						],
					);
				}else{
					$logger->warning("API key invalid to create missing general account.");
					header("X-Forbidden-Reason: Invalid API key.");
					http_response_code(403);
					return "";
				}
			}
		}

		#[Route("POST", "/accounts/create-account-from-client-dashboard")]
		#[UseJSON]
		#[ProcessRequestBody]
		public function createGeneralAccountFromDashboard(Request $request): JSONResult{
			$logger = MonologWrapper::getLogger();
			$logger->info("POSt received to create a general account via the view where the user came from the Client Dashboard.");
			$payload = $request->getPayload();
			try{
				$upliftApiKey = $payload->getTextPayload("uplift-api-key");
				$firstName = $payload->getTextPayload("first-name");
				$lastName = $payload->getTextPayload("last-name");
				$email = $payload->getTextPayload("email");
				$username = $payload->getTextPayload("username");
				$password = $payload->getTextPayload("password");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			// Verify the API key
			if (System::isAPIKeyValid($upliftApiKey->contents)){

				// Verify a role named "client" exists, and use that to assign the new account
				$clientRole = RolesService::getRoleByName("client");

				if ($clientRole === null){
					$clientRole = RolesService::createClientRole();
				}

				try {
					$account = AccountsService::createAccount(
						username: $username->contents,
						firstName: $firstName->contents,
						lastName: $lastName->contents,
						email: $email->contents,
						passwordPlainText: $password->contents,
						roleID: $clientRole->id,
					);
				} catch (UsernameInUse|\ValueError $e) {
					http_response_code(400);
					return new JSONError($e->getMessage());
				}

				$cookieSession = AccountsService::logUserIn($account);
				return new JSONSuccess([
					"token"=>$cookieSession->cookie,
					"loginTokenCookieName"=>Account::LOGIN_TOKEN_NAME
				]);
			}else{
				$logger->warning("API key invalid to POST create missing general account.");
				http_response_code(403);
				return new JSONError("API key is invalid.");
			}
		}
	}
