<?php

	namespace Accounts;

	use Nox\Http\Request;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Interfaces\ModelInstance;
	use Nox\ORM\Interfaces\MySQLModelInterface;
	use Nox\ORM\ModelClass;
	use Nox\Router\BaseController;
	use Roles\Permission;
	use Roles\PermissionCategories;
	use Roles\Role;
	use System\HttpHelper;

	class Account extends ModelClass implements ModelInstance
	{

		const LOGIN_TOKEN_NAME = "m_kurisu";
		const TOKEN_EXPIRATORY_IN_DAYS = 30;

		/**
		 * A cache variable for the current user. To avoid multiple
		 * checks and queries in one PHP process.
		 */
		public static ?Account $currentUser_cache = null;

		public ?int $id = null;
		public string $username;
		public string $firstName;
		public string $lastName;
		public string $password;
		public string $email;

		/** @deprecated  */
		public string $role;

		public int $roleID;
		public int $creationTime;
		/** @deprecated  */
		public int $canBeDeleted;

		public int $disabled;

		// Properties not meant for native models
		public ?int $lastActivity = null;

		/**
		 * Fetches a user from the current HTTP session
		 * by checking and then reading their login token (cookie) or Authorization header value
		 */
		public static function getCurrentUser(): ?Account{
			if (self::$currentUser_cache !== null){
				return self::$currentUser_cache;
			}

			// Two ways to authenticate.
			// One is by a cookie value, and the other is in the Authorization header formatted as below
			// Authorization: Bearer TOKEN_HERE
			$loginToken = null;

			$nox = BaseController::$noxInstance;
			$router = $nox->router;
			$request = $router->currentRequest;
			$loginCookieValue = $request->getCookie(Account::LOGIN_TOKEN_NAME);

			if ($loginCookieValue !== null) {
				$loginToken = $loginCookieValue;
			}else{
				$authorizationHeaderValue = HttpHelper::getFirstHeaderValue("authorization");
				if ($authorizationHeaderValue !== null){
					$authComponents = explode(" ", $authorizationHeaderValue);
					if (count($authComponents) === 2){
						$authTokenType = $authComponents[0];
						if (strtolower($authTokenType) === "bearer"){
							$loginToken = $authComponents[1];
						}
					}
				}
			}

			if ($loginToken !== null){
				$loginTokens = AccountCookieSession::query(
					columnQuery: (new ColumnQuery())
						->where("cookie", "=", $loginToken)
						->and()
						->where("expires",">",time()),
				);
				if (!empty($loginTokens)){
					/** @var AccountCookieSession $cookieSession */
					$cookieSession = $loginTokens[0];
					$userID = $cookieSession->userID;

					/** @var ?Account $account */
					$account = Account::fetch(
						primaryKey:$userID,
					);

					// Set the result as a cache
					self::$currentUser_cache = $account;

					return $account;
				}
			}

			return null;
		}

		public static function getModel(): MySQLModelInterface{
			return new \Accounts\AccountsModel();
		}

		public function __construct(){
			parent::__construct($this);
		}

		public function getFullName(): string{
			if (!empty($this->lastName)){
				return sprintf("%s %s", $this->firstName, $this->lastName);
			}else{
				return $this->firstName;
			}
		}

		/**
		 * Checks if the user is in a role which has the permission
		 */
		public function hasPermission(PermissionCategories $permissionCategory): bool{
			$userRoleID = $this->roleID;

			/** @var ?Role $role */
			$role = Role::fetch($userRoleID);

			if ($role === null){
				return false;
			}

			return $role->hasEnabledPermission($permissionCategory);
		}

		/**
		 * Checks if the user is in a role which has at least one of the provided
		 * permissions.
		 * @param PermissionCategories[] $requiredPermissions
		 */
		public function hasAnyPermission(array $requiredPermissions): bool{
			foreach($requiredPermissions as $requiredPermission){
				if ($this->hasPermission($requiredPermission)){
					return true;
				}
			}

			return false;
		}
	}