<?php
	namespace APIPublic;

	use Accounts\Account;
	use Accounts\AccountCookieSession;
	use Accounts\AccountsService;
	use MonologWrapper\MonologWrapper;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageType;
	use Roles\PermissionAlreadyExists;
	use Roles\PermissionCategories;
	use Roles\PermissionsService;
	use Roles\RolesService;
	use Settings\Settings;
	use Settings\SettingService;

	class APIPublicService{
		public static function getValidAccountSessionFromToken(
			string $token,
		): ?AccountCookieSession{
			/** @var AccountCookieSession|null $session */
			$session = AccountCookieSession::queryOne(
				columnQuery: (new ColumnQuery())
					->where("cookie","=",$token)
					->and()
					->where("expires",">",time())
			);

			return $session;
		}

		public static function setSiteSuspended(int $state): void{
			SettingService::saveSetting(Settings::SITE_SUSPENDED->value, $state);
		}

		public static function getAccountLoginToken(Account $account): string{
			$logger = MonologWrapper::getLogger();
			// Default expires in 1 year
			$tokenExpires = time() + (86400 * 365);
			$cookieSession = AccountsService::logUserIn($account, $tokenExpires);

			$logger->info("Created login token for user ($account->id) $account->username");

			return $cookieSession->cookie;
		}

		/**
		 * This method is used when the system cannot find an "undeletable" admin account that has all
		 * permissions. It will create a new role with all permissions and a new account assigned to that role.
		 * @return Account
		 */
		public static function createAdminAccountAndAdminRole(): Account{
			$roleName = "AutoAdminRole_" . time();
			$role = RolesService::createRole($roleName);
			$role->hasAllPermissions = 1;
			$role->save();

			foreach(PermissionCategories::cases() as $permissionCategory) {
				try {
					PermissionsService::createPermission(
						role: $role,
						category: $permissionCategory,
						isEnabled: true,
					);
				} catch (PermissionAlreadyExists $ignored) {}
			}

			$accountUsername = "AutoAdminAccount_" . time();
			$newAccount = new Account();
			$newAccount->roleID = $role->id;
			$newAccount->username = $accountUsername;
			$newAccount->firstName = "Admin";
			$newAccount->lastName = "Account";
			$newAccount->save();

			return $newAccount;
		}

		/**
		 * Fetches all pages filtered by a pageType. A blank pageType will return all pages regardless of type.
		 * @param string $pageType
		 * @param int $limit
		 * @param int $pageNumber
		 * @param string $sortColumnName
		 * @param int $sortColumnDirection
		 * @return Page[]
		 */
		public static function getPagesWithFilters(
			string $pageType,
			int $limit,
			int $pageNumber,
			string $sortColumnName,
			int $sortColumnDirection,
			?int $publicationStatus
		): array{
			$columnQuery = new ColumnQuery();
			$columnQuery->where("is_deleted", "=", "0");

			if (strlen($pageType) > 0) {
				$columnQuery->and();
				$columnQuery->where("pageType", "=", $pageType);
			}

			if ($publicationStatus !== null){
				$columnQuery->and();
				$columnQuery->where("publication_status", "=", $publicationStatus);
			}

			$pager = new Pager($pageNumber, $limit);
			$resultOrder = null;

			if (!empty($sortColumnName)){
				$resultOrder = (new ResultOrder())
					->by($sortColumnName, $sortColumnDirection === -1 ? "DESC" : "ASC");
			}

			/** @var Page[] $pages */
			$pages = Page::query(
				columnQuery: $columnQuery,
				resultOrder: $resultOrder,
				pager: $pager
			);

			return $pages;
		}

		/**
		 * Counts all pages grouped by their type
		 * @return array
		 */
		public static function countPagesByPageType(): array{
			return [
				"numAllPages"=>Page::count(
					columnQuery: (new ColumnQuery())
						->where("is_deleted","=",0)
				),
				"numProjectPages"=>Page::count(
					columnQuery: (new ColumnQuery())
						->where("is_deleted","=",0)
						->and()
						->where("pageType","=",PageType::Project->name)
				),
				"numBlogPages"=>Page::count(
					columnQuery: (new ColumnQuery())
						->where("is_deleted","=",0)
						->and()
						->where("pageType","=",PageType::Blog->name)
				),
				"numCityPages"=>Page::count(
					columnQuery: (new ColumnQuery())
						->where("is_deleted","=",0)
						->and()
						->where("pageType","=",PageType::City->name)
				),
				"numGeneralPages"=>Page::count(
					columnQuery: (new ColumnQuery())
						->where("is_deleted","=",0)
						->and()
						->where("pageType","=",PageType::General->name)
				),
				"numServicePages"=>Page::count(
					columnQuery: (new ColumnQuery())
						->where("is_deleted","=",0)
						->and()
						->where("pageType","=",PageType::Service->name)
				),
			];
		}
	}