<?php

/**
 * API for receiving remote requests
 */

namespace APIPublic;

use Accounts\Account;
use Accounts\AccountsService;
use Accounts\NoAccountWithAllPermissionsFound;
use ActivityLogs\ActivityLog;
use ActivityLogs\ActivityLogCategories;
use MonologWrapper\MonologWrapper;
use Nox\Http\Attributes\ProcessRequestBody;
use Nox\Http\Attributes\UseJSON;
use Nox\Http\Exceptions\NoPayloadFound;
use Nox\Http\JSON\JSONError;
use Nox\Http\JSON\JSONResult;
use Nox\Http\JSON\JSONSuccess;
use Nox\Http\Request;
use Nox\Router\Attributes\Controller;
use Nox\Router\Attributes\Route;
use Nox\Router\Attributes\RouteBase;
use Nox\Router\BaseController;
use Nox\Http\Redirect;

#[Controller]
#[RouteBase("/uplift/api/v1")]
class APIPublicController extends BaseController
{

  /**
   * Fetches a login token belonging to an account that matches the first result of the firstName and lastName query parameters.
   * Both of these parameters must be present and both must be non-empty strings.
   *
   * The JSONError will have additional keys to help the interfacing external server understand more about the error thrown.
   * The key name will be errorTypeName and the possible values will be
   * - Missing Parameters
   * - Invalid Parameters
   * - No Account Found
   *
   * Note: It's technically possible to have this endpoint return admin accounts, or any account, if the proper
   * first and last name is provided.
   * @param Request $request
   * @return JSONResult
   */
  #[Route("GET", "/accounts/login-token-by-name")]
  #[AuthorizeByAPIKey]
  #[UseJSON]
  public function getLoginTokenByAccountName(Request $request): JSONResult
  {
    $logger = MonologWrapper::getLogger();
    $logger->info("Request to get login token by account name.");

    $firstName = $request->getQueryValue("firstName");
    $lastName = $request->getQueryValue("lastName");

    if ($firstName === null || $lastName === null) {
      http_response_code(400);
      return new JSONError("Both a firstName and lastName query parameters are required.", [
        "errorTypeName" => "Missing Parameters",
      ]);
    }

    if (empty($firstName) || empty($lastName)) {
      http_response_code(400);
      return new JSONError("Both a firstName and lastName query parameters must be non-empty strings.", [
        "errorTypeName" => "Invalid Parameters",
      ]);
    }

    $account = AccountsService::getAccountByName($firstName, $lastName);

    if ($account === null) {
      $logger->warning("No account found in getLoginTokenByAccountName with first name: '$firstName' and last name: '$lastName'");
      http_response_code(400);
      return new JSONError("No account found with the provided first and last names.", [
        "errorTypeName" => "No Account Found",
      ]);
    }

    // Account found, get the login token
    $loginToken = APIPublicService::getAccountLoginToken($account);
    $logger->info("Generated login token for account " . $account->id . " with username: " . $account->username);

    ActivityLog::log(
      categoryID: ActivityLogCategories::LOGIN_TOKEN_FETCHED_VIA_REMOTE->value,
      accountID: $account->id,
      ip: $request->getIP(),
      jsonData: json_encode([
        "accountId" => $account->id,
        "accountUsername" => $account->username
      ]),
    );

    return new JSONSuccess([
      "token" => $loginToken,
    ]);
  }

  /**
   * Returns a login token for an account by the account Id.
   */
  #[Route("GET", "/accounts/login-token-by-id")]
  #[AuthorizeByAPIKey]
  #[UseJSON]
  public function getLoginTokenByAccountId(Request $request): JSONResult
  {
    $logger = MonologWrapper::getLogger();
    $logger->info("Request to get login token by account Id.");

    $id = $request->getQueryValue("id");

    $account = AccountsService::getAccountById($id);

    if ($account === null) {
      $logger->warning("No account found in getLoginTokenByAccountId with Id: '$id''");
      http_response_code(400);
      return new JSONError("No account found with the provided account Id.");
    }

    // Account found, get the login token
    $loginToken = APIPublicService::getAccountLoginToken($account);
    $logger->info("Generated login token for account " . $account->id . " with username: " . $account->username);

    ActivityLog::log(
      categoryID: ActivityLogCategories::LOGIN_TOKEN_FETCHED_VIA_REMOTE->value,
      accountID: $account->id,
      ip: $request->getIP(),
      jsonData: json_encode([
        "accountId" => $account->id,
        "accountUsername" => $account->username
      ]),
    );

    return new JSONSuccess([
      "token" => $loginToken,
    ]);
  }

  /**
   * Finds an account with a role that has all permissions in the system. If none exists, a new account and a new
   * role that meets this criteria will be created. The login token for this account will be generated and returned
   * by the API. This is meant to be accessed by a remote API
   * to help log a user in.
   */
  #[Route("GET", "/accounts/get-admin-account-login-token")]
  #[AuthorizeByAPIKey]
  #[UseJSON]
  public function getAdminAccountLoginTokenUsingAPIKey(Request $request): JSONResult
  {
    $logger = MonologWrapper::getLogger();
    $logger->info("Processing request to get a login token for an admin account.");

    try {
      $adminAccount = AccountsService::getAccountWithAllPermissions();
    } catch (NoAccountWithAllPermissionsFound) {
      $adminAccount = APIPublicService::createAdminAccountAndAdminRole();
    }

    $loginToken = APIPublicService::getAccountLoginToken($adminAccount);

    ActivityLog::log(
      categoryID: ActivityLogCategories::LOGIN_TOKEN_FETCHED_VIA_REMOTE->value,
      accountID: $adminAccount->id,
      ip: $request->getIP(),
      jsonData: json_encode([
        "accountId" => $adminAccount->id,
        "accountUsername" => $adminAccount->username
      ]),
    );

    $logger->info($adminAccount->getFullName() . " had a login token generated by remote API.");

    return new JSONSuccess([
      "token" => $loginToken,
    ]);
  }

  /**
   * List system accounts.
   */
  #[Route("GET", "/accounts")]
  #[AuthorizeByAPIKey]
  #[UseJSON]
  public function listAccounts(Request $request): JSONResult
  {
    $accounts = AccountsService::getAccounts(
      1,
      100,
      ""
    );

    // Remove the "password" key
    $accounts = array_map(function ($account) {
      $account = json_decode(json_encode($account), true);
      unset($account["password"]);
      return $account;
    }, $accounts);

    return new JSONSuccess([
      "accounts" => $accounts,
    ]);
  }

  #[Route("GET", "/accounts/login-with-token")]
  #[UseJSON]
  public function loginWithToken(Request $request): string|Redirect
  {
    $logger = MonologWrapper::getLogger();
    $logger->info("Requesting login with token.");

    $token = $request->getQueryValue("token");
    $redirectTo = $request->getQueryValue("to");

    if ($token === null) {
      $logger->warning("Missing token from request.");
      http_response_code(500);
      return "Missing token parameter in GET request.";
    }

    $activeSession = APIPublicService::getValidAccountSessionFromToken($token);

    if ($activeSession !== null) {
      $logger->info("Found active session from login token. Setting the login token as a cookie.");
      setcookie(
        name: Account::LOGIN_TOKEN_NAME,
        value: $token,
        expires_or_options: $activeSession->expires,
        path: "/"
      );

      ActivityLog::log(
        categoryID: ActivityLogCategories::LOGIN_WITH_TOKEN_VIA_REMOTE->value,
        accountID: $activeSession->userID,
        ip: $request->getIP(),
        jsonData: json_encode([]),
      );

      if ($redirectTo !== null) {
        return new Redirect(
          path: $redirectTo,
          statusCode: 302,
        );
      } else {
        $logger->warning("No login token found for token: " . $token);
        return new Redirect(
          path: "/uplift/dashboard",
          statusCode: 302,
        );
      }

    } else {
      $logger->warning("A login attempt using an outdated or non-existent login token from remote API request failed.");
      http_response_code(500);
      return "Token has no valid login session. You must get a new token as this one has expired or was never valid. If you cannot get a new token, then you should login with valid credentials at the /admin-panel URL.";
    }
  }

  #[Route("PATCH", "/settings/suspended")]
  #[AuthorizeByAPIKey]
  #[UseJSON]
  #[ProcessRequestBody]
  public function setSiteSuspendedState(Request $request): JSONResult
  {
    $logger = MonologWrapper::getLogger();
    $payload = $request->getPayload();

    try {
      $suspendedState = $payload->getTextPayload("suspended");
    } catch (NoPayloadFound $e) {
      return new JSONError($e->getMessage());
    }

    APIPublicService::setSiteSuspended($suspendedState->contents);
    $logger->info("Site suspended setting was changed via remote request to value: " . $suspendedState->contents);

    return new JSONSuccess();
  }

  #[AuthorizeByAPIKey]
  #[Route("GET", "/pages")]
  #[UseJSON]
  public function getPagesByType(Request $request): JSONResult
  {
    $logger = MonologWrapper::getLogger();
    $logger->info("Processing request to get pages by type.");

    $pageType = $_GET['pageType'] ?? "";
    $limit = (int) ($_GET['limit'] ?? 3000);
    $pageNumber = (int) ($_GET['pageNumber'] ?? 1);
    $sortColumnName = $_GET['sortColumnName'] ?? "";
    $sortColumnDirection = (int) ($_GET['sortColumnDirection'] ?? 1);
    $publicationStatus = $_GET['publicationStatus'] ?? null;

    $pages = APIPublicService::getPagesWithFilters(
      pageType: $pageType,
      limit: $limit,
      pageNumber: $pageNumber,
      sortColumnName: $sortColumnName,
      sortColumnDirection: $sortColumnDirection,
      publicationStatus: $publicationStatus
    );

    return new JSONSuccess([
      "pages" => $pages,
    ]);
  }

  /**
   * Fetches a summary of the page types and total number of them as key value pairs.
   */
  #[AuthorizeByAPIKey]
  #[Route("GET", "/pages/summary-count")]
  #[UseJSON]
  public function countNumOfPagesByType(Request $request): JSONResult
  {
    $logger = MonologWrapper::getLogger();
    $logger->info("Processing request get count of number of pages by page type");

    return new JSONSuccess(APIPublicService::countPagesByPageType());
  }
}