export class Paginator {

	private CurrentPage: number;
	private MaxPages: number;
	private PaginatorElement: HTMLElement;
	private PageForm: HTMLFormElement;
	private PageInput: HTMLInputElement;
	private MaxPagesElement: HTMLElement;
	private NextButton: HTMLElement;
	private PrevButton: HTMLElement;

	private OnPageManuallyEnteredCallbacks = [];
	private OnPageNextCallbacks = [];
	private OnPagePrevCallbacks = [];

	/**
	 * Creates Paginator Element which displays page form, input, and previous/next buttons. Event listeners are created for next/previous buttons as well as form submission.
	 * @param PaginatorElement 
	 */
	public constructor(PaginatorElement: HTMLElement) {
		this.CurrentPage = 1;
		this.MaxPages = 1;
		this.PaginatorElement = PaginatorElement;
		this.PageForm = this.PaginatorElement.querySelector("form");
		this.PageInput = this.PaginatorElement.querySelector(`form input[type="number"]`);
		this.MaxPagesElement = this.PaginatorElement.querySelector(".paginator-max-pages");
		this.NextButton = this.PaginatorElement.querySelector(".next-button");
		this.PrevButton = this.PaginatorElement.querySelector(".prev-button");

		this.PageForm.addEventListener("submit", e => {
			e.preventDefault();
			this.OnPageFormSubmit();
		});

		this.NextButton.addEventListener("click", () => {
			this.OnNextButtonClicked();
		});

		this.PrevButton.addEventListener("click", () => {
			this.OnPrevButtonClicked();
		});
	}

	/**
	 * Updates current page and sets value based on new page number.
	 * @param CurrentPage 
	 */
	public SetCurrentPage(CurrentPage: number) {
		this.CurrentPage = CurrentPage;
		this.PageInput.value = String(CurrentPage);
	}

	/**
	 * Updates the Max Page number and displays that total number.
	 * @param MaxPages 
	 */
	public SetMaxPages(MaxPages: number) {
		this.MaxPages = MaxPages;
		this.MaxPagesElement.textContent = String(MaxPages);
	}

	/**
	 * Callback for manually entered page number.
	 * @param callback 
	 */
	public OnPageManuallyEntered(callback: ((pageNumber: number) => void)) {
		this.OnPageManuallyEnteredCallbacks.push(callback);
	}

	/**
	 * Callback for when the next button is clicked.
	 * @param callback 
	 */
	public OnPageNext(callback: ((pageNumber: number) => void)) {
		this.OnPageNextCallbacks.push(callback);
	}

	/**
	 * Callback for when the previous button is clicked.
	 * @param callback 
	 */
	public OnPagePrev(callback: ((pageNumber: number) => void)) {
		this.OnPagePrevCallbacks.push(callback);
	}

	/**
	 * Parses the page input as an integer and, if it's a valid integer, fires all the connected event listeners.
	 */
	private OnPageFormSubmit() {
		const newPage = parseInt(this.PageInput.value);
		if (!isNaN(newPage)) {
			if (newPage > 0 && newPage <= this.MaxPages) {
				this.SetCurrentPage(newPage);
				for (const callback of this.OnPageManuallyEnteredCallbacks) {
					callback(newPage);
				}
			}
		}
	}

	/**
	 * Method when next button is clicked. Adds +1 to current page. 
	 */
	private OnNextButtonClicked() {
		const possibleNextPageNumber = this.CurrentPage + 1;
		if (possibleNextPageNumber <= this.MaxPages) {
			this.SetCurrentPage(possibleNextPageNumber);
			for (const callback of this.OnPageNextCallbacks) {
				callback(possibleNextPageNumber);
			}
		}
	}

	/**
	 * Method when previous button is clicked. Subtracts -1 from current page.
	 */
	private OnPrevButtonClicked() {
		const possiblePrevPageNumber = this.CurrentPage - 1;
		if (possiblePrevPageNumber > 0) {
			this.SetCurrentPage(possiblePrevPageNumber);
			for (const callback of this.OnPagePrevCallbacks) {
				callback(possiblePrevPageNumber);
			}
		}
	}
}