import { PageResult } from "../Components/PageResult";
import { FileResult } from "../Components/FileResult";
import { Scheduler } from "../../Utils/Scheduler";
import {IFindAndReplaceFoundDto} from "../../Interfaces/FindAndReplace/IFindAndReplaceFoundDto";
import {IFoundFileDto} from "../../Interfaces/FindAndReplace/IFoundFile";
import {IFoundPagesMatchDto} from "../../Interfaces/FindAndReplace/IFoundPagesMatch";
import { IFindResult } from "../../Interfaces/FindAndReplace/IFindResult";
import {FindAndReplace} from "../FindAndReplace";

type FindAndReplaceResponse = {
	status: number,
	error?: string | null,
	foundPagesMatches: IFoundPagesMatchDto | null,
	foundFiles: IFoundFileDto[]
}

export class SearchForm{
    private NoResultsContainer: HTMLElement | null = document.querySelector("#no-results-message-container");
    private QueryInput: HTMLElement | null = document.querySelector("#query-input");
    private ReplaceOccurrencesFormContainer: HTMLElement | null = document.querySelector("#replace-form-container");
    private LoaderContainer: HTMLElement | null = document.querySelector("#results-loader");
    private Form: HTMLFormElement | null = document.querySelector("#search-form");
	private ReplaceAllOccurrencesButton: HTMLButtonElement | null = document.querySelector("#replace-all-occurrences-button");
    private IsProcessing: boolean = false;

	public FindAndReplaceEntryInstance: FindAndReplace;
    public SearchInputChangedSinceLastSubmission: boolean = false;

    public constructor(findAndReplaceInstance: FindAndReplace){
		this.FindAndReplaceEntryInstance = findAndReplaceInstance;
        this.Form.addEventListener("submit", e => {
			e.preventDefault();
			this.OnSubmit();
		});

		this.QueryInput.addEventListener("input", () => {
			this.SearchInputChangedSinceLastSubmission = true;
		});

		if (this.ReplaceAllOccurrencesButton) {
            this.ReplaceAllOccurrencesButton.addEventListener("click", () => {
                this.OnReplaceAllOccurrencesClicked();
            });
        }
    }

	// Clears components after replacement.
	private OnReplaceAllOccurrencesClicked() {
        FileResult.ClearAll();
        PageResult.ClearAll();
    }

    public async OnSubmit(): Promise<void> {
        if (this.IsProcessing){
            return;
        }

        this.IsProcessing = true;
        this.IsProcessing = true;
		this.NoResultsContainer.style.display = "none";
		PageResult.ClearAll();
		FileResult.ClearAll();
		this.LoaderContainer.style.display = null;
		this.ReplaceOccurrencesFormContainer.style.display = "none";
		this.SearchInputChangedSinceLastSubmission = false;

        const fData = new FormData(this.Form as HTMLFormElement);
		const queryString = fData.get("query").toString();
		const urlParams = new URLSearchParams();
		urlParams.set("query", queryString);
		this.FindAndReplaceEntryInstance.LastQueryUsed = queryString;

        const response = await fetch(`/uplift/find-and-replace/find?${urlParams.toString()}`, {
			method:"GET",
			cache:"no-cache",
			credentials:"same-origin"
		});


        let data: FindAndReplaceResponse;
		try{
			data = await response.json();
		}catch(jsonSyntaxError){
			alert("The server responded with invalid JSON.");
			this.IsProcessing = false;
			return;
		}

		await Scheduler.Wait(330);

		if (data.status === 1){

			const pageResults: PageResult[] = [];
			const fileResults: FileResult[] = [];

			if (data.foundPagesMatches.foundPages?.length > 0){
				this.ReplaceOccurrencesFormContainer.style.display = null;
				for (const foundPageDto of data.foundPagesMatches.foundPages) {
					const pageResult = new PageResult(foundPageDto, this);
					pageResult.RenderInto(document.querySelector("#search-results-container"));
					pageResults.push(pageResult);
				}
			}

			if (data.foundFiles.length > 0){
				this.ReplaceOccurrencesFormContainer.style.display = null;
				for (const foundFileDto of data.foundFiles) {
                    const fileResult = new FileResult(foundFileDto, this);
                    fileResult.RenderInto(document.querySelector("#search-results-container"));
                    fileResults.push(fileResult);
				}
			}

			// Only show no results if there are no found pages and there are no found files
			if (data.foundPagesMatches.foundPages?.length === 0 && data.foundFiles.length === 0){
				this.NoResultsContainer.style.display = null;
			}

			this.FindAndReplaceEntryInstance.CurrentFileResults = fileResults;
			this.FindAndReplaceEntryInstance.CurrentPageResults = pageResults;
		}else if (data.status === -1){

		}

		this.LoaderContainer.style.display = "none";
		this.IsProcessing = false;
    }
}